function aom_model
format short
global x

%   This is a 1D transport-reaction model based on the FORTRAN model C.CANDI (Boudreau, 1996; Luff et al., 2001)
%   Methane (CH4) fluxes are determined via modelling pore water profiles of sulphate (SO4)
%   The depth distribution of methane and sulphate is described by a partial differential equation that takes into account:
%       - molecular diffusion
%       - fluid advection
%       - bioirrigation
%       - sulphate reduction (via organic matter degradation + anaerobic oxidation of methane, AOM)
%       - methanogenesis
% 
%   input:
%       - length of pore water profile [cm]
%       - pressure [bar], temperature [degrees celsius] and salinity
%       - porosity parameters, fi0, fif, b
%       - pore water concentrations at the top and the bottom of the sediment core for the respective elements[mM]
%       - upward fluid flow u [cm/yr]
%       - irrigation depth xmix [cm]
%       - irrigation exchange coefficient icoef [1/yr]
%       - rate constants for AOM, kaom [1/(mM yr)] and methanogenesis, kg [1/yr]
%       - half-saturation constants [mM], Kso4, Kiso4, Ksaom   
%
%   output:
%       - element fluxes across the bottom of the sediment domain and across the sediment seawater interface for methane [mmol/m2/yr]
%       - rates of AOM, methanogenesis and sulphate reduction via OM degradation  [mmol/m2/yr]
        
%_______________________________________________________________________________________
% written by: (1) Heleen Vanneste - part of PhD thesis (November 2010)
%			  National Oceanography Centre, Southampton
%			  University of Southampton, United Kingdom
%
%			  (2) Matthias Haeckel
%			  IFM-GEOMAR, Kiel, Germany
%
%        and  (3) Boris Kelly-Gerreyn
%             National Oceanography Centre, Southampton
%_______________________________________________________________________________________

% Read data
carl = [inputfile];                              % input file [.txt format]
for m=1:length(carl)
    filename=num2str(carl(m));
    infile = strcat(filename,'.txt');
    fid = fopen(infile);
    tline = fgets(fid);
    tline = fgets(fid);
    [d c] = fscanf(fid,'%f');
    ibad = find(d==-9999);                        % put -9999 in empty spaces in .txt file
    d(ibad)=NaN*ones(size(ibad));
    fclose(fid);
    par    =18;                                   % total number of columns in input file
    depth  = d(1 : par : c);                      % column number in input file
    so4    = d(7 : par : c)
    ch4    = d(16: par : c);
    clear d; 
       
 % plotting
 sym = ['ko';'k*';'k+'];    
   subplot(1,3,1);
   plot(ch4,depth,sym(m,:))
   %set(gca,'XLim',[500 5000]);
   hold on;
   
   subplot(1,3,2);
   plot(so4,depth,sym(m,:))
   %set(gca,'XLim',[500 5000]);
   hold on;
end
for k=1:3
    subplot(1,3,k);
    ylabel('depth / cmbsf');
    set(gca,'YDir','reverse');
    set(gca,'YLim',[0 250]);
end

%% time
tspan=[0 1E2 1E3];                              % time [yr]- necessary to reach steady state

%% depth [cm]
% centre station
xmax=500.5; dx=1; x=[0:dx:xmax]; 

% off-centre station
%xmax=110; dx=1; x=[0:dx:xmax]; 

% margin station
%xmax=217; dx=1; x=[0:dx:xmax]; 

% mudflow station
%xmax=179.5; dx=1; x=[0:dx:xmax];

%% parameters
P=220; T=4.3; S=34.2;        % pressure [bar], temperature [C], salinity

% upward fluid flow [cm/a]
vent = 4;                % centre station
%vent = 3.3;                 % off-centre station
%vent =0.4;               % margin station
%vent=0;                  % slope station 
%vent =0;                 % mudflow station


% porosity [volPW/volbulkSed]
fi0 = 0.7530; fif=0.5799; b=0.0605;       % centre station
%fi0 = 0.7593; fif=0.5718; b=0.0778;       % off-centre station
%fi0 = 0.7484; fif=0.5301; b=0.0445;       % margin station
%fi0 = 0.7766; fif=0.5384; b=0.0644;       % slope station
%fi0 = 0.6870; fif=0.5111; b=0.0357;       % mudflow station

% irrigation depth [cm]
irmix = 32;                % centre station
%irmix = 32;               % off-centre station
%irmix = 40;               % margin station
% irmix = 39;              % slope station
%irmix = 50;               % mudflow station

% pore water-mixing coefficient [1/a]
icoef = 3.1;                 % centre station
%icoef = 2.5;                % off-centre station
% icoef = 0.6;               % margin station
%icoef = 0.5;                % slope station 
%icoef = 0.6;                % mudflow station


% Rate constants
kg = 1E-6;                      % rate constant for OM oxidation [1/a]
kaom =7;                        % rate constant for AOM [1/(mM a)]


% half-saturation constants [mM]
Kso4 = 1.0;
Kiso4 = 1.0;
Ksaom = 1.0;

diffc = diffcoeff(P,T,S);
poros =[fi0 fif b];
irrigat = [icoef irmix];
sed = [vent];
monodc = [Kso4 Kiso4 Ksaom];
krate = [kg kaom];

%% Initial conditions
% initial and final concentrations
% centre station
so40 = 28.2;     so4L = 0.1;            % sulphate  [mM]
ch40 = 0.002;    ch4L = 180;            % methane   [mM]

% off-centre station
%  so40 = 28.5;     so4L = 0.18;        % sulphate [mM]
%  ch40 = 0.002;     ch4L = 100;        % methane [mM]

% margin station
% so40 = 28.2;     so4L = 0.23;         % sulphate [mM]
% ch40 = 0.002;    ch4L = 112.5;        % methane [mM]

% slope station
%so40 = 28.3;     so4L = 0.08;          % sulphate [mM]
%ch40 = 0.001;     ch4L = 72;           % methane [mM]

% mudflow station
% so40 = 28.19;     so4L = 0.08;        % sulphate [mM]
% ch40 = 0.002;     ch4L = 30;          % methane [mM]

% initialize model domain                   
so4i = so4L*ones(size(x));
ch4i = ch4L*ones(size(x));

% initial conditions 
so4i(1) = so40;     so4i(end) = so4L;
ch4i(1) = ch40;     ch4i(end) = ch4L;

conci =[so4i ch4i];

%% solutions
[t conc]=ode15s(@diagenesis,tspan,conci,[],x,diffc,poros,sed,irrigat,monodc,krate,dx);
global Mso4 Mch4 fi to raom sr ch4prod u irrch4   
%save amo_model_results_52_62 conc t x poros krate sed raom 

% PLotting results
sym = ['k-';'b-';'r-'];
so4f = conc(:,0*length(x)+1:1*length(x));
ch4f = conc(:,1*length(x)+1:2*length(x));
irrch4f = irrch4(1:end);
%irrf = alpha;


for i=1:length(t)      
    subplot(1,3,1)
    plot(ch4f(end,:),x,sym(i,:),'LineWidth',2);
    xlabel('CH_4 / mM');
    hold on; 
    
    subplot(1,3,2)
    plot(so4f(end,:),x,sym(i,:),'LineWidth',2);
    xlabel('SO_4^2^- / mM');
    hold on; 
    
%     subplot(1,3,3)
%     plot(raom,x(2:end-1),sym(i,:),'LineWidth',2);
%     xlabel('R_a_o_m /mM yr^-^1')
%     ylabel('depth / cmbsf');
%     set(gca,'YDir','reverse');
%     set(gca,'YLim',[0 250]);
%     hold on;
end

% Title
subplot(1,3,1)
title('Diffusion, Advection AND reaction','FontSize',20,'FontWeight','demi');

orient landscape
%print -depsc lstf52_14.eps

%% Output- fluxes
D = diffcoeff(P,T,S);                                     % diffusion coefficient                           

% source/sink term (irrigation)
irrch4 = sum(fi(2:end-1).*irrch4f.*dx*10)                                   %mmol/m2/year			 

% fluxes across the sediment water interface
diffch4=((D(6)/to(1))*fi(1)*(ch4f(end,1) - ch4f(end,2))/dx)*10              % mmol/m2/year
advch4= fi(1)*u(1)*ch4f(end,1)*10                                           % mmol/m2/year
fluxswich4= diffch4 + advch4 - irrch4                                       % mmol/m2/year
transtop= [diffch4 advch4 fluxswich4];

% fluxes across the bottom of sediment domain
diffch4b=((D(6)/to(end))*fi(end)*(ch4f(end,end-1) - ch4f(end,end))/dx)*10   % mmol/m2/year                                                                                  
advch4b= fi(end)*u(end)*ch4f(end,end)*10                                    % mmol/m2/year
fluxbtch4=  diffch4b + advch4b                                              % mmol/m2/year
transbot=[diffch4b advch4b fluxbtch4]; 

% methane production by organic matter degradation 
% Depth integrated methane production [mmol/m2/yr]
ch4p=sum(fi(2:end-1).*ch4prod.*dx*10) 

% Output - reaction rates
% Depth integrated AOM rate [mmol/m2/yr]
raomo=sum(fi(2:end-1).*raom.*dx*10)
% Depth integrated SR rate [mmol/m2/yr]
sro=sum(fi(2:end-1).*sr.*dx*10) 

fin = (abs(fluxbtch4) + ch4p) - (abs(fluxswich4)+ abs(raomo))                       % check the mass balance
relerror= (abs(fluxswich4)-(abs(fluxbtch4)- abs(raomo)+ch4p))/abs(fluxbtch4)*100    % error on mass balance

%save amomassbal_52_62 irrch4 transtop transbot ch4p raomo sro fin relerror    

%%
function dconc = diagenesis(t,conc,x,D,poros,sed,irrigat,monodc,krate,dx);
global so4 ch4 Mso4 Mch4 fi to raom sr ch4prod u irrch4

so4 = conc(1:length(x));
ch4 = conc(length(x)+1:2*length(x));

% seawater concentrations - upper boundary
so40 = 28.2;          % so4 [mM]
ch40 = 0;         % ch4 [mM]

% initialisation for calculation of irrigation term
cso4 = so40 *ones(size(x))';
cch4 = ch40 *ones(size(x))';

% upper boundary condition and initialization
dso4 = zeros(size(x))';
dch4 = zeros(size(x))';

% porosity profile
fi = ((poros(1)-poros(2))*exp(-poros(3)*x)+poros(2))';
fis = 1 -fi;

% porewater advection and externally impressed fluid flow
u = (-sed(1)*fi(1))./ fi;

% tortuosity correction for diffusion coefficients
to = 1-2*log(fi);          % Boudreau (1997)          

% irrigation term
irr= irrigat(1)*(ones(size(x)))';                           % initialise vector size
xir = find(x>=irrigat(2));                                  % all x-values > irrigation zone
irr(xir)=(irrigat(1)*exp(-(x(xir)-irrigat(2))/.1));


% Calling reaction function
reac(monodc,krate);
Corg =625;                                                  % organic matter [mmol/l(dry sediment)]

%% SO42-
[diff adv] = fdiffer(so4,x,D(5),u,fi,dx);
% sulphate reduction by the degradation of OM [mM/a]
sr = krate(1)*Corg*Mso4*0.5.*fis(2:end-1)./fi(2:end-1);

% Total reaction term [mM/a]
reacso4 = -sr - raom;

% irrigation term
irrso4=(irr(2:end-1).*(so4(2:end-1)-cso4(2:end-1)));  

% Differential equation for sulphate [mM/a]
dso4(2:end-1) = diff - adv + reacso4 - irrso4; 


%% Ch4
[diff adv] = fdiffer(ch4,x,D(6),u,fi,dx);
% Methane production by organic matter degradation
ch4prod=krate(1)*Corg*Mch4*0.5.*fis(2:end-1)./fi(2:end-1);

% irrigation term
irrch4=(irr(2:end-1).*(ch4(2:end-1)-cch4(2:end-1))); 

% Total reaction term [mM/a]
reacch4 = ch4prod - raom;

% Differential equation for methane [mM/a]
dch4(2:end-1) = diff - adv + reacch4 - irrch4; 
					 
%% lower boundary condition: 
% Neumann conditions: no flux
dso4(end)=dso4(end-1);

% Dirichlet conditions: fix concentration
% we use fixed concentration conditions for mehane because otherwise you don't have a
% flux from below

dch4(end)= 0;

dconc = cat(1,dso4,dch4);

%-----------------------------------------------------------------------------------------------------------
%% diffusion and advection term         
 
function [diff adv] = fdiffer(conc,x,D,u,poros,dx)
global fi to

% solutes 
Ds = (1./(to.*fi))*D;

difft1= fi(2:end-1).*((conc(3:end)-2.*conc(2:end-1)+conc(1:end-2))/(dx.^2));

difft2=(fi(2:end-1)-fi(1:end-2))/dx;

difft3=(fi(2:end-1)./to(2:end-1)).*((to(2:end-1)-to(1:end-2))/dx);


diff= Ds(2:end-1).*(difft1+(difft2-difft3).*((conc(2:end-1)-conc(1:end-2))/dx));

adv = u(2:end-1).*(conc(3:end)- conc(2:end-1))/dx;

%-----------------------------------------------------------------------------------------------------------       
%% Calculation of diffusion coefficients
% after Li & Gregory (1974), Hayduk & Laudie (1974),
% and Boudreau (1997)
% D [cm2/a], p [bar], t [degC], S [g/kg]

function D = diffcoeff(P,T,S)

% dynamic viscosity of pure water at T, P=1 bar
muo = viscosity(1,T,0);
% dynamic viscosity of seawater at P, T, S 
mu = viscosity(P,T,S);

% O2:       Hayduk & Laudie (1974)
D(1) = 4.72E-9*(T+273.15)/(muo/100*27.9^.6);
% NO3-:     Boudreau (1997)
D(2) = (9.50+.388*T)*10^-6;
% Mn2+:     Boudreau (1997)
D(3) = (3.18+.155*T)*10^-6;
% Fe2+:     Boudreau (1997)
D(4) = (3.31+.150*T)*10^-6;
% SO42-:    Boudreau (1997)
D(5) = (4.88+.232*T)*10^-6;
% CH4:      Hayduk & Laudie (1974)
D(6) = 4.72E-9*(T+273.15)/(muo/100*(37.7^.6));
% HS-:      Boudreau (1997)
D(7) = (10.4+.273*T)*10^-6;
% NH4+:     Boudreau (1997)
D(8) = (9.50+.413*T)*10^-6;
% HCO3-:    Boudreau (1997)
D(9) = (5.06+.275*T)*10^-6;
% Ca2+:     Boudreau (1997)
D(10) = (3.60+.179*T)*10^-6;
% Cl-:      Boudreau (1997)
D(11) = (9.60+.438*T)*10^-6;
% Li+       Boudreau (1997)
D(12) = (4.43+.241*T)*10^-6;


D = D * muo/mu * 365*24*3600;   % P and S correction using Stokes-Einstein relation

% Boric acid: D = 1.12 +- 0.02*10^-5 cm^2*s^-1 at T=25 and I = 0.5-0.6 M     Mackin (1986) 
% dynamic viscosity of seawater at P, T=25, S=29.2
muT1 = viscosity(1,25,29.2);
% dynamic viscosity of seawater at P, T, S=29.2
muT2 = viscosity(1,T,29.2);
Dboric = 1.12E-5;
% temperature correction using Stokes-Einstein relation
Dboric = ((Dboric * muT1 * (T+273.15)) / (298.15 * muT2)); 
% P and S correction using Stokes-Einstein relation
Dboric = Dboric * muT2/mu * 365*24*3600;                                                            
Dborate = 0.875*Dboric; % Boudreau and Canfield (1988)
D(13) = 0.5*(Dboric+Dborate);      

%-----------------------------------------------------------------------------------------------------------     
%% Reactions

function reac(monodc,krate)
global so4 ch4 Mso4 Mch4 raom 

    % sulfate reduction
    Mso4 = so4(2:end-1)./(so4(2:end-1)+monodc(1));    
 
    % methanogenesis
    Mch4 = monodc(2)./(so4(2:end-1)+monodc(2));

    % AOM reaction
    raom = krate(2)*ch4(2:end-1).*(so4(2:end-1)./(so4(2:end-1)+monodc(3))); %[mM/a]
    
%----------------------------------------------------------------------------------------------------------- 
%% Calculation of the dynamic viscosity of water 
% after Kukulka (1987) (see Boudreau, 1997)
%
% mu [cPoise=0.01 g/cm/s] = f(p,T,S) 
% for ranges of p=1-1000 bar, t=0-30 degC, and S=0-36

function mu = viscosity(P,T,S)

mu = 1.7910 - 6.144E-2*T +1.4510E-3*T^2 - 1.6826E-5*T^3 ...
    - 1.5290E-4*P + 8.3885E-8*P^2 + 2.4727E-3*S ...
    + T*(6.0574E-6*P - 2.6760E-9*P^2) ...
    + S*(4.8429E-5*T - 4.7172E-6*T^2 + 7.5986E-8*T^3);   

    