function conc=fluidflow_model(param)
format short
global x 

%   This is a 1D transport model based on the FORTRAN model C.CANDI (Boudreau, 1996; Luff et al., 2001)
%   Flow velocities are determined by fitting the model results to the measured pore water profiles of conservative elements
%   i.e. Chloride(Cl), boron(B) and sodium(Na)
%   The depth distribution of these conservative elements is described by a partial differential equation that takes into account:
%       - molecular diffusion
%       - fluid advection
%       - bioirrigation 
% 
%   input:
%       - length of pore water profile [cm]
%       - pressure [bar], temperature [degrees celsius] and salinity
%       - porosity parameters, fi0, fif, b
%       - pore water concentrations at the top and the bottom of the sediment core for the respective elements[mM]
%
%Note: to be used with optimalresults.m
%_______________________________________________________________________________________
% written by: (1) Heleen Vanneste - part of PhD thesis (November 2010)
%			  National Oceanography Centre, Southampton
%			  University of Southampton, United Kingdom
%
%			  (2) Matthias Haeckel
%			  IFM-GEOMAR, Kiel, Germany
%
%        and  (3) Boris Kelly-Gerreyn
%             National Oceanography Centre, Southampton
%_______________________________________________________________________________________

%% time [yr]
tspan=[0 1E2 1E3];               % time [yr]- necessary to reach steady state

%% depth [cm]
% centre station
%xmax=500.5; dx=1; x=[0:dx:xmax]; 

% off-centre station
xmax=110; dx=1; x=[0:dx:xmax]; 

% margin station
%xmax=217; dx=1; x=[0:dx:xmax]; 

% slope
%xmax=120; dx=1; x=[0:dx:xmax];

% mudflow station
%xmax=179.5; dx=1; x=[0:dx:xmax];

%% parameters
P=220; T=4.3; S=34.2;           % pressure [bar], temperature [C], salinity
% upward fluid flow [cm/a]
%vent    = 4;                    % centre station
%vent   = 2.7;                  % off-centre station
%vent   = 0.4;                  % margin station
%vent   = 0;                    % slope station 
%vent   = 0;                    % mudflow station

% porosity [volPW/volbulkSed]- from leastsquare fit of porosity profiles
%fi0 = 0.7530; fif=0.5799; b=0.0605;         % centre station
fi0 = 0.7593; fif=0.5718; b=0.0778;        % off-centre station
%fi0 = 0.7484; fif=0.5301; b=0.0445;        % margin station
%fi0 = 0.7766; fif=0.5384; b=0.0644;        % slope station
%fi0 = 0.6870; fif=0.5111; b=0.0357;        % mudflow station

% irrigation depth [cm]
%irmix = 32;                     % centre station
irmix =28.8;                   % off-centre station
%irmix = 40;                    % margin station
% irmix = 39;                   % slope station
%irmix = 50;                    % mudflow station

% pore water-mixing coefficient [1/yr]
%icoef = 3.1;                    % centre station
icoef = 2.5;                   % off-centre station
%icoef = 0.6;                   % margin station
%icoef = 0.5;                   % slope station 
%icoef = 0.6;                   % mudflow station
 

vent= param(1);             % the parameter that you want to calculate using lsqfit

diffc = diffcoeff(P,T,S);
sed = [vent];
poros =[fi0 fif b];
irrigat = [icoef irmix];


%% Initial conditions
% upper and lower boundary
% centre station
% Cl0 = 551.0;            ClL = 198.7;          % Cl [mM]
% B0 =0.5 ;               BL= 10.0;             % B [mM] 
% Na0 = 473.8;            NaL=211.2;            % Na [mM]
 
% off-centre station
Cl0 = 555.0;          ClL = 252.1;          % Cl [mM]
B0 =0.5 ;             BL= 10.2;             % B [mM] 
Na0 = 464.4;          NaL=228.0;            % Na [mM]

% margin station
% Cl0 = 552.3;          ClL = 296.1;          % Cl [mM]
% B0 =0.5 ;             BL= 8.5;              % B [mM] 
% Na0 =462.9 ;          NaL= 294.0;           % Na [mM]

% slope station
% Cl0 = 557.0;          ClL = 460.7;          % Cl [mM]
% B0 =0.6 ;             BL= 2.3;              % B [mM] 
% Na0 =463.3 ;          NaL= 459.4;           % Na [mM]
% 
% mudflow station
% Cl0 = 551.9;          ClL = 460.7;          % Cl [mM]
% B0 =0.5 ;             BL= 2.9;              % B [mM] 
% Na0 =467.8 ;          NaL= 432.7;           % Na [mM]

% initialize model domain                   
Cli = ClL*ones(size(x));
Bi= BL*ones(size(x));
Nai= NaL*ones(size(x));

% initial conditions                       
Cli(1) = Cl0;     Cli(end) = ClL;
Bi(1)=B0;         Bi(end)=BL;
Nai(1)=Na0;       Nai(end)=NaL;

conci =[Cli Bi Nai];

%% solutions
[t conc]=ode15s(@diagenesis,tspan,conci,[],x,diffc,poros,sed,irrigat,dx);
global fi u to irrB irrCl irrNa

%save flux_sens_52_5 conc t x poros sed irrigat irrB irrCl irrLi

%%
function dconc = diagenesis(t,conc,x,D,poros,sed,irrigat,dx);
global fi u to irrB irrCl irrNa

Cl  = conc(1:length(x));
B   = conc(1*length(x)+1:2*length(x));
Na  = conc(2*length(x)+1:3*length(x));

% seawater concentrations - upper boundary
Cl0 = 555.0;                % Cl [mM]
B0  = 0.5;                  % B [mM] 
Na0 = 464.4;                % Na [mM]

% initialisation for calculation of irrigation term
cCl = Cl0 *ones(size(x))';
cB  = B0 *ones(size(x))';
cNa = Na0 *ones(size(x))';

% upper boundary condition and initialization
dCl = zeros(size(x))';
dB  = zeros(size(x))';
dNa = zeros(size(x))';

% porosity profile
fi = ((poros(1)-poros(2))*exp(-poros(3)*x)+poros(2))';

% porewater advection and externally impressed fluid flow
u = (-sed(1)*fi(1))./ fi;

% tortuosity correction for diffusion coefficients
to = 1-2*log(fi);                                   % Boudreau (1997)   

% irrigation term
irr= irrigat(1)*(ones(size(x)))';                   % initialise vector size
xir = find(x>=irrigat(2));                          % all x-values > irrigation zone
irr(xir)=(irrigat(1)*exp(-(x(xir)-irrigat(2))/.1));

%% Solving differential equation for each constituent
%% Cl
[diff adv] = fdiffer(Cl,x,D(11),u,fi,dx);           % calling diffusion and advection 
irrCl=(irr(2:end-1).*(Cl(2:end-1)-cCl(2:end-1)));   % irrigation term
dCl(2:end-1) = diff - adv - irrCl; 

%% B
[diff adv] = fdiffer(B,x,D(13),u,fi,dx);            % calling diffusion and advection 
irrB=(irr(2:end-1).*(B(2:end-1)-cB(2:end-1)));      % irrigation term
dB(2:end-1) = diff - adv - irrB; 

%% Na
[diff adv] = fdiffer(Na,x,D(14),u,fi,dx);           % calling diffusion and advection
irrNa=(irr(2:end-1).*(Na(2:end-1)-cNa(2:end-1)));   % irrigation term
dNa(2:end-1) = diff - adv - irrNa; 

%% lower boundary condition: 
% Dirichlet conditions: fix concentration

dCl(end)= 0;
dB(end)=0;
dNa(end)=0;

dconc = cat(1,dCl,dB,dNa);

%-----------------------------------------------------------------------------------------------------------
%% diffusion and advection term       
function [diff adv] = fdiffer(conc,x,D,u,poros,dx)
global fi to

% solutes 
Ds = (1./(to.*fi))*D;

difft1= fi(2:end-1).*((conc(3:end)-2.*conc(2:end-1)+conc(1:end-2))/(dx.^2));

difft2=(fi(2:end-1)-fi(1:end-2))/dx;

difft3=(fi(2:end-1)./to(2:end-1)).*((to(2:end-1)-to(1:end-2))/dx);


diff= Ds(2:end-1).*(difft1+(difft2-difft3).*((conc(2:end-1)-conc(1:end-2))/dx));

adv = u(2:end-1).*(conc(3:end)- conc(2:end-1))/dx;

%-----------------------------------------------------------------------------------------------------------
%% Calculation of diffusion coefficients
% after Li & Gregory (1974), Hayduk & Laudie (1974),
% and Boudreau (1997)
% D [cm2/a], p [bar], t [degC], S [g/kg]

function D = diffcoeff(P,T,S)

% dynamic viscosity of pure water at T, P=1 bar
muo = viscosity(1,T,0);
% dynamic viscosity of seawater at P, T, S 
mu = viscosity(P,T,S);

% O2:       Hayduk & Laudie (1974)
D(1) = 4.72E-9*(T+273.15)/(muo/100*27.9^.6);
% NO3-:     Boudreau (1997)
D(2) = (9.50+.388*T)*10^-6;
% Mn2+:     Boudreau (1997)
D(3) = (3.18+.155*T)*10^-6;
% Fe2+:     Boudreau (1997)
D(4) = (3.31+.150*T)*10^-6;
% SO42-:    Boudreau (1997)
D(5) = (4.88+.232*T)*10^-6;
% CH4:      Hayduk & Laudie (1974)
D(6) = 4.72E-9*(T+273.15)/(muo/100*(37.7^.6));
% HS-:      Boudreau (1997)
D(7) = (10.4+.273*T)*10^-6;
% NH4+:     Boudreau (1997)
D(8) = (9.50+.413*T)*10^-6;
% HCO3-:    Boudreau (1997)
D(9) = (5.06+.275*T)*10^-6;
% Ca2+:     Boudreau (1997)
D(10) = (3.60+.179*T)*10^-6;
% Cl-:      Boudreau (1997)
D(11) = (9.60+.438*T)*10^-6;
% Li+       Boudreau (1997)
D(12) = (4.43+.241*T)*10^-6;
% Na+       Boudreau (1997)
D(14) = (6.06+.297*T)*10^-6;

D = D * muo/mu * 365*24*3600;   % P and S correction using Stokes-Einstein relation

% Boric acid: D = 1.12 +- 0.02*10^-5 cm^2*s^-1 at T=25 and I = 0.5-0.6 M     Mackin (1986) 
% dynamic viscosity of seawater at P, T=25, S=29.2
muT1 = viscosity(1,25,29.2);
% dynamic viscosity of seawater at P, T, S=29.2
muT2 = viscosity(1,T,29.2);
Dboric = 1.12E-5;
% temperature correction using Stokes-Einstein relation
Dboric = ((Dboric * muT1 * (T+273.15)) / (298.15 * muT2)); 
% P and S correction using Stokes-Einstein relation
Dboric = Dboric * muT2/mu * 365*24*3600;                                                            
Dborate = 0.875*Dboric; % Boudreau and Canfield (1988)
D(13) = 0.5*(Dboric+Dborate);  

%-----------------------------------------------------------------------------------------------------------
%% Calculation of the dynamic viscosity of water 
% after Kukulka (1987) (see Boudreau, 1997)
%
% mu [cPoise=0.01 g/cm/s] = f(p,T,S) 
% for ranges of p=1-1000 bar, t=0-30 degC, and S=0-36

function mu = viscosity(P,T,S)

mu = 1.7910 - 6.144E-2*T +1.4510E-3*T^2 - 1.6826E-5*T^3 ...
    - 1.5290E-4*P + 8.3885E-8*P^2 + 2.4727E-3*S ...
    + T*(6.0574E-6*P - 2.6760E-9*P^2) ...
    + S*(4.8429E-5*T - 4.7172E-6*T^2 + 7.5986E-8*T^3);    


            
   
