function conc=fluidflow_model
format short
clear all;clf;
global x 

%   This is a 1D transport model based on the FORTRAN model C.CANDI (Boudreau, 1996; Luff et al., 2001)
%   Flow velocities are determined by fitting the model results to the measured pore water profiles of conservative elements
%   i.e. Chloride(Cl), boron(B) and sodium(Na)
%   The depth distribution of these conservative elements is described by a partial differential equation that takes into account:
%       - molecular diffusion
%       - fluid advection
%       - bioirrigation 
% 
%   input:
%       - length of pore water profile [cm]
%       - pressure [bar], temperature [degrees celsius] and salinity
%       - porosity parameters, fi0, fif, b
%       - pore water concentrations at the top and the bottom of the sediment core for the respective elements[mM]
%
%   output:
%       - upward fluid flow u [cm/yr]
%       - irrigation depth xmix [cm]
%       - irrigation exchange coefficient icoef [1/yr]
%       - element fluxes across the bottom of the sediment domain and across the sediment seawater interface for B and Li [mmol/m2/year]
        
%_______________________________________________________________________________________
% written by: (1) Heleen Vanneste - part of PhD thesis (November 2010)
%			  National Oceanography Centre, Southampton
%			  University of Southampton, United Kingdom
%
%			  (2) Matthias Haeckel
%			  IFM-GEOMAR, Kiel, Germany
%
%        and  (3) Boris Kelly-Gerreyn
%             National Oceanography Centre, Southampton
%_______________________________________________________________________________________


%% Read data
carl = [49502];                         % input file [.txt format]
for m=1:length(carl)
    filename=num2str(carl(m));
    infile = strcat(filename,'.txt');
    fid = fopen(infile);
    tline = fgets(fid);
    tline = fgets(fid);
    [d c] = fscanf(fid,'%f');
    ibad = find(d==-9999);                  % put -9999 in empty spaces in .txt file
    d(ibad)=NaN*ones(size(ibad));
    fclose(fid);
    par    = 18;                            % total number of columns in input file
    depth = d(1 : par : c);                 % column number in input file
    Cl    = d(5 : par : c);
    Na    = d(10 : par : c);
    B     = d(9 : par : c);
    Li    = d(8 : par :c);
    clear d; 
       
 % plotting data from input file
 sym = ['ko';'k*';'k+'];    
   subplot(1,4,1);
   plot(Cl,depth,sym(m,:))
   %set(gca,'XLim',[500 5000]);
   hold on;
   
   subplot(1,4,2);
   plot(B,depth,sym(m,:))
   %set(gca,'XLim',[500 5000]);
   hold on;
   
   subplot(1,4,3);
   Li=Li/1000;
   plot(Li,depth,sym(m,:))
   %set(gca,'XLim',[500 5000]);
   hold on;
      
   subplot(1,4,4);
   plot(Na,depth,sym(m,:))
   %set(gca,'XLim',[500 5000]);
   hold on;   
end
for k=1:4
    subplot(1,4,k);
    ylabel('depth / cmbsf');
    set(gca,'YDir','reverse');
    set(gca,'YLim',[0 250]);
end

%% time [yr]
tspan=[0 1E2 1E3];               % time [yr]- necessary to reach steady state

%% depth [cm]
% centre station
%xmax=500.5; dx=1; x=[0:dx:xmax]; 

% off-centre station
xmax=110; dx=1; x=[0:dx:xmax]; 

% margin station
%xmax=217; dx=1; x=[0:dx:xmax]; 

% slope
%xmax=120; dx=1; x=[0:dx:xmax];

% mudflow station
%xmax=179.5; dx=1; x=[0:dx:xmax];

%% parameters
P=220; T=4.3; S=34.2;           % pressure [bar], temperature [C], salinity
% upward fluid flow [cm/a]
%vent    = 4;                    % centre station
vent   = 2.7;                  % off-centre station
%vent   = 0.4;                  % margin station
%vent   = 0;                    % slope station 
%vent   = 0;                    % mudflow station

% porosity [volPW/volbulkSed]- from leastsquare fit of porosity profiles
%fi0 = 0.7530; fif=0.5799; b=0.0605;         % centre station
fi0 = 0.7593; fif=0.5718; b=0.0778;        % off-centre station
%fi0 = 0.7484; fif=0.5301; b=0.0445;        % margin station
%fi0 = 0.7766; fif=0.5384; b=0.0644;        % slope station
%fi0 = 0.6870; fif=0.5111; b=0.0357;        % mudflow station

% irrigation depth [cm]
%irmix = 32;                     % centre station
irmix =28.8;                   % off-centre station
%irmix = 40;                    % margin station
% irmix = 39;                   % slope station
%irmix = 50;                    % mudflow station

% pore water-mixing coefficient [1/yr]
%icoef = 3.1;                    % centre station
icoef = 2.5;                   % off-centre station
%icoef = 0.6;                   % margin station
%icoef = 0.5;                   % slope station 
%icoef = 0.6;                   % mudflow station
 

diffc = diffcoeff(P,T,S);
sed = [vent];
poros =[fi0 fif b];
irrigat = [icoef irmix];


%% Initial conditions
% upper and lower boundary
% centre station
% Cl0 = 551.0;            ClL = 198.7;          % Cl [mM]
% B0 =0.5 ;               BL= 10.0;             % B [mM] 
% Li0=0.0264;             LiL=0.1846;           % Li [mM]
% Na0 = 473.8;            NaL=211.2;            % Na [mM]
 
% off-centre station
Cl0 = 555.0;          ClL = 252.1;          % Cl [mM]
B0 =0.5 ;             BL= 10.2;             % B [mM] 
Li0 = 0.0257;         LiL = 0.1869;         % Li [mM] 
Na0 = 464.4;          NaL=228.0;            % Na [mM]

% margin station
% Cl0 = 552.3;          ClL = 296.1;          % Cl [mM]
% B0 =0.5 ;             BL= 8.5;              % B [mM] 
% Li0 = 0.0264;         LiL =0.2041;          % Li [mM]
% Na0 =462.9 ;          NaL= 294.0;           % Na [mM]

% slope station
% Cl0 = 557.0;          ClL = 460.7;          % Cl [mM]
% B0 =0.6 ;             BL= 2.3;              % B [mM] 
% Li0 = 0.0294;         LiL = 0.1067;         % Li [mM]
% Na0 =463.3 ;          NaL= 459.4;           % Na [mM]
% 
% mudflow station
% Cl0 = 551.9;          ClL = 460.7;          % Cl [mM]
% B0 =0.5 ;             BL= 2.9;              % B [mM] 
% Li0 = 0.0238;         LiL = 0.2301;         % Li [mM]
% Na0 =467.8 ;          NaL= 432.7;           % Na [mM]

% initialize model domain                   
Cli = ClL*ones(size(x));
Bi= BL*ones(size(x));
Lii= LiL*ones(size(x));
Nai= NaL*ones(size(x));

% initial conditions                       
Cli(1) = Cl0;     Cli(end) = ClL;
Bi(1)=B0;         Bi(end)=BL;
Lii(1)=Li0;       Lii(end)=LiL;
Nai(1)=Na0;       Nai(end)=NaL;

conci =[Cli Bi Lii Nai];

%% solutions
[t conc]=ode15s(@diagenesis,tspan,conci,[],x,diffc,poros,sed,irrigat,dx);
global fi u to irrB irrLi irrCl irrNa

%save flux_sens_52_5 conc t x poros sed irrigat irrB irrCl irrLi

%% PLotting of results
sym = ['k-';'b-';'r-'];
% Extract concentration profiles from output file
Clf     = conc(:,0*length(x)+1:1*length(x));
Bf      = conc(:,1*length(x)+1:2*length(x));
Lif     = conc(:,2*length(x)+1:3*length(x));
Naf     = conc(:,3*length(x)+1:4*length(x));
irrBf   = irrB(1:end);
irrLif  = irrLi(1:end);

for i=1:length(t)      
    subplot(1,4,1)
    plot(Clf(end,:),x,sym(i,:),'LineWidth',2);
    xlabel('Cl / mM','fontsize',14,'fontweight','b');
    hold on; 
    
    subplot(1,4,2)
    plot(Bf(end,:),x,sym(i,:),'LineWidth',2);
    xlabel('B / mM','fontsize',14,'fontweight','b');
    hold on;
       
    subplot(1,4,3)
    plot(Lif(end,:),x ,sym(i,:),'LineWidth',2);
    xlabel('Li / mM','fontsize',14,'fontweight','b');
    hold on;
    
    subplot(1,4,4)
    plot(Naf(end,:),x ,sym(i,:),'LineWidth',2);
    xlabel('Na / mM','fontsize',14,'fontweight','b');
    hold on;
end

% Title
subplot(1,4,2)
title('59 / u = 0 cm/a','FontSize',20,'FontWeight','demi');
% Page orientation 
orient landscape
%print -depsc 59lstf2.eps

%% Output- fluxes
D = diffcoeff(P,T,S);                                     % diffusion coefficient                           

%% Fluxes for B
% source/sink term (irrigation)
irrB = sum(fi(2:end-1).*irrBf.*dx*10);                                          %mmol/m2/year			 	 

% fluxes across the sediment water interface
diffBt=((D(13)/to(1))*fi(1)*(Bf(end,1) - Bf(end,2))/dx)*10;                     % mmol/m2/year	
advBt= fi(1)*u(1)*Bf(end,1)*10;                                                 % mmol/m2/year   
fluxswiB= diffBt + advBt  -irrB;                                                % mmol/m2/year

% fluxes across the bottom of sediment domain
diffBb=((D(13)/to(end))*fi(end)*(Bf(end,end-1) - Bf(end,end))/dx)*10;           % mmol/m2/year  
advBb= fi(end)*u(end)*Bf(end,end)*10;                                           % mmol/m2/year
fluxbtB= diffBb + advBb;                                                        % mmol/m2/year

massbal_B= abs(fluxswiB)/abs(fluxbtB)*100;                                      % check the mass balance
relerror_B=(abs(fluxswiB)-abs(fluxbtB))/abs(fluxswiB)*100;                      % error on mass balance   

flux_B=[irrB diffBt diffBb advBt advBb fluxswiB fluxbtB massbal_B relerror_B];
                              
%% Fluxes for Li
% source/sink term (irrigation)
 irrLi = sum(fi(2:end-1).*irrLif.*dx*10);                                        % mmol/m2/year	

 % fluxes across the sediment water interface
 diffLit=((D(12)/to(1))*fi(1)*(Lif(end,1) - Lif(end,2))/dx)*10;                  % mmol/m2/year	
 advLit= fi(1)*u(1)*Lif(end,1)*10;                                               % mmol/m2/year   
 fluxswiLi= diffLit + advLit - irrLi;                                            % mmol/m2/year
 
% fluxes across the bottom of sediment domain
diffLib=((D(12)/to(end))*fi(end)*(Lif(end,end-1) - Lif(end,end))/dx)*10;         % mmol/m2/year  
advLib= fi(end)*u(end)*Lif(end,end)*10 ;                                         % mmol/m2/year
fluxbtLi= diffLib + advLib;                                                      % mmol/m2/year		 
 
massbal_Li= abs(fluxswiLi)/abs(fluxbtLi)*100;                                    % check the mass balance
relerror_Li=(abs(fluxswiLi)-abs(fluxbtLi))/abs(fluxswiLi)*100;                   % error on mass balance
 
flux_Li=[irrLi diffLit diffLib advLit advLib  fluxswiLi fluxbtLi massbal_Li relerror_Li];   
                
%save massbal_sens52_5 flux_B flux_Li

%%
function dconc = diagenesis(t,conc,x,D,poros,sed,irrigat,dx);
global fi u to irrB irrCl irrLi irrNa

Cl  = conc(1:length(x));
B   = conc(1*length(x)+1:2*length(x));
Li  = conc(2*length(x)+1:3*length(x));
Na  = conc(3*length(x)+1:4*length(x));

% seawater concentrations - upper boundary
Cl0 = 555.0;                % Cl [mM]
B0  = 0.5;                  % B [mM]
Li0 = 0.0257;               % Li [mM] 
Na0 = 464.4;                % Na [mM]

% initialisation for calculation of irrigation term
cCl = Cl0 *ones(size(x))';
cB  = B0 *ones(size(x))';
cLi = Li0 *ones(size(x))';
cNa = Na0 *ones(size(x))';

% upper boundary condition and initialization
dCl = zeros(size(x))';
dB  = zeros(size(x))';
dLi = zeros(size(x))';
dNa = zeros(size(x))';

% porosity profile
fi = ((poros(1)-poros(2))*exp(-poros(3)*x)+poros(2))';

% porewater advection and externally impressed fluid flow
u = (-sed(1)*fi(1))./ fi;

% tortuosity correction for diffusion coefficients
to = 1-2*log(fi);                                   % Boudreau (1997)   

% irrigation term
irr= irrigat(1)*(ones(size(x)))';                   % initialise vector size
xir = find(x>=irrigat(2));                          % all x-values > irrigation zone
irr(xir)=(irrigat(1)*exp(-(x(xir)-irrigat(2))/.1));

%% Solving differential equation for each constituent
%% Cl
[diff adv] = fdiffer(Cl,x,D(11),u,fi,dx);           % calling diffusion and advection 
irrCl=(irr(2:end-1).*(Cl(2:end-1)-cCl(2:end-1)));   % irrigation term
dCl(2:end-1) = diff - adv - irrCl; 

%% B
[diff adv] = fdiffer(B,x,D(13),u,fi,dx);            % calling diffusion and advection 
irrB=(irr(2:end-1).*(B(2:end-1)-cB(2:end-1)));      % irrigation term
dB(2:end-1) = diff - adv - irrB; 

%% Li
[diff adv] = fdiffer(Li,x,D(12),u,fi,dx);           % calling diffusion and advection
irrLi=(irr(2:end-1).*(Li(2:end-1)-cLi(2:end-1)));   % irrigation term
dLi(2:end-1) = diff - adv - irrLi; 

%% Na
[diff adv] = fdiffer(Na,x,D(14),u,fi,dx);           % calling diffusion and advection
irrNa=(irr(2:end-1).*(Na(2:end-1)-cNa(2:end-1)));   % irrigation term
dNa(2:end-1) = diff - adv - irrNa; 

%% lower boundary condition: 
% Dirichlet conditions: fix concentration

dCl(end)= 0;
dB(end)=0;
dLi(end) =0;
dNa(end)=0;

dconc = cat(1,dCl,dB,dLi,dNa);

%-----------------------------------------------------------------------------------------------------------
%% diffusion and advection term       
function [diff adv] = fdiffer(conc,x,D,u,poros,dx)
global fi to

% solutes 
Ds = (1./(to.*fi))*D;

difft1= fi(2:end-1).*((conc(3:end)-2.*conc(2:end-1)+conc(1:end-2))/(dx.^2));

difft2=(fi(2:end-1)-fi(1:end-2))/dx;

difft3=(fi(2:end-1)./to(2:end-1)).*((to(2:end-1)-to(1:end-2))/dx);


diff= Ds(2:end-1).*(difft1+(difft2-difft3).*((conc(2:end-1)-conc(1:end-2))/dx));

adv = u(2:end-1).*(conc(3:end)- conc(2:end-1))/dx;

%-----------------------------------------------------------------------------------------------------------
%% Calculation of diffusion coefficients
% after Li & Gregory (1974), Hayduk & Laudie (1974),
% and Boudreau (1997)
% D [cm2/a], p [bar], t [degC], S [g/kg]

function D = diffcoeff(P,T,S)

% dynamic viscosity of pure water at T, P=1 bar
muo = viscosity(1,T,0);
% dynamic viscosity of seawater at P, T, S 
mu = viscosity(P,T,S);

% O2:       Hayduk & Laudie (1974)
D(1) = 4.72E-9*(T+273.15)/(muo/100*27.9^.6);
% NO3-:     Boudreau (1997)
D(2) = (9.50+.388*T)*10^-6;
% Mn2+:     Boudreau (1997)
D(3) = (3.18+.155*T)*10^-6;
% Fe2+:     Boudreau (1997)
D(4) = (3.31+.150*T)*10^-6;
% SO42-:    Boudreau (1997)
D(5) = (4.88+.232*T)*10^-6;
% CH4:      Hayduk & Laudie (1974)
D(6) = 4.72E-9*(T+273.15)/(muo/100*(37.7^.6));
% HS-:      Boudreau (1997)
D(7) = (10.4+.273*T)*10^-6;
% NH4+:     Boudreau (1997)
D(8) = (9.50+.413*T)*10^-6;
% HCO3-:    Boudreau (1997)
D(9) = (5.06+.275*T)*10^-6;
% Ca2+:     Boudreau (1997)
D(10) = (3.60+.179*T)*10^-6;
% Cl-:      Boudreau (1997)
D(11) = (9.60+.438*T)*10^-6;
% Li+       Boudreau (1997)
D(12) = (4.43+.241*T)*10^-6;
% Na+       Boudreau (1997)
D(14) = (6.06+.297*T)*10^-6;

D = D * muo/mu * 365*24*3600;   % P and S correction using Stokes-Einstein relation

% Boric acid: D = 1.12 +- 0.02*10^-5 cm^2*s^-1 at T=25 and I = 0.5-0.6 M     Mackin (1986) 
% dynamic viscosity of seawater at P, T=25, S=29.2
muT1 = viscosity(1,25,29.2);
% dynamic viscosity of seawater at P, T, S=29.2
muT2 = viscosity(1,T,29.2);
Dboric = 1.12E-5;
% temperature correction using Stokes-Einstein relation
Dboric = ((Dboric * muT1 * (T+273.15)) / (298.15 * muT2)); 
% P and S correction using Stokes-Einstein relation
Dboric = Dboric * muT2/mu * 365*24*3600;                                                            
Dborate = 0.875*Dboric; % Boudreau and Canfield (1988)
D(13) = 0.5*(Dboric+Dborate);  

%-----------------------------------------------------------------------------------------------------------
%% Calculation of the dynamic viscosity of water 
% after Kukulka (1987) (see Boudreau, 1997)
%
% mu [cPoise=0.01 g/cm/s] = f(p,T,S) 
% for ranges of p=1-1000 bar, t=0-30 degC, and S=0-36

function mu = viscosity(P,T,S)

mu = 1.7910 - 6.144E-2*T +1.4510E-3*T^2 - 1.6826E-5*T^3 ...
    - 1.5290E-4*P + 8.3885E-8*P^2 + 2.4727E-3*S ...
    + T*(6.0574E-6*P - 2.6760E-9*P^2) ...
    + S*(4.8429E-5*T - 4.7172E-6*T^2 + 7.5986E-8*T^3);    


            
   
