package linMap;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.text.*;
import java.util.*;

import javax.swing.*;

import linMap.cellLineage.*;
import linMap.cxMap.*;
import linMap.cxProfile.CxPoint;
import linMap.cxProfile.CxProfile;
import linMap.geneNetwork.*;

import com.jgoodies.forms.builder.PanelBuilder;
import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;


/**
 * A <code>ControlPanel</code> object contains the primary LinMap controls,
 * including network and lineage parameters and controls that affect the 
 * global state of the visualisation model.
 * 
 * It listens for <code>SelectionEvents</code> and <code>ProfileEvents</code>
 * to determine when buttons should be enabled or disabled.
 * 
 * @author nic
 *
 */
public class ControlPanel extends JPanel implements SelectionListener, ProfileListener {

	/**
	 * Creates a new <code>ControlPanel</code> object.
	 * @param cxMapStack
	 * 			the visualisation model
	 * @param cxMapParams
	 * 			the complexity map parmaeters
	 * @param linParams
	 * 			the cell lineage parameters
	 * @param netParams
	 * 			the gene network parameters
	 */
	public ControlPanel(CxMapStack cxMapStack,
			CxMapParams cxMapParams, LineageParams linParams, 
			NetworkParams netParams) {
		this.generator = new Random();
		this.cxMapStack = cxMapStack;
		this.cxMapParams = cxMapParams;
		this.netParams = netParams;
		this.linParams = linParams;
		
		// construct components
		
		// network / lineage parameters
		JLabel sizeLabel = new JLabel("Size: ");
		sizeField = new JFormattedTextField(NumberFormat.getIntegerInstance());
		sizeField.setColumns(DEFAULT_FIELD_WIDTH);
		sizeField.setValue(new Integer(netParams.hiddenNodes));
		
		JLabel connLabel = new JLabel("Conn: ");
		connField = new JFormattedTextField(NumberFormat.getIntegerInstance());
		connField.setColumns(DEFAULT_FIELD_WIDTH);
		connField.setValue(new Integer(netParams.connectivity));
		
		JLabel fateLabel = new JLabel("# Fates: ");
		fateField = new JFormattedTextField(NumberFormat.getIntegerInstance());
		fateField.setColumns(DEFAULT_FIELD_WIDTH);
		fateField.setValue(new Integer(netParams.outputNodes-1));
		
		JLabel depthLabel = new JLabel("Depth: ");
		depthField = new JFormattedTextField(NumberFormat.getIntegerInstance());
		depthField.setColumns(DEFAULT_FIELD_WIDTH);
		depthField.setValue(new Integer(linParams.maxDepth));
		
		JLabel seedLabel = new JLabel("Seed: ");
		seedField = new JFormattedTextField(NumberFormat.getIntegerInstance());
		seedField.setColumns(DEFAULT_FIELD_WIDTH);
		seedField.setValue(new Integer(netParams.seed));

		// network / lineage buttons
		NewMapAction nma = new NewMapAction();
		JButton seedButton = new JButton("New Seed");
		seedButton.addActionListener(nma);
		seedButton.setActionCommand("newseed");
		seedButton.setMnemonic(KeyEvent.VK_S);

		JButton genButton = new JButton("Generate");
		genButton.addActionListener(nma);
		genButton.setActionCommand("generate");
		genButton.setMnemonic(KeyEvent.VK_G);

		// map parameters
		JLabel weightMinLabel = new JLabel("X Min: ");
		weightMinField = new JFormattedTextField(NumberFormat.getNumberInstance());
		weightMinField.setValue(new Double(cxMapParams.weightMin));
		
		JLabel weightMaxLabel = new JLabel("X Max: ");
		weightMaxField = new JFormattedTextField(NumberFormat.getNumberInstance());
		weightMaxField.setValue(new Double(cxMapParams.weightMax));	
		
		JLabel weightStepLabel = new JLabel("X Step: ");
		weightStepField = new JFormattedTextField(NumberFormat.getNumberInstance());
		weightStepField.setValue(new Double(cxMapParams.weightStep));	
				
		JLabel lambdaMinLabel = new JLabel("Y Min: ");
		lambdaMinField = new JFormattedTextField(NumberFormat.getNumberInstance());
		lambdaMinField.setValue(new Double(cxMapParams.lambdaMin));
		
		JLabel lambdaMaxLabel = new JLabel("Y Max: ");
		lambdaMaxField = new JFormattedTextField(NumberFormat.getNumberInstance());
		lambdaMaxField.setValue(new Double(cxMapParams.lambdaMax));	

		JLabel lambdaDivLabel = new JLabel("Y Dvisions: ");
		lambdaDivField = new JFormattedTextField(NumberFormat.getNumberInstance());
		lambdaDivField.setValue(new Integer(cxMapParams.lambdaDiv));	
			
		// map control buttons
		DivideAction da = new DivideAction();
		
		JButton wtButton = new JButton("Subdivide Weights");
		wtButton.addActionListener(da);
		wtButton.setActionCommand("divideweight");
		wtButton.setMnemonic(KeyEvent.VK_W);
		
		JButton lambdaButton = new JButton("Subdivide Lambda");
		lambdaButton.addActionListener(da);
		lambdaButton.setActionCommand("dividelambda");
		lambdaButton.setMnemonic(KeyEvent.VK_L);
		
		JButton bothButton = new JButton("Subdivide Both");
		bothButton.addActionListener(da);
		bothButton.setActionCommand("divideboth");
		bothButton.setMnemonic(KeyEvent.VK_B);
		
		ZoomAction za = new ZoomAction();
		
		zoomInButton = new JButton("Zoom In");
		zoomInButton.addActionListener(za);
		zoomInButton.setActionCommand("zoomin");
		zoomInButton.setEnabled(false);
		zoomInButton.setMnemonic(KeyEvent.VK_I);
		
		zoomOutButton = new JButton("Zoom Out");
		zoomOutButton.addActionListener(za);
		zoomOutButton.setActionCommand("zoomout");
		zoomOutButton.setEnabled(false);
		zoomOutButton.setMnemonic(KeyEvent.VK_O);
		
		clearZoomButton = new JButton("Clear Zoom");
		clearZoomButton.addActionListener(za);
		clearZoomButton.setActionCommand("clearzoom");
		clearZoomButton.setEnabled(false);
		clearZoomButton.setMnemonic(KeyEvent.VK_C);
		
		ProfileAction ca = new ProfileAction();
		
		contourButton = new JButton("Write Contour");
		contourButton.addActionListener(ca);
		contourButton.setActionCommand("writecontour");
		contourButton.setEnabled(false);
		contourButton.setMnemonic(KeyEvent.VK_R);
		
		// layout components

		FormLayout layout = new FormLayout(
				"right:pref, 3dlu, fill:pref:grow, 3dlu, right:pref, 3dlu, fill:pref:grow",
				"p, " + MEDIUM_GAP +											// network/lineage separator
				"p, " + SMALL_GAP + "p, " + SMALL_GAP + "p, " + LARGE_GAP + 
				"p, " + MEDIUM_GAP +											// map separator
				"p, " + SMALL_GAP + "p, " + SMALL_GAP + "p, " + LARGE_GAP + 
				"p, " + MEDIUM_GAP +											// control separator
				"p, " + SMALL_GAP + "p, " + SMALL_GAP + "p, " + SMALL_GAP + "p");								
		

		layout.setColumnGroups(new int[][]{{1, 5}, {3, 7}});
		
		PanelBuilder builder = new PanelBuilder(layout, this);
		builder.setDefaultDialogBorder();
		
		CellConstraints cc = new CellConstraints();
		
		builder.addSeparator("Network/Lineage Parameters", cc.xyw(1, 1, 7));
		builder.add(sizeLabel,		cc.xy(1, 3));
		builder.add(sizeField,		cc.xy(3, 3));
		builder.add(connLabel,		cc.xy(1, 5));
		builder.add(connField,		cc.xy(3, 5));		
		builder.add(fateLabel,		cc.xy(5, 3));
		builder.add(fateField,		cc.xy(7, 3));	
		builder.add(depthLabel,		cc.xy(5, 5));
		builder.add(depthField,		cc.xy(7, 5));	
		builder.add(seedLabel,		cc.xy(1, 7));
		builder.add(seedField,		cc.xy(3, 7));	
		builder.add(seedButton,		cc.xyw(5, 7, 3));
		
		builder.addSeparator("Map Parameters", cc.xyw(1, 9, 7));
		builder.add(lambdaMinLabel,	cc.xy(1, 11));
		builder.add(lambdaMinField, cc.xy(3, 11));
		builder.add(lambdaMaxLabel, cc.xy(1, 13));
		builder.add(lambdaMaxField, cc.xy(3, 13));
		builder.add(lambdaDivLabel, cc.xy(1, 15));
		builder.add(lambdaDivField, cc.xy(3, 15));
		builder.add(weightMinLabel, cc.xy(5, 11));
		builder.add(weightMinField, cc.xy(7, 11));
		builder.add(weightMaxLabel, cc.xy(5, 13));
		builder.add(weightMaxField, cc.xy(7, 13));
		builder.add(weightStepLabel, cc.xy(5, 15));
		builder.add(weightStepField, cc.xy(7, 15));
		
		builder.addSeparator("Controls", cc.xyw(1, 17, 7));
		builder.add(genButton,		cc.xyw(1, 19, 3));
		builder.add(wtButton,		cc.xyw(1, 21, 3));
		builder.add(lambdaButton,	cc.xyw(1, 23, 3));
		builder.add(zoomInButton,	cc.xyw(5, 19, 3));
		builder.add(zoomOutButton,	cc.xyw(5, 21, 3));
		builder.add(clearZoomButton,cc.xyw(5, 23, 3));
		builder.add(bothButton,     cc.xyw(1, 25, 3));
		builder.add(contourButton,  cc.xyw(5, 25, 3));
	}
	
	public synchronized void selectionReceived(SelectionEvent e) {
		zoomInButton.setEnabled(true);
		if (e.params() != null) {
			this.newParams = e.params();	
		} else {
			if (cxMapStack.isMaxZoom()) {
				zoomInButton.setEnabled(false);
			}
			this.newParams = null;
		}

	}
	
	public synchronized void profileReceived(ProfileEvent e) {
		contourButton.setEnabled(true);
		if (e.getStart() != null && e.getEnd() != null) {
			this.contourStart = e.getStart();
			this.contourEnd = e.getEnd();
		} else {
			contourButton.setEnabled(false);
		}
	}
	
	/**
	 * A <code>NewMapAction</code> generates a new visualisation model.
	 * 
	 * @author nic
	 *
	 */
	private class NewMapAction implements ActionListener {
	
		public void actionPerformed(ActionEvent e) {
			String command = e.getActionCommand();
			if (command == "newseed") 
				seedField.setValue(generator.nextInt());
			else if (command == "generate") {
				getParamsFromFields();
				setCursor(new Cursor(Cursor.WAIT_CURSOR));
				ComplexityMap cxMap = new ComplexityMap(cxMapParams, linParams, netParams);
				cxMapStack.resetStack(cxMap);
				setCursor(Cursor.getDefaultCursor());
			}
		}

		private void getParamsFromFields() {
			netParams.hiddenNodes = getIntValue(sizeField);
			netParams.outputNodes = getIntValue(fateField) + 1;	// add 1 for div output
			netParams.seed = getIntValue(seedField);
			linParams.maxDepth = getIntValue(depthField);
			cxMapParams.lambdaMin = getDoubleValue(lambdaMinField);
			cxMapParams.lambdaMax = getDoubleValue(lambdaMaxField);
			cxMapParams.weightMin = getDoubleValue(weightMinField);
			cxMapParams.weightMax = getDoubleValue(weightMaxField);
		}
		
		private int getIntValue(JFormattedTextField field) {
			Number value = (Number)field.getValue();
			return value.intValue();
		}
		
		private double getDoubleValue(JFormattedTextField field) {
			Number value = (Number)field.getValue();
			return value.doubleValue();
		}	
	}
	
	/**
	 * A <code>DivideAction</code> updates the current visualisation model by
	 * subdividing either the x axis, the y axis, or both axes.
	 * 
	 * @author nic
	 *
	 */
	private class DivideAction implements ActionListener {

		public void actionPerformed(ActionEvent e) {
			final String command = e.getActionCommand();
			setCursor(new Cursor(Cursor.WAIT_CURSOR));
				if (command == "divideboth") {
					cxMapStack.subdivideBoth();
				} else if (command == "divideweight") {
					cxMapStack.subdivideWeights();
				} else if (command == "dividelambda") {
					cxMapStack.subdivideLambda();
				}
			setCursor(Cursor.getDefaultCursor());	
		}
	}
	
	/**
	 * A <code>ZoomAction</code> updates the current visualisation model by
	 * either zooming in, zooming out or clearing current zoomed regions.
	 * 
	 * @author nic
	 *
	 */
	private class ZoomAction implements ActionListener {
		
		public void actionPerformed(ActionEvent e) {
			String command = e.getActionCommand();
			if (command == "zoomin") {
				cxMapStack.zoomIn(newParams);
			} else if (command == "zoomout") {
				cxMapStack.zoomOut();
			} else if (command == "clearzoom") {
				cxMapStack.clearZoom();
			}
			if (cxMapStack.isMinZoom()) {
				zoomOutButton.setEnabled(false);
			} else {
				zoomOutButton.setEnabled(true);
			}
			if (cxMapStack.isMaxZoom()) {
				zoomInButton.setEnabled(false);
				clearZoomButton.setEnabled(false);
			} else {
				zoomInButton.setEnabled(true);
				clearZoomButton.setEnabled(true);
			}
		}
	}

	/**
	 * A <code>ProfileAction</code> writes out a data file containing
	 * the complexity values for the currently selected profile.
	 * 
	 * @author nic
	 *
	 */
	private class ProfileAction implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			String command = e.getActionCommand();
			if (command == "writecontour") {
				int returnVal = fc.showSaveDialog(ControlPanel.this);
				if (returnVal == JFileChooser.APPROVE_OPTION) {
					File file = fc.getSelectedFile();
					System.out.println(contourStart);
					System.out.println(contourEnd);
					CxProfile contour = new CxProfile(cxMapParams, linParams, netParams, 
							contourStart, contourEnd, 6);
					try {
						FileWriter outFile = new FileWriter(file);
						PrintWriter out = new PrintWriter(outFile);
						contour.writeCx(out);
					} catch (IOException ex) {
						ex.printStackTrace();
					}
				}
			}
		}
	}
	
	/**
	 * A <code>FileAction</code> saves or retrieves a visualisation model from a file.
	 * 
	 * @author nic
	 *
	 */
	public class FileAction implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			String command = e.getActionCommand();
			if (command == "loadmap") {
				int returnVal = fc.showOpenDialog(ControlPanel.this);
				if (returnVal == JFileChooser.APPROVE_OPTION) {
					File file = fc.getSelectedFile();
					cxMapStack.loadMap(file);
				}
			} else if (command == "savemap") {
		        int returnVal = fc.showSaveDialog(ControlPanel.this);
		        if (returnVal == JFileChooser.APPROVE_OPTION) {
		            File file = fc.getSelectedFile();
		            cxMapStack.saveMap(file);
		        }
			}
		}
	}
	
	private static final String SMALL_GAP = "2dlu, ";
	private static final String MEDIUM_GAP = "4dlu, ";
	private static final String LARGE_GAP = "8dlu, ";

	private static final long serialVersionUID = -7138074047158636835L;
	
	private LineageParams linParams;
	private NetworkParams netParams;
	private CxMapParams cxMapParams;
	
	private CxPoint contourStart;
	private CxPoint contourEnd;

	private Random generator;
		
	private JFormattedTextField sizeField;
	private JFormattedTextField connField;
	private JFormattedTextField fateField;
	private JFormattedTextField seedField;
	private JFormattedTextField depthField;
	
	private JFormattedTextField weightMinField;
	private JFormattedTextField weightMaxField;
	private JFormattedTextField weightStepField;
	private JFormattedTextField lambdaMinField;
	private JFormattedTextField lambdaMaxField;
	private JFormattedTextField lambdaDivField;
	
	private JButton zoomInButton;
	private JButton zoomOutButton;
	private JButton clearZoomButton;
	private JButton contourButton;
	
	private JFileChooser fc = new JFileChooser();
	
	private CxMapStack cxMapStack;
	
	private CxMapParams newParams = null;
	
	private static final int DEFAULT_FIELD_WIDTH = 6;
	
}