package linMap;

import java.io.*;
import java.util.*;

import linMap.cxMap.*;
import linMap.cxMap.CxBlock.ComplexityType;


/** 
 * A <code>CxMapStack</code> object contains a LinMap visualisation model.
 * The model is structured as a 'stack' (implemented using an <code>ArrayList</code>) 
 * of <code>ComplexityMap</code> objects at increasing magnified resolutions.  
 * <code>CxMapStack</code> also provides functions for manipulating the state of the 
 * model, and fires an <code>UpdateMapEvent</code> whenever the state of the model 
 * changes.  The local variable <code>index</code> is effectively a pointer to the 
 * currently active 'base' map.
 * 
 * @author nic
 *
 */
public class CxMapStack {
	
	/** 
	 * Constructs a new model based on <code>linMap.cxMap</code>.
	 * @param linMap.cxMap
	 * 			the initial <code>ComplexityMap</code> in the model
	 */
	public CxMapStack(ComplexityMap cxMap) {
		this.maps = new ArrayList<ComplexityMap>();
		this.maps.add(cxMap);
		this.index = 0;
	}

	public synchronized void addUpdateMapListener(UpdateMapListener l) {
		this.updateMapListeners.add(l);
	}

	public synchronized void removeUpdateMapListener(UpdateMapListener l) {
		this.updateMapListeners.remove(l);
	}

	/**
	 * Accessor for the current map.
	 * @return
	 * 		the current base <code>ComplexityMap</code>
	 */
	public ComplexityMap getCurrentMap() {
		return maps.get(index);
	}

	/**
	 * Calculate and return the maximum complexity value for a particular
	 * <code>ComplexityType</code>
	 * @param type
	 * 			the type of complexity to consider
	 * @return
	 * 			the maximum complexity value for the complexity type
	 */
	public double getMaxComplexity(ComplexityType type) {
		double curMax = 0.0;
		for (ComplexityMap cxMap : maps) {
			if (cxMap.getCxMax(type) > curMax)
				curMax = cxMap.getCxMax(type);
		}
		return curMax;
	}
	
	/**
	 * Accessor for all currently visible maps, that is, from the 
	 * current base map up to the top of the stack.
	 * @return
	 * 		a list of the currently visible maps
	 */
	public ArrayList<ComplexityMap> getVisibleMaps() {
		ArrayList<ComplexityMap> visibleMaps = new ArrayList<ComplexityMap>();
		for (int i=index, n=maps.size(); i<n; i++) {
			visibleMaps.add(maps.get(i));
		}
		return visibleMaps;
	}
		
	/**
	 * Clear the current model and replace with <code>linMap.cxMap</code>
	 * @param linMap.cxMap
	 */
	public void resetStack(ComplexityMap cxMap) {
		this.maps.clear();
		this.maps.add(cxMap);
		this.index = 0;
		this.fireUpdateMapEvent(true);
	}

	/** 
	 * Subdivide the x axis of the current base map.
	 */
	public void subdivideWeights() {
		this.getCurrentMap().subdivideWeights();
		this.fireUpdateMapEvent(true);
	}
	
	/**
	 * Subdivide the y axis of the current base map.
	 */
	public void subdivideLambda() {
		this.getCurrentMap().subdivideLambda();
		this.fireUpdateMapEvent(true);
	}
	
	/**
	 * Subdivide both axes of the current base map
	 */
	public void subdivideBoth() {
		this.getCurrentMap().subdivideWeights();
		this.getCurrentMap().subdivideLambda();
		this.fireUpdateMapEvent(true);
	}
	
	/**
	 * Zoom in to the currently selected region, creating a new 
	 * <code>ComplexityMap</code> object if necessary (if the current base
	 * map is at the top of the stack).
	 * 
	 * @param newParams
	 * 			the map parameters that define the new zoomed region.
	 */
	public void zoomIn(CxMapParams newParams) {
		if (!this.isMaxZoom()) {
			index++;
			this.fireUpdateMapEvent(false);
		} else {
			if (newParams != null) {
				ComplexityMap newCxMap = new ComplexityMap(this.getCurrentMap(), newParams);
				this.addMap(newCxMap);
			}
		}
	}
	
	/**
	 * Zoom out, if the current base map is not at the bottom of the stack.
	 */
	public void zoomOut() {
		if (index > 0) {
			index--;
			this.fireUpdateMapEvent(false);
		}
	}
	
	/**
	 * Clear any magnified maps above the current base map.
	 */
	public void clearZoom() {
		for (int i=maps.size()-1; i>index; i--) {
			maps.remove(i);
		}
		this.fireUpdateMapEvent(false);
	}

	/**
	 * Check if currently at maximum magnification level.
	 * @return <code>true</code> if the current map is at the top of the stack
	 */
	public boolean isMaxZoom() {
		return (index == maps.size()-1);
	}
	
	/**
	 * Check if currently at the minimum magnification level.
	 * @return <code>true</code> if the current map is at the bottom of the stack
	 */
	public boolean isMinZoom() {
		return (index == 0);
	}
	
	/**
	 * Retrieve a <code>CxMapStack</code> object from <code>file</code>
	 */
	public void loadMap(File file) {
		try {
			FileInputStream fis = new FileInputStream(file);
			ObjectInputStream in = new ObjectInputStream(fis);
			ArrayList<?> newCxMapStack = (ArrayList<?>)in.readObject();
			maps.clear();
			for (int i=0; i<newCxMapStack.size(); i++) {
				maps.add((ComplexityMap)newCxMapStack.get(i));
			}
			in.close();
			this.index = 0;
			this.fireUpdateMapEvent(true);
		} catch(IOException ex) {
			ex.printStackTrace();
		} catch (ClassNotFoundException ex) {
			ex.printStackTrace();
		}
	}
	
	/**
	 * Save this <code>CxMapStack</code> object to <code>file</code>
	 */
	public void saveMap(File file) {
		try {
			FileOutputStream fos = new FileOutputStream(file);
			ObjectOutputStream out = new ObjectOutputStream(fos);
			out.writeObject(maps);
			out.close();
		} catch(IOException ex) {
			ex.printStackTrace();
		}		
	}
	
	private void addMap(ComplexityMap cxMap) {
		maps.add(cxMap);
		index++;
		this.fireUpdateMapEvent(false);
	}
		
	private synchronized void fireUpdateMapEvent(boolean recalculate) {
		UpdateMapEvent update = new UpdateMapEvent(this, recalculate);
		Iterator listenerIt = updateMapListeners.iterator();
		while (listenerIt.hasNext()) {
			((UpdateMapListener) listenerIt.next()).updateMapReceived(update);
		}
	}
	
	private ArrayList<UpdateMapListener> updateMapListeners = new ArrayList<UpdateMapListener>();
	
	private ArrayList<ComplexityMap> maps;
	private int index;

}
