package linMap;

import java.awt.*;
import java.awt.geom.*;
import java.util.*;
import java.util.List;

import javax.swing.*;

import linMap.cellLineage.*;

/**
 * A <code>LineagePanel</code> displays a cell lineage in graphical form.  A 
 * <code>LineagePanel</code> listens for <code>FocusLineageEvents</code> and
 * updates the displayed lineage accordingly.
 * 
 * @author nic
 *
 */
public class LineagePanel extends JPanel implements FocusLineageListener {

	private static final long serialVersionUID = 3432000020699335746L;
	
	public static final int MARGIN = 30;	// space around the cell lineage
	public static final int BORDER = 1;		// the width of the cell border
	public static final int CELL_SIZE = 12; // the default size of a cell
	
	// the colours used for the first 8 cell fates
	// NB: will need to be modified if > 8 cell fates are required
	public static final Color[] colours = {Color.RED, Color.GREEN, Color.BLUE, 
		Color.YELLOW, Color.CYAN, Color.MAGENTA, Color.ORANGE, Color.PINK };

	private Lineage lin;			// the current lineage
	private float levelHeight;		// the current level spacing

	/**
	 * Construct a <code>Lineage Panel</code>.
	 * @param lin the initial lineage to display
	 */
	public LineagePanel(Lineage lin) {
		this.lin = lin;
	}
	
	/** 
	 * Set the lineage to be displayed and repaint the panel.
	 * @param lin the new lineage to display
	 */
	public void setLineage(Lineage lin) {
		this.lin = lin;
		this.repaint();
	}

	public synchronized void focusLineageReceived(FocusLineageEvent e) {
		this.lin = e.lineage();
		this.repaint();	
	}
	
	public void paintComponent(Graphics g) {
		Graphics2D g2 = (Graphics2D)g;

		// draw the background
		g2.setPaint(Color.LIGHT_GRAY);
		Rectangle2D bg = new Rectangle2D.Float(0, 0, getWidth(), getHeight());
		g2.fill(bg);

		// calculate cell and level spacing
		float cellSpacing = (float)((getWidth() - MARGIN) / 2);
		levelHeight = (getHeight() - (MARGIN/2)) / (lin.getMaxDepth() + 1);
		
		// recursively draw the cell lineage
		drawCell(g2, getWidth()/2, MARGIN, cellSpacing, 0, lin.getZygote());
	}

	/**
	 * Recursively draws the cells in a cell lineage.
	 * @param g2 the graphics context
	 * @param x the x position on which to centre the current cell
	 * @param y the y position on which to centre the current cell
	 * @param cellSpacing the spacing between cells for the current level
	 * @param cell the cell object to draw
	 */
	private void drawCell(Graphics2D g2, float x, float y, float cellSpacing, int depth, Cell cell) {
		// calculate cell size
		float cellWidth = CELL_SIZE;
		float borderWidth = BORDER;
		
		while ((Math.pow(2.0, depth) * ( cellWidth + 2*BORDER + 3 )) >= (getWidth() - (2*BORDER)))
			cellWidth = cellWidth - 1;

		if (cellWidth < 4) {
			borderWidth = 0;
			if (cellWidth < 1)
				cellWidth = 1;
		}
		
		float curX = x - (cellWidth / 2);
		float curY = y - (CELL_SIZE / 2);
		List<Integer> diff = cell.getFate().getDiff();
		Collections.sort(diff);
		float dSize = cellWidth / diff.size();
		
		// draw background
		Rectangle2D cellRect = new Rectangle2D.Float(curX - borderWidth, curY - borderWidth, 
				cellWidth + (2*borderWidth), CELL_SIZE + (2*borderWidth));
		g2.setPaint(Color.BLACK);
		g2.fill(cellRect);
		
		// draw differentation state
		for (int diffID : diff) {
			if (diffID != Fate.NON_DIFF) {
				g2.setPaint(colours[diffID]);
			} else {
				g2.setPaint(Color.BLACK);
			}
			Rectangle2D diffRect = new Rectangle2D.Float(curX, curY, dSize, CELL_SIZE);
			g2.fill(diffRect);
			curX += dSize;
		}
		
		// draw children
		if (!cell.getFate().terminal()) {
			cellSpacing /= 2;
			drawConnect(g2, x, y, cellSpacing);
			drawCell(g2, x-cellSpacing, y+levelHeight, cellSpacing, depth+1, cell.getLeft());
			drawCell(g2, x+cellSpacing, y+levelHeight, cellSpacing, depth+1, cell.getRight());
		}
	}
	
	/** 
	 * Draw the connecting lines between a cell and its children.
	 * @param g2 the graphics context
	 * @param x the x position on which the parent cell is centred
	 * @param y the y position on which the parent cell is centred
	 * @param cellSpacing the spacing between cells for the children's level
	 */
	private void drawConnect(Graphics2D g2, float x, float y, float cellSpacing) {
		g2.setColor(Color.BLACK);

		float yTop = y + (CELL_SIZE/2);
		float yMid = y + (levelHeight/2);
		Line2D vTopLine = new Line2D.Float(x, yTop, x, yMid);
		g2.draw(vTopLine);

		float xStart = x + cellSpacing;
		float xEnd = x - cellSpacing;
		Line2D hMidLine = new Line2D.Float(xStart, yMid, xEnd, yMid);
		g2.draw(hMidLine);

		float yBot = y + levelHeight - (CELL_SIZE/2);
		Line2D vBotLineL = new Line2D.Float(xStart, yMid, xStart, yBot);
		g2.draw(vBotLineL);
		Line2D vBotLineR = new Line2D.Float(xEnd, yMid, xEnd, yBot);
		g2.draw(vBotLineR);
	}
	
}
