package linMap;

import javax.swing.*;

import linMap.cxMap.*;
import linMap.cxMap.CxBlock.ComplexityType;
import linMap.vTextIcon.*;

import com.jgoodies.forms.builder.*;
import com.jgoodies.forms.layout.*;

/**
 * A <code>MapBorderPanel</code> object contains a <code>MapPanel</code> and
 * adds some additional information and functionality.  Namely: axis labels
 * and ranges, a scale for the complexity gradient, and controllers to select
 * complexity type and lock/unlock the map.
 * 
 * @author nic
 *
 */
public class MapBorderPanel extends JPanel implements UpdateMapListener, 
		FocusLineageListener, ResetListener {
	
	/** 
	 * Constructs a new <code>MapBorderPanel</code>.
	 * @param mf
	 * 			the <code>MapFrame</code>containing the object
	 * @param cxMapStack
	 * 			the visualisation model viewed by the map
	 */
	public MapBorderPanel(MapFrame mf, CxMapStack cxMapStack) {
	
		this.mf = mf;
		this.cxMapStack = cxMapStack;
		double cxMax = cxMapStack.getMaxComplexity(curCxType);
		this.colours = new CxColour(curCxType, cxMax);
		this.cxScalePanel = new CxScalePanel(colours);

		mp = new MapPanel(this.cxMapStack.getVisibleMaps(), colours, curCxType, cxMax);
		mapButtons = new MapButtonPanel(this);

		this.mf.setTitle("Map - " + (String)mapButtons.cxTypeList.getSelectedItem());	

		// create components
		
		JLabel lambdaLabel = new JLabel();
		lambdaLabel.setIcon(new VTextIcon(lambdaLabel, "lambda"));
		lambdaMaxLabel = new JLabel();
		lambdaMinLabel = new JLabel();
		JLabel weightLabel = new JLabel("weight factor");
		weightMaxLabel = new JLabel();
		weightMinLabel = new JLabel();

		JLabel cxLabel = new JLabel();
		cxLabel.setIcon(new VTextIcon(cxLabel, "complexity"));
		JLabel cxMinLabel = new JLabel(Double.toString(0.0));
		this.cxMaxLabel = new JLabel();
		this.updateLabels();
		
		// layout components

		FormLayout layout = new FormLayout(
				"right:min, 3dlu, left:20dlu, center:80dlu:grow, right:20dlu, 10dlu, 8dlu, 3dlu, 20dlu",
				"top:min, center:80dlu:grow, bottom:min, 3dlu, pref, 10dlu, min");
		
		layout.setColumnGroups(new int[][]{{1, 3, 5}});
		layout.setRowGroups(new int[][]{{1, 3, 5}});
		
		PanelBuilder builder = new PanelBuilder(layout, this);
		builder.setDefaultDialogBorder();

		CellConstraints cc = new CellConstraints();
		
		builder.add(lambdaMaxLabel, 	cc.xy(1, 1));
		builder.add(lambdaLabel, 		cc.xy(1, 2));
		builder.add(lambdaMinLabel,		cc.xy(1, 3));
		builder.add(weightMinLabel,		cc.xy(3, 5));
		builder.add(weightLabel,		cc.xy(4, 5));
		builder.add(weightMaxLabel,		cc.xy(5, 5));
		builder.add(mp,					cc.xywh(3, 1, 3, 3));
		builder.add(cxScalePanel,		cc.xywh(7, 1, 1, 3));
		builder.add(cxMaxLabel,			cc.xy(9, 1));
		builder.add(cxLabel,			cc.xy(9, 2));
		builder.add(cxMinLabel,			cc.xy(9, 3));
		builder.add(mapButtons, 		cc.xyw(3, 7, 3));
	}

	/* Event handling routines *************************************/
	
	public synchronized void updateMapReceived(UpdateMapEvent e) {
		this.updateMap();
		mp.recalculateMap();
		mp.updateFocus();
	}
	
	public synchronized void focusLineageReceived(FocusLineageEvent e) {
		updateMap();
		mp.setNormLoc(e.currentLoc());
		mp.resetSelection();
		mp.fireSelectionEvent();
		mp.resetContour();
		mp.fireContourEvent();
		mp.repaint();
	}

	public synchronized void resetReceived(ResetEvent e) {
		mp.resetSelection();
		mp.resetContour();
		mp.repaint();
	}
	
	public synchronized void addSelectionListener(SelectionListener l) {
		mp.selectionListeners.add(l);
	}

	public synchronized void addContourListener(ProfileListener l) {
		mp.contourListeners.add(l);
	}
	
	public synchronized void addFocusLineageListener(FocusLineageListener l) {
		mp.focusListeners.add(l);
	}
	
	public synchronized void addResetListener(ResetListener l) {
		mp.resetListeners.add(l);
	}
	
	public synchronized void removeSelectionListener(SelectionListener l) {
		mp.selectionListeners.remove(l);
	}

	public synchronized void removeContourListener(ProfileListener l) {
		mp.contourListeners.remove(l);
	}
	
	public synchronized void removeFocusLineageListener(FocusLineageListener l) {
		mp.focusListeners.remove(l);
	}
	
	public synchronized void removeResetListener(ResetListener l) {
		mp.resetListeners.remove(l);
	}

	/* Map updating / modification routines ************************/
	
	/**
	 * Update the <code>MapPanel</code>.  Saves the current focus location 
	 * as parameters first so that any change in the map due to zooming, 
	 * etc. will be reflected in the new focus location.
	 */
	protected void updateMap() {
		this.updateLabels();
		mp.focusToNormLoc();
		mp.normLocToParams(); 		
		mp.setMap(cxMapStack.getVisibleMaps(), colours, curCxType, cxMapStack.getMaxComplexity(curCxType));
		mp.paramsToNormLoc();
		mp.normLocToFocus();
		mp.resetSelection();		
	}

	protected void updateCxType(ComplexityType type) {
		curCxType = type;
		colours.setCxType(type, cxMapStack.getMaxComplexity(type));
		cxScalePanel.updateColours(colours);
		updateMap();
		mf.setTitle("Map - " + type.getLabel());
	}
	
	/**
	 * Updates whether borders around zoomed regions are painted.
	 * @param showBorder
	 * 			<code>true</code> if borders are to be painted
	 */
	protected void updateBorder(boolean showBorder) {
		mp.showBorder = showBorder;
		mp.repaint();
	}

	/**
	 * Locks the current map view.  Removes all listeners, disables mouse
	 * interaction and other controllers.
	 */
	protected void lockMap() {
		LinMap.removeMap(this);
		mp.contourListeners.clear();
		mp.focusListeners.clear();
		mp.selectionListeners.clear();
		cxMapStack.removeUpdateMapListener(this);
		mapButtons.cxTypeList.setEnabled(false);
		mapButtons.borderButton.setEnabled(false);
		mp.disableMouse();
		mp.showFocus = false;
		mp.showContour = false;
		mp.showSelection = false;
		mp.repaint();
	}

	/**
	 * Unlocks the current map view.  Adds listeners, enables mouse interaction
	 * and other controllers.
	 */
	protected void unlockMap() {
		LinMap.addMap(this);
		mp.enableMouse();
		this.updateMap();
		mp.recalculateMap();
		mp.updateFocus();
		mapButtons.cxTypeList.setEnabled(true);
		mapButtons.borderButton.setEnabled(true);
		mp.showFocus = true;
		mp.showContour = true;
		mp.showSelection = true;
		mp.repaint();
	}
	
	private void updateLabels() {
		ComplexityMap currentMap = cxMapStack.getCurrentMap();
		int tmp = (int)(currentMap.getWeightMax() * PREC_TWO);
		weightMaxLabel.setText(Double.toString(tmp/PREC_TWO));
		tmp = (int)(currentMap.getWeightMin() * PREC_TWO);
		weightMinLabel.setText(Double.toString(tmp/PREC_TWO));
		tmp = (int)(currentMap.getLambdaMax() * PREC_TWO);
		lambdaMaxLabel.setText(Double.toString(tmp/PREC_TWO));
		tmp = (int)(currentMap.getLambdaMin() * PREC_TWO);
		lambdaMinLabel.setText(Double.toString(tmp/PREC_TWO));
		tmp = (int)(cxMapStack.getMaxComplexity(curCxType) * PREC_TWO); 
		cxMaxLabel.setText(Double.toString(tmp/PREC_TWO));
	}
	
	protected MapFrame mf;
	private CxMapStack cxMapStack;
	private MapPanel mp;
	private MapButtonPanel mapButtons;
	private ComplexityType curCxType = ComplexityType.NONDET_WT;
	private CxColour colours;
	private CxScalePanel cxScalePanel;
	
	private JLabel cxMaxLabel;
	private JLabel lambdaMinLabel;
	private JLabel lambdaMaxLabel;
	private JLabel weightMinLabel;
	private JLabel weightMaxLabel;
	
	
	private static final double PREC_TWO = 100.0;
	private static final long serialVersionUID = -5777872152459921091L;

}

