package linMap;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.util.ArrayList;
import java.util.Iterator;
import javax.swing.*;

import linMap.cxMap.*;
import linMap.cxMap.CxBlock.ComplexityType;
import linMap.cxProfile.CxPoint;

/** 
 * A <code>MapPanel</code> object displays a heat map.  The heat map is both
 * a view - in that it provides a perspective on the visualisation model - 
 * and a controller - in that provides an interface by which the model may be 
 * manipulated.
 * <br/>
 * A <code>MapPanel</code> listens for <code>UpdateMapEvents</code> to ensure that
 * the displayed view reflects changes to the underlying model.  It sends
 * <code>SelectionEvents</code>, <code>ProfileEvents</code> and 
 * <code>FocusLineageEvents</code> to indicate user interaction with the map.
 * <br/>
 * A <code>MapPanel</code> can be locked or unlocked.  The default state is 
 * unlocked.  When a <code>MapPanel</code> is locked, it no longer listens for, or 
 * sends, events, or responds to mouse actions.  Rather, it provides a 'frozen'
 * view of a past state of the visualisation model.  When unlocked, the
 * <code>MapPanel</code> will resume being linked to the visualisation model.
 * 
 * @author nic
 *
 */
public class MapPanel extends JPanel {

	/**
	 * Construct a <code>MapPanel</code>.
	 * 
	 * @param linMap.cxMap
	 *            the initial complexity map to display
	 * @param lp
	 *            the lineage panel to be linked
	 */
	public MapPanel(ArrayList<ComplexityMap> cxMapStack, CxColour colours,
			ComplexityType curCxType, double cxMax) {
		this.setOpaque(true);
		this.enableMouse();
		this.prevSize = this.getSize();	// store current size so we know when window has been resized
		setMap(cxMapStack, colours, curCxType, cxMax);
	}

	/**
	 * Start handling the following mouse events: 
	 * clicking (for focus), pressing, releasing and dragging 
	 * (for selecting regions and profiles).
	 */
	protected void enableMouse() {
		MouseHandler handler = new MouseHandler();
		this.addMouseListener(handler);
		this.addMouseMotionListener(handler);		
	}

	/**
	 * Stop processing all mouse events.
	 */
	protected void disableMouse() {
		for (MouseListener ml : this.getMouseListeners())
			this.removeMouseListener(ml);
		for (MouseMotionListener mml : this.getMouseMotionListeners())
			this.removeMouseMotionListener(mml);
	}
	
	/**
	 * Set the complexity maps to be displayed.
	 * 
	 * @param cxMapStack
	 *            the new set of complexity maps
	 * @param curCxType
	 * 		 	  the current complexity type
	 * @param 
	 */
	public void setMap(ArrayList<ComplexityMap> cxMapStack, CxColour colours,
			ComplexityType curCxType, double cxMax) {
		this.cxMapStack = cxMapStack;
		this.baseWeightRange = (float) cxMapStack.get(0).getWeightRange();
		this.baseWeightMin = (float) cxMapStack.get(0).getWeightMin();
		this.baseLambdaRange = (float) cxMapStack.get(0).getLambdaRange();
		this.baseLambdaMax = (float) cxMapStack.get(0).getLambdaMax();
		this.cxMax = cxMax;
		this.curCxType = curCxType;
		this.colours = colours;
		this.redrawingMaps = true;
		this.repaint();
	}

	/**
	 * Clear the currently selected region.
	 */
	public void resetSelection() {
		this.currentRect = null;
	}

	/**
	 * Clear the currently selected contour.
	 */
	public void resetContour() {
		this.contourStart = null;
		this.contourEnd = null;
	}

	/**
	 * Force a <code>FocusLineageEvent</code> to be fired.
	 */
	public void updateFocus() {
		this.fireFocusLineageEvent();
	}
	
	public void recalculateMap() {
		this.prevSize = this.getSize();
		
		// column width for main map
		baseColumnWidth = (float) getWidth()
				/ cxMapStack.get(0).getColumnCount();
		cxMapRectangles.clear();
		cxMapColWidths.clear();

		// draw any zoomed maps
		for (int i = 0, n = cxMapStack.size(); i < n; i++) {
			ComplexityMap cxMap = cxMapStack.get(i);

			float baseX = (((float) cxMap.getWeightMin() - this.baseWeightMin)
					/ this.baseWeightRange * (float) this.getWidth());
			float baseY = (float) lambdaToY(cxMap.getLambdaMax());
			float curMapWidth = ((float) cxMap.getWeightRange()
					/ this.baseWeightRange * (float) this.getWidth());
			float curMapHeight = ((float) cxMap.getLambdaRange()
					/ this.baseLambdaRange * this.getHeight());

			cxMapRectangles.add(new Rectangle2D.Float(baseX, baseY,
					curMapWidth, curMapHeight));

			cxMapColWidths.add(curMapWidth / cxMap.getColumnCount());
		}
		
		redrawingMaps = false;
	}

	/* Painting routines *******************************************/
	
	public void paintComponent(Graphics g) {
		// draw background
		Graphics2D g2 = (Graphics2D) g;
		g2.setPaint(Color.BLACK);
		g2.fill(new Rectangle2D.Float(0, 0, getWidth(), getHeight()));

		// if size has changed, recalculate coordinates
		if ((this.getSize() != this.prevSize) || redrawingMaps) {
			paramsToNormLoc();
			normLocToFocus();
			recalculateMap();
		}

		for (int i = 0, n = cxMapStack.size(); i < n; i++) 
			// draw maps
			drawMap(g2, cxMapStack.get(i), (float) cxMapRectangles.get(i)
					.getMinX(), cxMapColWidths.get(i));

		if (showBorder) {
			// highlight zoomed regions
			for (int i = 1, n = cxMapRectangles.size(); i < n; i++) 
				drawBorder(g2, cxMapRectangles.get(i));
		}

		if (showSelection && currentRect != null) {
			// draw a selection rectangle on top of the image
			g2.setXORMode(Color.white);
			g2.drawRect(rectToDraw.x, rectToDraw.y, rectToDraw.width - 1,
					rectToDraw.height - 1);
		}

		if (showContour && contourEnd != null) {
			// draw a contour on top of the image
			drawContour(g2);
		}
		
		if (showFocus && currentFocus != null) {
			// draw a cross to indicate current lineage focus
			drawFocus(g2);
		}

	}

	private void drawFocus(Graphics2D g2) {
		g2.setXORMode(Color.white);
		g2.drawLine((int) currentFocus.getX() - 2,
				(int) currentFocus.getY() - 2, (int) currentFocus.getX() + 2,
				(int) currentFocus.getY() + 2);
		g2.drawLine((int) currentFocus.getX() - 2,
				(int) currentFocus.getY() + 2, (int) currentFocus.getX() + 2,
				(int) currentFocus.getY() - 2);
	}

	private void drawContour(Graphics2D g2) {
		g2.setXORMode(Color.white);
		g2.drawRect((int)contourStart.getX()-1, (int)contourStart.getY()-1, 3, 3);
		g2.drawRect((int)contourEnd.getX()-1, (int)contourEnd.getY()-1, 3, 3);
		g2.drawLine((int)contourStart.getX(), (int)contourStart.getY(),
				(int)contourEnd.getX(), (int)contourEnd.getY());
	}

	private void drawMap(Graphics2D g2, ComplexityMap cxMap, float baseX,
			float curColumnWidth) {
		// draw each column in current map
		for (int i = 0, n = cxMap.getColumnCount(); i < n; i++) {
			drawColumn(g2, cxMap.getColumn(i), baseX + (i * curColumnWidth),
					curColumnWidth);
		}
	}

	private void drawColumn(Graphics2D g2, CxColumn col, float curX,
			float curColumnWidth) {
		// draw each block in the current column
		for (int i = 0, n = col.getBlockCount(); i < n; i++) {
			drawBlock(g2, col.getBlock(i), curX, curColumnWidth);
		}
	}

	private void drawBlock(Graphics2D g2, CxBlock block, float curX,
			float curColumnWidth) {
		float curY = (this.baseLambdaMax - (float) block.getLambdaMax())
				/ this.baseLambdaRange * (float) getHeight();
		float blockHeight = (float) block.getLambdaRange()
				/ this.baseLambdaRange * (float) getHeight() + 0.5f;

		g2.setPaint(colours.getColour((float) block.getCx(curCxType)
				/ cxMax));
		g2.fill(new Rectangle2D.Float(curX, curY, curColumnWidth, blockHeight));
	}

	private void drawBorder(Graphics2D g2, Rectangle2D cxMapRectangle) {
		g2.setColor(Color.RED);
		g2.draw(cxMapRectangle);
		g2.setColor(Color.BLACK);
	}

	/* focus / params adjustment routines **************************/
	/* Normalised coordinates provide a safe way of converting parameter pairs
	 * to focus locations the is robust to changes in the size of the map.
	 */
	
	/**
	 * Set the current focus of the map using normalised coordinates.
	 * @param normLoc
	 * 			the new set of normalised coordinates
	 */
	public void setNormLoc(Point2D normLoc) {
		this.currentNormLoc.setLocation(normLoc);
		normLocToFocus();	// update focus
		focusToParams();  // update params
	}
	
	/**
	 * Convert the current focus of the map to normalised coordinates.
	 */
	public void focusToNormLoc() {
		float normX = (float) (currentFocus.getX() / getWidth());
		float normY = (float) (currentFocus.getY() / getHeight());
		this.currentNormLoc.setLocation(normX, normY);
	}

	/**
	 * Convert the current normalised coordinates of the map to a focus location.
	 */
	public void normLocToFocus() {
		float curX = (float) (currentNormLoc.getX() * getWidth());
		float curY = (float) (currentNormLoc.getY() * getHeight());
		this.currentFocus.setLocation(curX, curY);
	}

	/** 
	 * Convert the current normalised coordinates of the map to a parameter pair.
	 */
	public void normLocToParams() {
		double curLambda = yToLambda(currentNormLoc.getY() * getHeight());
		double curWeight = xToWeightScale(currentNormLoc.getX() * getWidth());
		this.currentParams.setLocation(curWeight, curLambda);
	}
	
	/** 
	 * Convert the current parameter pair to a normalised location.
	 */
	public void paramsToNormLoc() {
		float normX = (float) (weightScaleToX(currentParams.getX()) / getWidth());
		float normY = (float) (lambdaToY(currentParams.getY()) / getHeight());
		this.currentNormLoc.setLocation(normX, normY);
	}
	
	private void focusToParams() {
		double curLambda = yToLambda(currentFocus.getY());
		double curWeight = xToWeightScale(currentFocus.getX());
		this.currentParams.setLocation(curWeight, curLambda);
	}

/*	public void paramsToFocus() {
		float curX = (float) (weightScaleToX(currentParams.getX()));
		float curY = (float) (lambdaToY(currentParams.getY()));
		this.currentFocus.setLocation(curX, curY);
	}
*/
	/* Conversion routines *****************************************/

	private double xToWeightScale(double x) {
		return this.baseWeightMin + this.baseWeightRange
				* (x / getWidth());
	}
	
	private double weightScaleToX(double wtScale) {
		return (wtScale - this.baseWeightMin)
				/ this.baseWeightRange * getWidth();
	}
	
	private double yToLambda(double y) {
		return (1.0 - (y / getHeight())) * cxMapStack.get(0).getLambdaRange()
				+ cxMapStack.get(0).getLambdaMin();
	}

	private double lambdaToY(double lambda) {
		return getHeight()
				* (1.0 - ((lambda - cxMapStack.get(0).getLambdaMin()) / 
						cxMapStack.get(0).getLambdaRange()));
	}

	/* Event handling **********************************************/
	
	protected synchronized void fireContourEvent() {
		CxPoint cxContourStart = null;
		if (contourStart != null)
			cxContourStart = new CxPoint(
					xToWeightScale(contourStart.getX()), 
					yToLambda(contourStart.getY()));
		CxPoint cxContourEnd = null;
		if (contourEnd != null)
			cxContourEnd = new CxPoint(
					xToWeightScale(contourEnd.getX()), 
					yToLambda(contourEnd.getY()));
		ProfileEvent contour = new ProfileEvent(this, cxContourStart, cxContourEnd);
		Iterator listenerIt = contourListeners.iterator();
		while (listenerIt.hasNext()) {
			((ProfileListener) listenerIt.next()).profileReceived(contour);
		}
	}
	
	protected synchronized void fireSelectionEvent() {
		SelectionEvent selection = new SelectionEvent(this, 
				this.getSelectedParams());
		Iterator listenerIt = selectionListeners.iterator();
		while (listenerIt.hasNext()) {
			((SelectionListener) listenerIt.next()).selectionReceived(selection);
		}
	}

	private synchronized void fireFocusLineageEvent() {
		// for each map (from top of stack down)
		for (int i = cxMapStack.size() - 1; i >= 0; i--) {
			// check if current focus is within that map's bounds...
			if (checkBounds(cxMapRectangles.get(i))) {
				// ...and redirect focus request accordingly
				int colIndex = (int) (((float) currentFocus.getX() - cxMapRectangles
						.get(i).getMinX()) / cxMapColWidths.get(i));
				CxColumn col = cxMapStack.get(i).getColumn(colIndex);
				double curLambda = currentParams.getY();
				CxBlock block = col.getBlockByLambdaRecursive(curLambda, 0, col.getBlockCount() - 1);
				this.focusToNormLoc();
				FocusLineageEvent focus = new FocusLineageEvent(this, 
						currentNormLoc, currentParams, block.getLin(), block.getCx(curCxType));
				Iterator listenerIt = focusListeners.iterator();
				while (listenerIt.hasNext()) {
					((FocusLineageListener) listenerIt.next()).focusLineageReceived(focus);
				}
				break;
			}
		}
	}

	private synchronized void fireResetEvent() {
		ResetEvent reset = new ResetEvent(this);
		Iterator listenerIt = resetListeners.iterator();
		while (listenerIt.hasNext()) {
			((ResetListener) listenerIt.next()).resetReceived(reset);
		}
	}
	
	/**
	 * Check whether the <code>currentFocus</code> of the map is located within the 
	 * rectangle <code>rect</code>.
	 * @param rect 
	 * 				the current bounds to check
	 */
	private boolean checkBounds(Rectangle2D rect) {
		return ((currentFocus.getX() > rect.getMinX())
				&& (currentFocus.getX() < rect.getMaxX())
				&& (currentFocus.getY() > rect.getMinY()) && (currentFocus.getY() < rect
				.getMaxY()));
	}
	
	/**
	 * Get Complexity Map Parameters for currently selected region. Returns
	 * <code>null</code> if no region is currently selected.
	 * 
	 * @return the <code>CxMapParams</code> for the currently selected region;
	 *         <code>null</code> if no such region exists
	 */
	private CxMapParams getSelectedParams() {
		CxMapParams selectedParams = null;
		if (currentRect != null) {
			selectedParams = new CxMapParams();
			selectedParams.colMinIndex = (int) ((currentRect.x + 1) / baseColumnWidth); 
				// add 1 to deal with fractional columns
			selectedParams.colMaxIndex = (int) ((currentRect.x
					+ currentRect.width - 1) / baseColumnWidth); 
						// subtract 1 to deal with fractional columns
			ComplexityMap curCxMap = cxMapStack.get(0);
			selectedParams.weightMin = curCxMap.getColumn(
					selectedParams.colMinIndex).getWeightScale();
			if (selectedParams.colMaxIndex < curCxMap.getColumnCount() - 1)
				selectedParams.weightMax = curCxMap.getColumn(
						selectedParams.colMaxIndex + 1).getWeightScale();
			else
				selectedParams.weightMax = curCxMap.getWeightMax();
			selectedParams.weightStep = curCxMap.getWeightStep();
			selectedParams.lambdaMin = yToLambda(currentRect.y
					+ currentRect.height);
			selectedParams.lambdaMax = yToLambda(currentRect.y);
		}
		return selectedParams;
	}

	/**
	 * Mouse event handler for the heat map.
	 * @author nic
	 */
	private class MouseHandler implements MouseListener, MouseMotionListener {

		public void mouseClicked(MouseEvent e) {
			Point2D prevFocus = currentFocus;
			currentFocus = e.getPoint();
			focusToNormLoc();
			focusToParams();
			fireFocusLineageEvent();

			// cancel current selection
			resetSelection();
			resetContour();
			
			repaint((int)currentFocus.getX()-5, (int)currentFocus.getY()-5, 10, 10);
			repaint((int)prevFocus.getX()-5, (int)prevFocus.getY()-5, 10, 10);			
		}

		public void mousePressed(MouseEvent e) {
			int currentState = e.getModifiersEx() & 
				(MouseEvent.BUTTON1_DOWN_MASK |
						MouseEvent.BUTTON2_DOWN_MASK |
						MouseEvent.BUTTON3_DOWN_MASK);
			if (currentState == MouseEvent.BUTTON1_DOWN_MASK) {
				if (cxMapStack.size() == 1) {
					resetContour();
					currentSelectionExists = false;
					int x = snapX(e.getX(), 0);
					int y = e.getY();
					currentRect = new Rectangle(x, y, 0, 0);
					updateDrawableRect(getWidth(), getHeight(), 
							currentRect.x, currentRect.y,
							currentRect.width, currentRect.height);
					currentRect.setRect(rectToDraw);
					Rectangle totalRepaint = rectToDraw.union(previousRectDrawn);
					repaint(totalRepaint.x, totalRepaint.y, totalRepaint.width,
							totalRepaint.height);
				}
			} else if (currentState == MouseEvent.BUTTON3_DOWN_MASK) {
				resetSelection();
				resetContour();
				currentContourExists = false;
				contourStart = new Point(e.getX(), e.getY());
				updateDrawableRect(getWidth(), getHeight(), e.getX(), e.getY(), 0, 0);
				Rectangle totalRepaint = rectToDraw.union(previousRectDrawn);
				repaint(totalRepaint.x, totalRepaint.y, totalRepaint.width,
						totalRepaint.height);	
			}
		}

		public void mouseReleased(MouseEvent e) {
			if (currentlySelecting) {
				if (cxMapStack.size() == 1) {
					updateRectSize(e);
					if (currentSelectionExists)
						fireSelectionEvent();
				}
				currentlySelecting = false;
			} else if (currentlyContouring) {
				updateContourSize(e);
				if (currentContourExists)
					fireContourEvent();
				currentlyContouring = false;
			}
			fireResetEvent();
		}

		public void mouseEntered(MouseEvent e) {
			setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
		}

		public void mouseExited(MouseEvent e) {
			setCursor(Cursor.getDefaultCursor());
		}

		public void mouseDragged(MouseEvent e) {
			int currentState = e.getModifiersEx() & 
				(MouseEvent.BUTTON1_DOWN_MASK |
						MouseEvent.BUTTON2_DOWN_MASK |
						MouseEvent.BUTTON3_DOWN_MASK);
			if (currentState == MouseEvent.BUTTON1_DOWN_MASK) {
				if (cxMapStack.size() == 1) {
					updateRectSize(e);
					currentSelectionExists = true;
					currentlySelecting = true;
				}
			} else if (currentState == MouseEvent.BUTTON3_DOWN_MASK) {
				updateContourSize(e);
				currentContourExists = true;
				currentlyContouring = true;
			}
		}

		public void mouseMoved(MouseEvent e) {
		}

		int snapX(int x, int offset) {
			int curColumn = (int) (x / baseColumnWidth) + offset;
			int newX = (int) (curColumn * baseColumnWidth);
			return newX;
		}

		void updateContourSize(MouseEvent e) {
			contourEnd = new Point(e.getX(), e.getY());
			updateDrawableRect(getWidth(), getHeight(),
					(int)contourStart.getX(), (int)contourStart.getY(),
					e.getX() - (int)contourStart.getX(), 
					e.getY() - (int)contourStart.getY());
			Rectangle totalRepaint = rectToDraw.union(previousRectDrawn);
			repaint(totalRepaint.x-1, totalRepaint.y-1, totalRepaint.width+2,
					totalRepaint.height+2);			
			// NOTE: -1, +2 etc. deals with rounding errors
		}
		
		void updateRectSize(MouseEvent e) {
			int x = snapX(e.getX(), 1);
			int y = e.getY();
			currentRect.setSize(x - currentRect.x, y - currentRect.y);
			updateDrawableRect(getWidth(), getHeight(), 
					currentRect.x, currentRect.y,
					currentRect.width, currentRect.height);
			currentRect.setRect(rectToDraw);
			Rectangle totalRepaint = rectToDraw.union(previousRectDrawn);
			repaint(totalRepaint.x, totalRepaint.y, totalRepaint.width,
					totalRepaint.height);
		}

		private void updateDrawableRect(int compWidth, int compHeight, int x, int y, int width, int height) {
			// Make the width and height positive, if necessary.
			if (width < 0) {
				width = 0 - width;
				x = x - width + 1;
				if (x < 0) {
					width += x;
					x = 0;
				}
			}
			if (height < 0) {
				height = 0 - height;
				y = y - height + 1;
				if (y < 0) {
					height += y;
					y = 0;
				}
			}

			// The rectangle shouldn't extend past the drawing area.
			if ((x + width) > compWidth) {
				width = compWidth - x;
			}
			if ((y + height) > compHeight) {
				height = compHeight - y;
			}

			// Update rectToDraw after saving old value.
			if (rectToDraw != null) {
				previousRectDrawn.setBounds(rectToDraw.x-2, rectToDraw.y-2,
						rectToDraw.width+5, rectToDraw.height+5);
				rectToDraw.setBounds(x, y, width, height);
			} else {
				rectToDraw = new Rectangle(x, y, width, height);
			}
		}
		
		private boolean currentSelectionExists = false;
		private boolean currentContourExists = false;
		private boolean currentlySelecting = false;
		private boolean currentlyContouring = false;

	}

	private static final long serialVersionUID = 6883655743902938703L;

	private ArrayList<ComplexityMap> cxMapStack; // the current complexity map stack
	private Dimension prevSize;
	
	private ArrayList<Rectangle2D> cxMapRectangles = new ArrayList<Rectangle2D>();
	private ArrayList<Float> cxMapColWidths = new ArrayList<Float>();

	// base (ie largest) map parameters
	private float baseLambdaRange;
	private float baseLambdaMax;
	private float baseWeightRange;
	private float baseWeightMin;
	private float baseColumnWidth;

	// global maximum complexity
	private double cxMax = 0.0;
	private ComplexityType curCxType = ComplexityType.NONDET_WT;

	// selected point and range
	private Point2D currentFocus = new Point2D.Float();
	private Point2D currentNormLoc = new Point2D.Float(); // location normalised to ([0,1],[0,1]) 
	private Point2D currentParams = new Point2D.Double();
	
	private Point2D contourStart = null;
	private Point2D contourEnd = null;
	
	private Rectangle currentRect = null;
	private Rectangle rectToDraw = null;
	private Rectangle previousRectDrawn = new Rectangle();

	protected ArrayList<ProfileListener> contourListeners = new ArrayList<ProfileListener>();
	protected ArrayList<SelectionListener> selectionListeners = new ArrayList<SelectionListener>();
	protected ArrayList<FocusLineageListener> focusListeners = new ArrayList<FocusLineageListener>();
	protected ArrayList<ResetListener> resetListeners = new ArrayList<ResetListener>();
	
	private boolean redrawingMaps = true;
	protected boolean showBorder = true;
	protected boolean showFocus = true;
	protected boolean showContour = true;
	protected boolean showSelection = true;
	
	private CxColour colours;
	
}
