package linMap.cellLineage;

import java.util.*;

import linMap.geneNetwork.Network;

/**
 * A <code>Cell</code> object represents a single node in a cell lineage.
 * @author nic
 */
public class Cell implements Cloneable {

	private List<Integer> id;
	
	private double[] input;
	private double[] hidden;
	private double[] output;

	private Fate fate;
	
	protected Cell leftChild;
	protected Cell rightChild;
	
	protected boolean terminal;		// true if cell is terminal
	protected boolean differentiated; // true if cell has differentiated
	protected boolean updating;		// true if cell needs to be updated
	protected boolean dividing;		// true if cell needs to divide
	
	public static final double OFF = 0.0;
	public static final double ON = 1.0;
	
	public static final int ZYGOTE = -1;
	public static final int LEFT = 0;
	public static final int RIGHT = 1;
	
	public static final int DIVISION_OUTPUT = 0;
	public static final int FIRST_DIFF = 1;
	
	public Cell() {
		super();
	}
	
	/**
	 * Basic cell constructor
	 * @param id
	 * 			the new cell ID
	 * @param network
	 * 			the new cell control network
	 */		
	public Cell(List<Integer> id, Network network) {
		this.id = id;
		this.input = new double[network.getInputSize()];
		this.hidden = new double[network.getLayerSize(1)];
		for (int i=0; i<this.hidden.length; i++) {
			this.hidden[i] = 0.0;
		}
		this.output = new double[network.getOutputSize()];
		this.fate = null;
	}

	/**
	 * Zygote constructor
	 * @param network
	 * 			the new cell control network
	 */
	public Cell(Network network) {
		this.id = new ArrayList<Integer>();
		this.id.add(ZYGOTE);
		this.input = new double[network.getInputSize()];
		this.hidden = new double[network.getLayerSize(1)];
		for (int i=0; i<this.hidden.length; i++) {
			this.hidden[i] = 0.0;
		}
		this.output = new double[network.getOutputSize()];
		this.fate = null;
	}
	
	/**
	 * Construct a cell from a parent cell
	 * @param parent
	 * 			the parent cell
	 * @param direction
	 * 			the division direction of the new cell
	 */
	public Cell(Cell parent, int direction) {
		this.id = parent.getId();
		this.id.add(direction);
		this.input = parent.input.clone();
		this.hidden = parent.hidden.clone();
		this.output = parent.output.clone();
		this.fate = null;
	}

	@Override
	public String toString() {
		return getClass().getName() 
			+ "[id=" + id
			+ ",terminal=" + terminal
			+ ",differentiated=" + differentiated
			+ ",updating=" + updating
			+ ",dividing=" + dividing
			+ ",input=" + Arrays.toString(input)
			+ ",hidden=" + Arrays.toString(hidden)
			+ ",output=" + Arrays.toString(output)
			+ ",fate=" + fate
			+ "]";
	}

	/**
	 * Two <code>Cells</code> are considered equal if they share the same ID 
	 * and the same fate.
	 */
	@Override
	public boolean equals(Object otherObject) {
		if (this == otherObject) return true;
		if (otherObject == null) return false;
		if (this.getClass() != otherObject.getClass()) return false;
		Cell other = (Cell)otherObject;
		return this.id.equals(other.id) && this.getFate().equals(other.getFate());
	}
	
	@Override 
	public Object clone() {
		Cell o = null;
		try {
			o = (Cell)super.clone();
		} catch(CloneNotSupportedException e) {
			e.printStackTrace();
		}
		o.id = new ArrayList<Integer>(this.id);		// NECESSARY ???
		o.input = this.input.clone();
		o.hidden = this.hidden.clone();
		o.output = this.output.clone();
		return o;
	}
	
	/**
	 * Cell ID accessor.
	 * @return a copy of the cell's ID
	 */
	public List<Integer> getId() {
		return new ArrayList<Integer>(this.id);
	}
	
	/**
	 * Set cell input. *NOTE* This method should probably be deprecated.
	 * @param input
	 * 			the new cellular input
	 */
	protected void setInput(double[] input) {
		this.input = input;
	}
	
	/**
	 * Set cell input automatically.  Currently implements 
	 * Relative Positional Information (RPI) only.
	 */
	protected void setInput() {
		for (int i=0; i<input.length; i++) {
			input[i] = OFF;
		}
		if (id.size() > 1) {
			input[id.get(id.size()-1)] = ON;
		}
	}
	
	/**
	 * Cell output accessor.
	 * @return a copy of the cell's current output
	 */
	protected double[] getOutput() {
		return (double[])this.output.clone();
	}
	
	public Cell getLeft() {
		return leftChild;
	}
	
	public Cell getRight() {
		return rightChild;
	}
	
	public boolean getTerm() {
		return terminal;
	}
	
	/**
	 * Activate the cell's control network.  
	 */
	protected void activate(Network network) {
		if (!updating) {
			List<double[]> state = new ArrayList<double[]>();
			state.add(input);
			state.add(hidden);
			System.arraycopy(network.activate(state), 0, output, 0, output.length);
		} else {
			System.out.println("ERROR: " + getClass().getName() + " -- " +
					"attempting to activate a non-updated cell.");
		}
		updating = true;
	}
	
	/**
	 * Update the cell's current output if updating flag is set.  
	 * Set the dividing status of the cell.  Checks the cell's current
	 * division output against the current division threshold, and the cell's
	 * current depth against the maximum lineage depth.
	 * @param maxDepth
	 * 			the maximum lineage depth
	 * @param divThresh
	 * 			the current division threshold
	 * @return the dividing flag
	 */
	protected boolean update(int maxDepth, double divThresh) {
		updating = false;
		checkDivision(divThresh);
		checkDepth(maxDepth);
		return this.dividing;
	}
	
	private void checkDivision(double divThresh) {
		if (this.output[DIVISION_OUTPUT] >= divThresh) {
			this.dividing = true;
		} else {
			this.differentiated = true;
			this.terminal = true;
		}
	}
	
	private void checkDepth(double maxDepth) {
		if (this.id.size() > maxDepth) {
			this.dividing = false;
			this.terminal = true;
		}		
	}
	
	public Fate getFate() {
		if (fate == null) {
			buildFate();
		}
		return this.fate;
	}
	
	private void buildFate() {
		fate = new Fate();
		if (this.terminal) {
			fate.setTerminal();
			if (this.differentiated) {
				int x = getOneHotIndex();
				fate.add(x);
//				System.out.println("adding diff fate " + x);
			} else {
//				System.out.println("adding non-diff fate");
				fate.add(Fate.NON_DIFF);
			}
		} else {
			fate.add(leftChild.getFate());
			fate.add(rightChild.getFate());
			fate.setNonTerminal();
		}
	}
	
	// move me to a util package
	private int getOneHotIndex() {
		double max = 0.0;
		int maxI = 0;
		for (int i=FIRST_DIFF; i<output.length; i++) {
			if (output[i] > max) {
				max = output[i];
				maxI = i-FIRST_DIFF;
			}
		}
		return maxI;
	}

	public void print() {
		System.out.println(id + " : " + fate);
	}
	
	public void printState() {
		System.out.println(id);
		System.out.println(Arrays.toString(input));
		System.out.println(Arrays.toString(hidden));
		System.out.println(Arrays.toString(output));
	}
}
