package linMap.cellLineage;

import java.util.*;

import linMap.geneNetwork.*;

/**
 * A <code>Lineage</code> object represents a cell lineage.
 * @author nic
 */
public class Lineage {

	private Cell zygote;
	private CellSeq terminal;
	private LineageParams params;
	private Network network;
	
	public Lineage() {
		super();
	}

	public Lineage(LineageParams linParams, Network network) {
		this.params = linParams;
		this.network = network;
		this.zygote = new Cell(this.network);
		this.terminal = new CellSeq(this.zygote);
	}
	
	public Lineage(LineageParams linParams, NetworkParams netParams) {
		this.params = linParams;
		this.network = NetTools.createSimpleRecurrentNet(netParams);
		this.zygote = new Cell(this.network);
		this.terminal = new CellSeq(this.zygote);
	}

	@Override
	public String toString() {
		return getClass().getName() 
			+ "[params=" + params
			+ ",terminal=" + terminal 
			+ "]";
	}
	
	@Override
	public boolean equals(Object otherObject) {
		if (this == otherObject) return true;
		if (otherObject == null) return false;
		if (this.getClass() != otherObject.getClass()) return false;
		Lineage other = (Lineage)otherObject;
		return testEquality(zygote, other.zygote);
	}
	
	private boolean testEquality(Cell cell, Cell other) {
		if (cell == null && other == null) {
			return true;
		}
		boolean result = true;
		if (cell != null && other != null && cell.equals(other)) {
				result = testEquality(cell.leftChild, other.leftChild) &&
					testEquality(cell.rightChild, other.rightChild);
		} else {
			result = false;
		}
		return result;
	}
	
	public Cell getZygote() {
		return this.zygote;
	}
	
	public int getDepth() {
		return terminal.getDepth();
	}
	
	public int getMaxDepth() {
		return params.maxDepth;
	}
	
	public int getTerminalCount() {
		return terminal.getSize();
	}
	
	public int getDiffTerminalCount() {
		return terminal.getDiffCellCount();
	}
	
	public int getCellTypeCount() {
		int count = zygote.getFate().getDiff().size();
		if (zygote.getFate().getDiff().contains(Fate.NON_DIFF))
			count-=1;
		return count;
	}
	
	/**
	 * Generate the cell lineage.
	 */
	public void generate() {
		boolean dividing = false;
		for (int depth=0; depth<params.maxDepth+1; depth++) {
			if (dividing) {
				terminal.divide();
				dividing = false;
			}
			if (terminal.activate()) {
				dividing = terminal.update(calcDivThresh(depth));
			} else {
				break;
			}
		}
	}
	
	/**
	 * Calculate the current division threshold.
	 * @param depth
	 * 			the depth of the cell in the lineage
	 * @return the current division threshold
	 */
	private double calcDivThresh(int depth) {
		return 0.01 * Math.exp(params.divDecay * depth);
	}

	public void print() {
		print(zygote);
	}
	
	public void print(Cell cell) {
		cell.print();
		if (cell.getLeft() != null)
			print(cell.getLeft());
		if (cell.getRight() != null)
			print(cell.getRight());
	}

	public void printState() {
		printState(zygote);
	}
	
	public void printState(Cell cell) {
		cell.printState();
		if (cell.getLeft() != null)
			printState(cell.getLeft());
		if (cell.getRight() != null)
			printState(cell.getRight());
	}
	
	/**
	 * A <code>CellSeq</code> object represents a layer of (currently) 
	 * terminal cells.  These cells may exist at different depths in the lineage,
	 * but they all share the property of (currently) having no children.
	 * @author nic
	 */
	private class CellSeq {
		private List<Cell> cells;
		
		/**
		 * Constructor for initial cell sequence
		 * @param zygote
		 */
		public CellSeq(Cell zygote) {
			cells = new ArrayList<Cell>();
			cells.add(zygote);
		}

		public int getSize() {
			return cells.size();
		}

		public int getDiffCellCount() {
			int count = 0;
			for (Cell cell : cells) {
				if (cell.differentiated)
					count++;
			}
			return count;
		}
		
		public String toString() {
			return getClass().getName() 
				+ "[cells=" + cells
				+ "]";
		}
		
		/** 
		 * Activate all non-terminal cells in the sequence.
		 * @return true if at least one cell was activated
		 */
		public boolean activate() {
			boolean updating = false;
			for (Cell cell : cells) {
				if (!cell.terminal) {
					cell.setInput();
					cell.activate(network);
					updating = true;
				}
			}
			return updating;
		}
		
		/**
		 * Update all activated cells in the sequence.
		 * @param divThresh
		 * 			the division threshold at the current depth
		 * @return true if at least one cell is due to divide
		 */
		public boolean update(double divThresh) {
			boolean dividing = false;
			for (Cell cell : cells) {
				if (cell.update(params.maxDepth, divThresh)) {
					dividing = true;
				}
			}
			return dividing;
		}

		/**
		 * Divide any cells in sequence that are due to divide
		 */
		public void divide() {
			for (int i=0; i<cells.size(); i++) {
				Cell cell = cells.get(i);
				if (cell.dividing) {
					cell.leftChild = new Cell(cell, Cell.LEFT);
					cell.rightChild = new Cell(cell, Cell.RIGHT);
					cells.remove(i);
					cells.add(i, cell.leftChild);
					i += 1; // skip newly added cell
					cells.add(i, cell.rightChild);
				}
			}
		}
		
		public int getDepth() {
			int maxDepth = 0;
			for (Cell cell : cells) {
				int curDepth = cell.getId().size()-1;
				if (curDepth > maxDepth) {
					maxDepth = curDepth;
				}
			}
			return maxDepth;
		}
		
		public int getCellTypeCount() {
			HashSet<Integer> curFates = new HashSet<Integer>();
			for (Cell cell : cells) {
				int diff = cell.getFate().getDiff().get(0);
				if (diff != Fate.NON_DIFF) {
					curFates.add(diff);
				}
			}
			return curFates.size();
		}
	}
}
