package linMap.cxMap;

import java.io.*;
import java.util.*;

import linMap.cellLineage.*;
import linMap.cxMap.CxBlock.*;
import linMap.geneNetwork.*;



public class ComplexityMap implements Serializable {
	// stores a single complexity map bounded by:
	// - weightMin, weightMax
	// - lambdaMin, lamdbaMax
	
	private static final long serialVersionUID = 3929568625241639977L;

	private CxMapParams cxMapParams;
	private LineageParams linParams;
	private NetworkParams netParams;

	protected double lambdaRange;
	protected double weightRange;
	
	private Map<ComplexityType, Double> cxMax;
	
	private List<CxColumn> columns;
	
	public ComplexityMap(CxMapParams cxMapParams, 
			LineageParams linParams, NetworkParams netParams) {
		this.cxMapParams = (CxMapParams)cxMapParams.clone();
		this.linParams = linParams;
		this.netParams = netParams;
		this.lambdaRange = cxMapParams.lambdaMax - cxMapParams.lambdaMin;
		this.weightRange = cxMapParams.weightMax - cxMapParams.weightMin;
		this.columns = new ArrayList<CxColumn>();

		double[] wtScales = calcWtScales();

		this.cxMax = new EnumMap<ComplexityType, Double>(ComplexityType.class);
		for (ComplexityType type : ComplexityType.values()) {
			this.cxMax.put(type, 0.0);
		}
		
		for (double wtScale : wtScales) {
			CxColumn col = CxColumn.createRecursiveCxColumn(wtScale, cxMapParams.lambdaMin, cxMapParams.lambdaMax, 
					cxMapParams.lambdaDiv, linParams, netParams);
			columns.add(col);
		}
		recalculateCxMax();
	}
	
	public ComplexityMap(ComplexityMap parentMap, CxMapParams newParams) {
		this.cxMapParams = (CxMapParams)newParams.clone();
		this.linParams = parentMap.linParams;
		this.netParams = parentMap.netParams;
		this.lambdaRange = newParams.lambdaMax - newParams.lambdaMin;
		this.weightRange = newParams.weightMax - newParams.weightMin;
		this.columns = new ArrayList<CxColumn>();
		
		this.cxMax = parentMap.cxMax;
		
		for (int i=newParams.colMinIndex; i<=newParams.colMaxIndex; i++) {
			CxColumn col = new CxColumn(parentMap.getColumn(i), newParams.lambdaMin, newParams.lambdaMax);
			this.columns.add(col);
		}
	}
	
	private void recalculateCxMax() {
		for (CxColumn col : columns) {
			for (ComplexityType type : ComplexityType.values()) {
				if (col.getMaxCx(type) > this.cxMax.get(type)) {
					this.cxMax.put(type, col.getMaxCx(type));
				}								
			}
		}
	}
	
	public void subdivideWeights() {
		cxMapParams.weightStep /= 2;
		ArrayList<CxColumn> newColumns = new ArrayList<CxColumn>();
		// split all columns but last one
		for (int i=0; i<columns.size()-1; i++) {
			System.out.println("splitting column " + i);
			newColumns.add(columns.get(i));
			double newWtScale = (columns.get(i).getWeightScale() + columns.get(i+1).getWeightScale()) / 2.0;
			CxColumn col = CxColumn.createRecursiveCxColumn(newWtScale, cxMapParams.lambdaMin, cxMapParams.lambdaMax,
					columns.get(0).lambdaDiv, linParams, netParams);
			newColumns.add(col);
		}
		// split last column
		newColumns.add(columns.get(columns.size()-1));
		double newWtScale = (columns.get(columns.size()-1).getWeightScale() + this.getWeightMax()) / 2.0;
		CxColumn col = CxColumn.createRecursiveCxColumn(newWtScale, cxMapParams.lambdaMin, cxMapParams.lambdaMax,
				columns.get(0).lambdaDiv, linParams, netParams);	
		newColumns.add(col);

		columns = newColumns;
		recalculateCxMax();
	}

	public void subdivideLambda() {
		cxMapParams.lambdaDiv += 1;
		for (CxColumn col : columns) {
			col.subdivideLambda();
		}		
		recalculateCxMax();
	}
	
	public int getColumnCount() {
		return columns.size();
	}
	
	public CxColumn getColumn(int index) {
		return columns.get(index);
	}
	
	public double getLambdaMin() {
		return cxMapParams.lambdaMin;
	}
	
	public double getLambdaMax() {
		return cxMapParams.lambdaMax;
	}
	
	public double getLambdaRange() {
		return lambdaRange;
	}

	public double getWeightMin() {
		return cxMapParams.weightMin;
	}
	
	public double getWeightMax() {
		return cxMapParams.weightMax;
	}

	public double getWeightRange() {
		return weightRange;
	}
	
	public double getWeightStep() {
		return cxMapParams.weightStep;
	}
	
	public double getCxMax(ComplexityType type) {
		return cxMax.get(type);
	}
	
	public void print() {
		for (CxColumn col : columns) {
			col.print();
			System.out.println();
		}
	}
	
	public void printWtScales() {
		for (CxColumn col : columns) {
			System.out.println(col.getWeightScale());
		}
	}
	
	private double[] calcWtScales() {
		int size = (int)((cxMapParams.weightMax - cxMapParams.weightMin) / cxMapParams.weightStep);
		double[] wtScales = new double[size];
		for (int i=0; i<size; i++) {
			wtScales[i] = cxMapParams.weightMin + (cxMapParams.weightStep * i);
		}
		return wtScales;
	}

}
