package linMap.cxMap;

import java.awt.Color;
import java.io.*;
import java.util.*;

import linMap.cellLineage.*;
import linMap.geneNetwork.*;
import linMap.rulesets.deterministic.*;
import linMap.rulesets.nondeterministic.*;

public class CxBlock implements Serializable {

	private static final long serialVersionUID = -2029657699948270090L;

	protected double lambdaMin;
	protected double lambdaMax;
	protected double lambdaRange;

	private LineageParams linParams;
	private Network net;

	private Map<ComplexityType, Double> complexity;
	
	public enum ComplexityType {
		DET_WT("det wt alg cx", new Color(1f, 0, 0) ), 
		NONDET_WT("nondet wt alg cx", new Color(0, 1f, 0)),
		DET_CX("det alg cx", new Color(0, 1f, 1f)),
		NONDET_CX("nondet alg cx", new Color(1f, 1f, 0)),
		DET_RR("det rr size", new Color(1f, 0, 1f)),
		NONDET_RR("nondet rr size", new Color(0.67f, 0.5f, 1f)),
		TERM_SIZE("term size", new Color(0.67f, 0.9f, 1f)),
		DIFF_SIZE("diff size", new Color(0.6f, 1f, 0.6f)),
		CELL_TYPES("cell types", new Color(1f, 1f, 1f));
				
		private String label;
		private Color colour;
		
		ComplexityType(String label, Color colour) { 
			this.label = label;
			this.colour = colour;
		}
		public String getLabel() { return label; }
		public Color getColour() { return colour; }
	}
	
	/**
	 * <code>CxBlock</code> constructor.
	 * @param lambdaMin the lower lambda bound
	 * @param lambdaMax the upper lambda bound
	 * @param linParams the lineage parameters
	 * @param net the gene network with which to generate the lineage
	 */
	public CxBlock(double lambdaMin, double lambdaMax, LineageParams linParams, Network net) {
		this.lambdaMin = lambdaMin;
		this.lambdaMax = lambdaMax;
		this.lambdaRange = lambdaMax - lambdaMin;
		this.linParams = linParams;
		this.net = net;
		this.complexity = new EnumMap<ComplexityType, Double>(ComplexityType.class);

		linParams.divDecay = lambdaMin;
		Lineage lin = new Lineage(linParams, net);
		lin.generate();
		
		this.complexity.put(ComplexityType.TERM_SIZE, (double)lin.getTerminalCount());
		this.complexity.put(ComplexityType.DIFF_SIZE, (double)lin.getDiffTerminalCount());
		this.complexity.put(ComplexityType.CELL_TYPES, (double)lin.getCellTypeCount());
		
		NonDeterministicRecursiveRuleSet ndrs = new NonDeterministicRecursiveRuleSet(lin);
		this.complexity.put(ComplexityType.NONDET_WT, ndrs.getWeightedComplexity());
		this.complexity.put(ComplexityType.NONDET_RR, ndrs.getRuleSetSize());
		this.complexity.put(ComplexityType.NONDET_CX, ndrs.getRRComplexity());
		DeterministicRecursiveRuleSet drs = new DeterministicRecursiveRuleSet(lin);
		this.complexity.put(ComplexityType.DET_WT, drs.getWeightedComplexity());
		this.complexity.put(ComplexityType.DET_RR, drs.getRuleSetSize());
		this.complexity.put(ComplexityType.DET_CX, drs.getRRComplexity());
	}
	
	static public CxBlock createCxBlock(double lambdaMin, double lambdaMax, LineageParams linParams, Network net) {
		CxBlock block = new CxBlock(lambdaMin, lambdaMax, linParams, net);
		return block;
	}
	
	public CxBlock(CxBlock parentBlock) {
		this.lambdaMin = parentBlock.lambdaMin;
		this.lambdaMax = parentBlock.lambdaMax;
		this.lambdaRange = lambdaMax - lambdaMin;
		this.linParams = parentBlock.linParams;
		this.net = parentBlock.net;
		this.complexity = parentBlock.complexity;
	}
	
	public double getLambdaMin() {
		return lambdaMin;
	}

	public void setLambdaMin(double lambdaMin) {
		this.lambdaMin = lambdaMin;
		this.lambdaRange = lambdaMax - lambdaMin;
	}
		
	public double getLambdaMax() {
		return lambdaMax;
	}

	public void setLambdaMax(double lambdaMax) {
		this.lambdaMax = lambdaMax;
		this.lambdaRange = lambdaMax - lambdaMin;
	}
		
	public double getLambdaRange() {
		return lambdaRange;
	}
	
	public double getCx(ComplexityType type) {
		return complexity.get(type);
	}
		
	public Lineage getLin() {
		linParams.divDecay = lambdaMin;
		Lineage lin = new Lineage(linParams, net);
		lin.generate();
		return lin;
	}
	
	public void print() {
		System.out.println("lambda:" + lambdaMin + " -- " + lambdaMax + " (" + lambdaRange + ")");
		System.out.print("complexity: ");
		for (ComplexityType type : ComplexityType.values()) {
			System.out.print(complexity.get(type) + " ");
		}
		System.out.println();
	}
}
