package linMap.cxMap;

import java.util.*;
import java.io.*;

import linMap.cellLineage.*;
import linMap.cxMap.CxBlock.*;
import linMap.geneNetwork.*;


public class CxColumn implements Serializable {
	// stores a single complexity column for a given weight value, bounded by
	// lambdaMin and lambdaMax, with lambdaDiv recursive subdivisions

	private static final long serialVersionUID = 3808201836659617138L;

	private double weightScale;
	private double lambdaMin;
	private double lambdaMax;
	
	int lambdaDiv;
	double lambdaStep;
	
	private Network net;
	private LineageParams linParams;
	
	private List<CxBlock> blocks;
	
	private CxColumn(double weightScale, 
			double lambdaMin, double lambdaMax, 
			LineageParams linParams, NetworkParams netParams) {
		this.weightScale = weightScale;
		this.lambdaMin = lambdaMin;
		this.lambdaMax = lambdaMax;
		netParams.weightScale = weightScale;
		this.net = NetTools.createSimpleRecurrentNet(netParams);
		this.linParams = linParams;
		blocks = new ArrayList<CxBlock>();


	}

	public CxColumn(CxColumn parentCol) {
		this.weightScale = parentCol.weightScale;
		this.lambdaMin = parentCol.lambdaMin;
		this.lambdaMax = parentCol.lambdaMax;
		this.lambdaDiv = parentCol.lambdaDiv;
		this.lambdaStep = parentCol.lambdaStep;
		this.net = parentCol.net;
		this.linParams = parentCol.linParams;
		this.blocks = new ArrayList<CxBlock>();
		
		for (CxBlock block : parentCol.blocks) 
			this.blocks.add(new CxBlock(block));
	}
	
	public CxColumn(CxColumn parentCol, double lambdaMin, double lambdaMax) {
		this.weightScale = parentCol.weightScale;
		this.lambdaMin = lambdaMin;
		this.lambdaMax = lambdaMax;
		this.lambdaDiv = parentCol.lambdaDiv;
		this.lambdaStep = parentCol.lambdaStep;
		this.net = parentCol.net;
		this.linParams = parentCol.linParams;
		blocks = new ArrayList<CxBlock>();
		
		CxBlock firstBlock = parentCol.getBlockByLambda(this.lambdaMin);
		int firstBlockIndex = parentCol.blocks.indexOf(firstBlock);
		
		CxBlock lastBlock = parentCol.getBlockByLambda(this.lambdaMax);
		int lastBlockIndex = parentCol.blocks.indexOf(lastBlock);

//		System.out.println("blocks: " + firstBlockIndex + " -- " + lastBlockIndex);
		
		for (int i=firstBlockIndex; i<=lastBlockIndex; i++) {
			blocks.add(new CxBlock(parentCol.getBlock(i)));
		}
//		blocks.get(0).setLambdaMin(lambdaMin);
//		blocks.get(blocks.size()-1).setLambdaMax(lambdaMax);
	}

	public static CxColumn createRecursiveCxColumn(double weightScale, 
			double lambdaMin, double lambdaMax, int lambdaDiv, 
			LineageParams linParams, NetworkParams netParams) {
		CxColumn cxCol = new CxColumn(weightScale, lambdaMin, lambdaMax, linParams, netParams);
		cxCol.lambdaDiv = lambdaDiv;
		cxCol.lambdaStep = 0.0;
		
		cxCol.blocks.add(CxBlock.createCxBlock(cxCol.lambdaMin, cxCol.lambdaMax, cxCol.linParams, cxCol.net));
		cxCol.blocks.add(CxBlock.createCxBlock(cxCol.lambdaMax, cxCol.lambdaMax, cxCol.linParams, cxCol.net));
		cxCol.subdivideBlocks(0, cxCol.lambdaDiv);
		
		return cxCol;
	}
	
	public static CxColumn createRegularCxColumn(double weightScale, 
			double lambdaMin, double lambdaMax, double lambdaStep, 
			LineageParams linParams, NetworkParams netParams) {
		CxColumn cxCol = new CxColumn(weightScale, lambdaMin, lambdaMax, linParams, netParams);
		cxCol.lambdaStep = lambdaStep;
		cxCol.lambdaDiv = 0;
		
		double[] lambdaValues = cxCol.calcLambdaValues(lambdaStep);
		for (int i=0, n=lambdaValues.length-1; i<n; i++) {
			cxCol.blocks.add(CxBlock.createCxBlock(lambdaValues[i], lambdaValues[i+1], cxCol.linParams, cxCol.net));
		}
		
		return cxCol;
	}
	
	public double getWeightScale() {
		return weightScale;
	}
	
	public int getBlockCount() {
		return blocks.size();
	}
	
	public CxBlock getBlock(int index) {
		return blocks.get(index);
	}

	/**
	 * Do a linear search to locate the <code>CxBlock</code> containing the current lambda value.
	 * @param curLambda the current lambda value
	 * @return the <code>CxBlock<code> containing the current lambda value
	 */
	public CxBlock getBlockByLambda(double curLambda) {
		CxBlock result = null;
		for (CxBlock block : blocks) {
			if (block.lambdaMax >= curLambda) {
				result = block;
				break;
			}
		}
		return result;
	}
	
	/**
	 * Do a recursive binary search to locate the <code>CxBlock</code> containing the current lambda value.
	 * @param curLambda the current lambda value
	 * @return the <code>CxBlock<code> containing the current lambda value
	 */
	public CxBlock getBlockByLambdaRecursive(double curLambda, int low, int high) {
		int curIndex = (low + high) / 2;
		if (curLambda > blocks.get(curIndex).getLambdaMax()) 
			return getBlockByLambdaRecursive(curLambda, curIndex+1, high);
		else if (curLambda < blocks.get(curIndex).getLambdaMin())
			return getBlockByLambdaRecursive(curLambda, low, curIndex-1);
		else
			return blocks.get(curIndex);
	}
	
	/**
	 * Calculate maximum complexity in <code>CxColumn</code>.
	 * @return maximum complexity
	 */
	protected double getMaxCx(ComplexityType type) {
		double currentMax = 0.0;
		for (CxBlock block : blocks) {
			if (block.getCx(type) > currentMax) 
				currentMax = block.getCx(type);
		}
		return currentMax;
	}
	
	protected void subdivideLambda() {
		lambdaDiv += 1;
		for (int i=0; i<blocks.size()-1; i++) {
			double newLambda = (blocks.get(i).getLambdaMin() + blocks.get(i+1).getLambdaMin()) / 2.0;
			blocks.get(i).setLambdaMax(newLambda);
			blocks.add(i+1, CxBlock.createCxBlock(newLambda, blocks.get(i+1).getLambdaMin(), this.linParams, this.net));
			i++;
		}
	}
	
	public void print() {
		System.out.println("WeightScale: " + weightScale);
		for (int i=0, n=blocks.size(); i<n; i++) {
			System.out.println("Block " + i);
			blocks.get(i).print();
		}
	}

	private double[] calcLambdaValues(double lambdaStep) {
		int size = (int)((lambdaMax - lambdaMin) / lambdaStep) + 1;
		double[] lambdaValues = new double[size];
		for (int i=0; i<size; i++) {
			lambdaValues[i] = lambdaMin + (lambdaStep * i);
		}
		return lambdaValues;
	}
	
	/*
	private CxBlock createCxBlock(double lambdaMin, double lambdaMax) {
		CxBlock block = new CxBlock(lambdaMin, lambdaMax, linParams, net);
		return block;
	}
	*/
	
	private int subdivideBlocks(int index, int divisionsRemaining) {
//		System.out.println("subdivide blocks : index=" + index + " div=" + divisionsRemaining);
		// return if max bifurcation depth reached
		if (divisionsRemaining == 0)  
			return index;

		// create a new intermediate lineage
		double curMin = blocks.get(index).getLambdaMin();
		double curMax = blocks.get(index+1).getLambdaMin();
		double lambdaMid = (curMin + curMax) / 2.0;
		
//		System.out.println(" inserting to " + (index+1));
		blocks.add(index+1, CxBlock.createCxBlock(lambdaMid, curMax, this.linParams, this.net));
		blocks.get(index).setLambdaMax(lambdaMid);	// update previous block's upper bound

		// if that lineage differs from the lower bound, subdivide again
//		System.out.println(" checking between " + index + " and " + (index+1));
		if (!blocks.get(index).getLin().equals(blocks.get(index+1).getLin())) {
//			System.out.println(" dividing left");
			index = subdivideBlocks(index, divisionsRemaining-1);
		}
		index += 1;
//		System.out.println(" checking between " + (index) + " and " + (index+1));
		if (!blocks.get(index).getLin().equals(blocks.get(index+1).getLin())) {
//			System.out.println(" dividing right");
			index = subdivideBlocks(index, divisionsRemaining-1);
		}
		return index;
	}


	/**
	 * Remove duplicate lineages from the <code>CxColumn</code>.  NOT IMPLEMENTED
	 */
	protected void prune() {
		for (int i=0, n=blocks.size(); i<n; i++) {
			if (blocks.get(i).getLin() == blocks.get(i+1).getLin()) {
				// TODO: how to implement?
				// if not storing entire lineages to compare, store a hashcode representation?
			}
		}
	}
	

}
