package linMap.geneNetwork;

import java.util.*;
import java.io.*;

class WeightMatrix implements Cloneable, Serializable {

	private static final long serialVersionUID = -2862677461367330569L;
	
	private int targetSize;
	private int sourceSize;
	
	protected double[][] weights;
	
	/**
	 * Default weight matrix constructor.
	 */
	public WeightMatrix() {
		super();
	}
	
	/**
	 * Basic weight matrix constructor
	 * @param targetSize the number of nodes in the target layer
	 * @param sourceLayer the number of nodes in the source layer
	 */
	public WeightMatrix(int targetSize, int sourceSize) {
		super();
		this.targetSize = targetSize;
		this.sourceSize = sourceSize;
		this.weights = new double[this.targetSize][this.sourceSize];
	}

	@Override
	public boolean equals(Object otherObject) {
		if (this == otherObject) return true;
		if (otherObject == null) return false;
		if (this.getClass() != otherObject.getClass()) return false;
		WeightMatrix other = (WeightMatrix)otherObject;
		return this.targetSize == other.targetSize 
		    && this.sourceSize == other.sourceSize
			&& Arrays.equals(this.weights, other.weights);
	}
	
	@Override
	public String toString() {
		return getClass().getName()
			+ "[targetSize=" + this.targetSize
			+ ",sourceSize=" + this.sourceSize
			+ ",weights=" + Arrays.deepToString(this.weights)
			+ "]";
	}
	
	@Override
	public Object clone() {
		WeightMatrix o = null;
		try {
			o = (WeightMatrix)super.clone();
		} catch(CloneNotSupportedException e) {
			e.printStackTrace();
		}
		o.weights = new double[targetSize][];
		for (int i=0; i<o.weights.length; i++) {
			o.weights[i] = this.weights[i].clone();
		}
		return o;
	}
	
	/**
	 * Source size accessor.
	 * @return number of columns in weight matrix
	 */
	public int getSourceSize() {
		return this.sourceSize;
	}
	
	/** 
	 * Target size accessor.
	 * @return number of rows in weight matrix
	 */
	public int getTargetSize() {
		return this.targetSize;
	}
	

	/**
	 * Get number of mutable (i.e., non-fixed) weights in a weight matrix.
	 * @return the number of mutable weights
	 */
	protected int getWeightCount() {
		int count = 0;
		for (int tIndex=0; tIndex<this.targetSize; tIndex++) {
			for (int sIndex=0; sIndex<this.sourceSize; sIndex++) {
				count++;
			}
		}
		return count;
	}
	
	/**
	 * Randomise all non-fixed weights to normally distributed values with 
	 * mean 0.0 and standard deviation <code>weightRange</code>.
	 * @param generator the <code>Random</code> used to generate
	 * @param weightRange the standard deviation of the random values
	 */
	protected void randomise(Random generator, double weightRange) {
		for (int tIndex=0; tIndex < this.targetSize; tIndex++) {
			for (int sIndex=0; sIndex < this.sourceSize; sIndex++) {
				this.weights[tIndex][sIndex] = 
					generator.nextGaussian() * weightRange;
			}
		}
	}

}
