package linMap.rulesets.deterministic;

import java.util.*;

import linMap.cellLineage.*;


/**
 * A <code>DeterministicRecursiveRuleSet</code> object represents a single cell production algorithm.  That is,
 * a rule of the form <code>A -> B, C</code>.
 * @author nic
 */
public class DeterministicRecursiveRuleSet {

	/**
	 * Create a new <code>DeterministicRecursiveRuleSet</code> based on the passed <code>Lineage</code>.
	 * @param lin the cell lineage
	 */
	public DeterministicRecursiveRuleSet(Lineage lin) {
		this.lin = lin;
		this.rules = new HashMap<RHS, Rule>();
		this.terms = new HashMap<Integer, Integer>();
		this.cells = new HashMap<List<Integer>, Integer>();
		this.nextRuleId = 0;
		addRule(this.lin.getZygote());
		this.divisionCount = lin.getTerminalCount() - 1;
	}

	/**
	 * Write the <code>DeterministicRecursiveRuleSet</code> to the console.
	 */
	public void print() {
		for (Rule rule : rules.values()) {
			rule.print();
		}
	}
	
	/**
	 * Calculate the nondeterministic algorithmic complexity of the 
	 * <code>DeterministicRecursiveRuleSet</code>.  Nondeterministic algorithmic complexity
	 * is as the number of unique rules divided by the number of divisions.
	 * @return the nondeterministic algorithimc complexity
	 */
	public double getRRComplexity() {
		if (divisionCount > 0)
			return (double)rules.size() / divisionCount;
		else
			return 0.0;
	}

	public double getWeightedComplexity() {
		return getRRComplexity() * lin.getDiffTerminalCount();
	}
	
	public double getRuleSetSize() {
		return (double)rules.size();
	}
	
	// -- private ---------------------------------------------------
	
	private Lineage lin;
	// Map from RHS to Rule
	private Map<RHS, Rule> rules;
	// Map from Fate to ID (LHS)
	private Map<Integer, Integer> terms;
	
	private Map<List<Integer>, Integer> cells;
	
	private int nextRuleId;
	private int divisionCount;
		
	/**
	 * Recursively add <code>Rules</code> to a <code>DeterministicRecursiveRuleSet</code>.
	 * @param cell the current cell
	 */
	private void addRule(Cell cell) {
		if (cell.getTerm()) {
			createTerm(cell);
		} else {
			addRule(cell.getLeft());
			addRule(cell.getRight());
			createRule(cell);
		}
	}
	
	/**
	 * Create a new <code>Rule</code> and add it to the <code>DeterministicRecursiveRuleSet</code> if
	 * it doesn't already exist.
	 * @param cell the current cell
	 */
	private void createRule(Cell cell) {
		RHS rhs = new RHS(cells.get(cell.getLeft().getId()), cells.get(cell.getRight().getId()));
		if (rules.containsKey(rhs))
			rules.get(rhs).incCount();
		else {
			Rule rule = new Rule();
			rule.lhs = nextRuleId;
			nextRuleId++;
			rule.rhs = rhs;
			rules.put(rule.rhs, rule);
		}
		cells.put(cell.getId(), rules.get(rhs).lhs);
	}
	
	/** 
	 * Create a new terminal differentiation value and add it to the set of diff values.
	 * @param cell the current cell
	 */
	private void createTerm(Cell cell) {
		int curTerm = cell.getFate().diff;
		if (!terms.containsKey(curTerm)) {
			terms.put(curTerm, nextRuleId);
			nextRuleId++;
		}
		ArrayList<Integer> curId = (ArrayList<Integer>)cell.getId();
		int curTermId = terms.get(curTerm);
		cells.put(curId, curTermId);
	}
}

