package linMap.rulesets.nondeterministic;

import java.util.*;

import linMap.cellLineage.*;


/**
 * A <code>DeterministicRecursiveRuleSet</code> object represents a single cell production algorithm.  That is,
 * a rule of the form <code>A -> B, C</code>.
 * @author nic
 */
public class NonDeterministicRecursiveRuleSet {

	/**
	 * Create a new <code>DeterministicRecursiveRuleSet</code> based on the passed <code>Lineage</code>.
	 * @param lin the cell lineage
	 */
	public NonDeterministicRecursiveRuleSet(Lineage lin) {
		this.lin = lin;
		this.rules = new HashMap<RHS, Rule>();
		addRule(this.lin.getZygote());
		this.divisionCount = lin.getTerminalCount() - 1;
	}

	/**
	 * Write the <code>DeterministicRecursiveRuleSet</code> to the console.
	 */
	public void print() {
		for (Rule rule : rules.values()) {
			rule.print();
		}
	}
	
	/**
	 * Calculate the nondeterministic algorithmic complexity of the 
	 * <code>DeterministicRecursiveRuleSet</code>.  Nondeterministic algorithmic complexity
	 * is as the number of unique rules divided by the number of divisions.
	 * @return the nondeterministic algorithimc complexity
	 */
	public double getRRComplexity() {
		if (divisionCount > 0)
			return (double)rules.size() / divisionCount;
		else
			return 0.0;
	}

	public double getWeightedComplexity() {
		return getRRComplexity() * lin.getDiffTerminalCount();
	}

	public double getRuleSetSize() {
		return (double)rules.size();
	}
		
	// -- private ---------------------------------------------------
	
	private Lineage lin;
	private Map<RHS, Rule> rules;
	private int divisionCount;
		
	/**
	 * Recursively add <code>Rules</code> to a <code>DeterministicRecursiveRuleSet</code>.
	 * @param cell the current cell
	 */
	private void addRule(Cell cell) {
		if (!cell.getTerm()) {
			createRule(cell);
			addRule(cell.getLeft());
			addRule(cell.getRight());
		}
	}
	
	/**
	 * Create a new <code>Rule</code> and add it to the <code>DeterministicRecursiveRuleSet</code> if
	 * it doesn't already exist.
	 * @param cell the current cell
	 */
	private void createRule(Cell cell) {
		Rule rule = new Rule(cell);
		rule.clearTerminalFlags();
		if (rules.containsKey(rule.rhs))
			rules.get(rule.rhs).incCount();
		else
			rules.put(rule.rhs, rule);
	}
}

