import Utils  
from random import Random
from Simulation import Simulation

def run_sim(params, exp_prefix, rng, sim=True):

    """
    Run multiple initialisations of a single set of experimental conditions.

    If sim=False, output is estimated, rather than simulated.  Currently, this
    is just used for evaluating the "fixed time investment" heuristic.
    """
    
    roi_values = []
    total_r_values = []
    mean_tg_values = []

    # for each initialisation
    for run in range(params['runs']):
        params['seed'] = rng.randint(0,99999999)
        params['prefix'] = exp_prefix + "%d/" % params['seed']
        print "   run #%d" % run

        # run experiment; either simulation or estimation
        exp = Simulation(params)
        if sim:
            exp.run()
        else:
            exp.test_flat(params['fixed_time'])

        roi_values.append(exp.calc_roi(1)[2])
        total_r_values.append(exp.calc_mean_total_output())
        mean_tg_values.append(exp.calc_mean_time_grant())

        if params['write_output']:
            Utils.create_dir(params['prefix'])
            exp.write_output()

    # calculate and write summary statistics
    summ_data = [roi_values, total_r_values, mean_tg_values]

    summ_stats = []
    summ_stats.extend(Utils.get_mean_and_std(roi_values))
    summ_stats.extend(Utils.get_mean_and_std(total_r_values))
    summ_stats.extend(Utils.get_mean_and_std(mean_tg_values))
    
    if params['write_output']:
        Utils.write_data_2d(summ_data, exp_prefix + 'summ_data.csv')
        Utils.write_data(summ_stats, exp_prefix + 'summ_stats.csv')

    return summ_stats[:2]


def run_WCSS_sims(params, base_prefix, seed_rng):

    """
    Run basic simulations associated with the WCSS paper.
    """
    params['runs'] = 10

    # run THERMOSTAT model
    params['prefix'] = base_prefix + "thermostat/"
    params['learning_type'] = 'thermostat'
    print 'thermostat'
    run_sim(params, params['prefix'], seed_rng)

    # run MEMORY A model ("bad" parameters)
    params['prefix'] = base_prefix + "memory_A/"
    params['learning_type'] = 'memory'
    params['prob_reentry'] = 0.05
    params['run_length'] = 5
    print 'memory A'
    run_sim(params, params['prefix'], seed_rng)

    # run MEMORY B model ("good" parameters)
    params['prefix'] = base_prefix + "memory_B/"
    params['learning_type'] = 'memory'
    params['prob_reentry'] = 0.02
    params['run_length'] = 3
    print 'memory B'
    run_sim(params, params['prefix'], seed_rng)

    # run FIXED model
    params['prefix'] = base_prefix + "fixed/"
    print 'fixed'
    run_sim(params, params['prefix'], seed_rng, sim=False)


def init_params():

    params = {}

    # simulation parameters
    params['write_output'] = True     # whether or not to write output on individual runs
    params['prefix'] = '/home/nic/Projects/Grant model/results/test/'    # where to write output
    params['runs'] = 1    # number of runs per parameter combination
    params['random_seed'] = True    # whether to use random seed (or fixed)
    params['seed'] = 1234    # seed to use (if random_seed==False)
    params['pop_size'] = 100    # number of academic agents
    params['iterations'] = 100    # number of iterations to simulate
    params['init_time'] = 0.5    # upper bound on initial academic time_grant values
    params['fixed_time'] = 0.1    # ie, for legislated time alloc

    # grant parameters
    params['weight_grant'] = 1.0    # weighting on grant quality, rather than track record
    params['grant_slope'] = 2.0    # slope constant in tanh function
    params['research_slope'] = 2.0    # slope constant in tanh function
    params['grant_noise'] = 0.1    # std dev of gaussian noise on grant quality
    params['rq_counts'] = True    # is research_quality involved in grant_quality?
    params['grant_bonus'] = 1.25    # G: bonus to research output arising from grants
    params['grant_proportion'] = 0.4    # P: proportion of population who can obtain grants
    params['grant_pools'] = 1    # number of pools for grant evaluation

    # self learning parameters
    params['learning_type'] = 'thermostat'   # options 'thermostat', 'memory'
    params['self_update_width'] = 0.1    # "learning rate"
    params['self_update_width_fixed'] = True    # Are learning steps of fixed width? (or random)
    params['memory_size'] = 12    # number of memory steps to store (NB: this is NOT window length!)
    params['run_length'] = 3    # W: number of memory steps to consider (THIS is window length!)
    params['prob_reentry'] = 0.02    # E: probability of re-entering population after dropping out
    params['reentry_range'] = 0.2    # upper bound on re-entry time_grant values

    return params


if __name__ == '__main__':

    params = init_params()
    Utils.create_dir(params['prefix'])
    seed_rng = Random(params['seed'])

    # WCSS SIMULATIONS
    run_WCSS_sims(params, params['prefix'], seed_rng)

    # SINGLE TEST RUN
    #run_sim(params, params['prefix'], seed_rng, sim=True)


###############################################################################
###############################################################################
###############################################################################
###############################################################################
###############################################################################

