import pylab
from Population import Population
import Utils

class Simulation:

    """
    A single experimental run, together with data and derived statistics.
    """

    def __init__(self, params):

        self.params = params
        self.init_stats()
        self.sim = Population(params)
        self.colours = Utils.create_colour_dict(self.params['pop_size'])
        self.plot_colours = Utils.create_plot_colour_dict()
        self.endgame = int(self.params['iterations'] * 0.9)

    def init_stats(self):

        "Initialise data structures for storing data."""

        self.all_stats = []     # for endgame iterations
        self.final_stats = []   # for final iteration
        self.alloc = []
        self.accept = []

        self.all_tg = []
        self.tg_g = []
        self.tg_fg = []
        self.tg_ng = []
        
        self.all_r = []
        self.r_g = []
        self.r_fg = []
        self.r_ng = []
        
        self.rq_g = []
        self.rq_ng = []
        self.rq_fail = []
        self.rq_na = []

        self.corr_rq_tg = []
        self.corr_rq_apply = []
        self.corr_rq_held = []


    def run(self):

        "Run the experiment."

        for t in xrange(self.params['iterations']):       
            self.sim.produce_applications()
            self.sim.evaluate_applications()
            self.sim.produce_research()
            self.sim.update_strategies()
            self.save_base_stats(t)
            if self.params['write_output']:
                self.save_plot_stats()
            self.sim.clear_all()


    def test_flat(self, fixed_time):

	"""
	Run a simulation in which all agents use a fixed time allocation.

	That is, no learning occurs.

	NB: there will still be some variation from iteration to iteration
	due to noise in the calculation of grant quality.
	"""

        for a in self.sim.agents:
            a.time_grant = fixed_time

        for t in xrange(self.endgame, self.params['iterations']):
            self.sim.produce_applications()
            self.sim.evaluate_applications()
            self.sim.produce_research()
            self.save_base_stats(t)
            if self.params['write_output']:
                self.save_plot_stats()
            self.sim.clear_all()


    def save_base_stats(self, t):

        "Save numerical stats related to efficiency."

        if t >= self.endgame:
            self.all_stats.append(self.sim.all_stats())
        if t >= self.params['iterations'] - 1:
            self.final_stats = self.sim.all_stats()


    def save_plot_stats(self):

        "Save additional stats for plotting output."

        alloc_accept = self.sim.acceptance_rate()
        self.alloc.append(alloc_accept[0])
        self.accept.append(alloc_accept[1])

        self.all_tg.append(self.sim.all_tg())
        self.tg_g.append(self.sim.all_tg_grant())
        self.tg_fg.append(self.sim.all_tg_fail())
        self.tg_ng.append(self.sim.all_tg_no_grant())

        self.all_r.append(self.sim.all_r())
        self.r_g.append(self.sim.all_r_grant())
        self.r_fg.append(self.sim.all_r_fail())
        self.r_ng.append(self.sim.all_r_no_grant())

        self.rq_g.append(self.sim.all_rq_grant())
        self.rq_ng.append(self.sim.all_rq_no_grant())
        self.rq_fail.append(self.sim.all_rq_fail())
        self.rq_na.append(self.sim.all_rq_no_apply())

        self.corr_rq_tg.append(pylab.corrcoef(self.sim.all_rq(), 
            self.sim.all_tg())[0][1])
        self.corr_rq_apply.append(pylab.corrcoef(self.sim.all_rq(), 
            self.sim.all_apply())[0][1])
        self.corr_rq_held.append(pylab.corrcoef(self.sim.all_rq(), 
            self.sim.all_held())[0][1])

    
    def calc_mean_total_output(self):

        total_r_sum = 0
        count = 0
        for snap in self.all_stats:
            for a in snap:
                total_r_sum += a[5]
            count += 1
        return total_r_sum / count


    def calc_mean_time_grant(self):

        mean_tg_sum = 0
        count = 0
        for snap in self.all_stats:
            for a in snap:
                mean_tg_sum += a[3]
                count += 1
        return mean_tg_sum / count


    def calc_mean_corr_rq_held(self):
        
        corr_rq_held_sum = 0
        count = 0
        for snap in self.all_stats:
            rq_list = []
            held_list = []
            for a in snap:
                rq_list.append(a[1])
                held_list.append(a[4])
            corr_rq_held_sum += pylab.corrcoef(rq_list, held_list)[0][1]
            count += 1
        return corr_rq_held_sum / count
        
    
    def calc_roi(self, funding_units):

        "Calculate estimated return on investment obtained."

        total_r = self.calc_mean_total_output()
        no_funding = self.sim.estimate_output(0.0, 0.0, 0.0, 'max')
        roi = (total_r - no_funding) / funding_units
        return total_r, no_funding, roi


    def write_output(self):

        "Write data plots."

        # calculate means and sums of raw data
        t_all_tg = Utils.transpose(self.all_tg)
        mean_tg, sum_tg = Utils.get_mean_and_sum(self.all_tg)
        mean_tg_grant, sum_tg_grant = Utils.get_mean_and_sum(self.tg_g)
        mean_tg_fail, sum_tg_fail = Utils.get_mean_and_sum(self.tg_fg)
        mean_tg_no_grant, sum_tg_no_grant = Utils.get_mean_and_sum(self.tg_ng)

        t_all_r = Utils.transpose(self.all_r)
        mean_r, sum_r = Utils.get_mean_and_sum(self.all_r)
        mean_r_grant, sum_r_grant = Utils.get_mean_and_sum(self.r_g)
        mean_r_fail, sum_r_fail = Utils.get_mean_and_sum(self.r_fg)
        mean_r_no_grant, sum_r_no_grant = Utils.get_mean_and_sum(self.r_ng)

        mean_rq_grant, sum_rq_grant = Utils.get_mean_and_sum(self.rq_g)
        mean_rq_no_grant, sum_rq_no_grant = Utils.get_mean_and_sum(self.rq_ng)
        mean_rq_fail, sum_rq_fail = Utils.get_mean_and_sum(self.rq_fail)
        mean_rq_na, sum_rq_na = Utils.get_mean_and_sum(self.rq_na)


        Utils.write_data_2d(self.sim.funding_body.successful_app_stats,
                self.params['prefix']+"successful_app_stats.csv")

        # save final iteration stats
        Utils.write_data_2d(self.final_stats, 
                self.params['prefix']+"final_stats.csv")


        #### WRITE PLOTS

        # number of grants vs research quality
        Utils.write_plot([self.sim.all_rq()], [self.sim.all_grant_counts()],
                self.params['prefix']+"lifetime_grants", "lifetime grants",
                'research quality', 'grants awarded', self.plot_colours,
                ylim = (0, int(self.params['iterations'])), marker='o',lw=0)

        # number of grants awarded
        Utils.write_plot([range(len(self.alloc))], [self.alloc],
                self.params['prefix']+"grants_awarded", "grants awarded",
                'iterations', 'grants_awarded', self.plot_colours,
                ylim = (0, int(self.params['grant_proportion'] * 
                    self.params['pop_size'])))

        # acceptance rate (allocated / submitted)
        Utils.write_plot([range(len(self.accept))], [self.accept],
                self.params['prefix']+"accept_rate", 
                "acceptance rate",
                'iterations', 'acceptance rate', self.plot_colours,
                ylim = (0, 1))

        # all time_grant values
        Utils.write_plot([range(len(t_all_tg[0]))]*len(t_all_tg), t_all_tg,
                self.params['prefix']+"tg_all", "time_grant", 
                'iterations', 'time_grant', self.colours, 
                keys = self.sim.all_rq())

        # all research values
        Utils.write_plot([range(len(t_all_r[0]))]*len(t_all_r), t_all_r,
                self.params['prefix']+"r_all", "research", 
                'iterations', 'research', self.colours,
                keys = self.sim.all_rq())

        # mean time_grant
        Utils.write_plot(
                [range(len(mean_tg)), range(len(mean_tg_grant)), 
                    range(len(mean_tg_no_grant)), range(len(mean_tg_fail))], 
                [mean_tg, mean_tg_grant, mean_tg_no_grant, mean_tg_fail],
                self.params['prefix']+"tg_mean", '',
                'iterations', 'mean time_grant', self.plot_colours, 
                labels=('all', 'grant', 'no grant', 'fail'))

        # mean research
        Utils.write_plot(
                [range(len(mean_r)), range(len(mean_r_grant)), 
                    range(len(mean_r_no_grant)), range(len(mean_r_fail))], 
                [mean_r, mean_r_grant, mean_r_no_grant, mean_r_fail],
                self.params['prefix']+"r_mean", '',
                'iterations', 'mean research', self.plot_colours, 
                labels=('all', 'grant', 'no grant', 'fail'))

        # sum time_grant
        Utils.write_plot(
                [range(len(sum_tg)), range(len(sum_tg_grant)), 
                    range(len(sum_tg_no_grant)), range(len(sum_tg_fail))], 
                [sum_tg, sum_tg_grant, sum_tg_no_grant, sum_tg_fail],
                self.params['prefix']+"tg_sum", '',
                'iterations', 'sum time_grant', self.plot_colours, 
                labels=('all', 'grant', 'no grant', 'fail'))

        # sum research
        Utils.write_plot(
                [range(len(sum_r)), range(len(sum_r_grant)), 
                    range(len(sum_r_no_grant)), range(len(sum_r_fail))], 
                [sum_r, sum_r_grant, sum_r_no_grant, sum_r_fail],
                self.params['prefix']+"r_sum", '',
                'iterations', 'sum research', self.plot_colours, 
                labels=('all', 'grant', 'no grant', 'fail'))

        # mean research_quality of successful applicants
        Utils.write_plot(
                [[], range(len(mean_rq_grant)), range(len(mean_rq_no_grant)),
                    range(len(mean_rq_fail)), range(len(mean_rq_na))], 
                [[], mean_rq_grant, mean_rq_no_grant, mean_rq_fail, mean_rq_na],
                self.params['prefix']+"x_rq_spread", '',
                'iterations', 'mean research quality', self.plot_colours, 
                labels=('', 'grant', 'no grant', 'fail', 'not apply'))

        # correlations: rq v tg
        Utils.write_plot(
                [range(len(self.corr_rq_tg)), range(len(self.corr_rq_apply)), 
                    range(len(self.corr_rq_held))], 
                [self.corr_rq_tg, self.corr_rq_apply, self.corr_rq_held],
                self.params['prefix']+"corr_rq", '',
                'iterations', 'corrcoef', self.plot_colours,
                labels = ('(rq, tg)', '(rq, apply)', '(rq, held)'),
                ylim = (-1,1))

