from random import Random
from FundingAgency import FundingAgency
from Academic import Academic
from Application import Application


class Population:

    """
    A population of acadmic agents.

    This class handles population level actions such as producing grants and
    research, and allocating funding.  It also provides an interface to obtain
    data/statistics on agents.
    """

    def __init__(self, params):
        self.params = params

        if self.params['random_seed']:
            self.rng = Random()
        else:
            self.rng = Random(self.params['seed'])

        # calculate derived parameters
        params['grant_count'] = int(self.params['pop_size'] * 
                self.params['grant_proportion'])

        # initialise agents
        self.funding_body = FundingAgency(params)
        self.agents = []
        for i in xrange(self.params['pop_size']):
            self.agents.append(Academic(i, params, self.rng))


    def estimate_output(self, bonus, prop, time=0.0, type='rnd'):

        """
        Estimate the total research output of a population given:
        - type='max' : best possible allocation of grants
        - type='rnd' : random allocation of grants
			(averaged over several attempts)
        - type='min' : worst possible allocation of grants
        and that each individual spends a fixed and equal amount of 
        time on their applications.
        """
        
        attempts = 1

        rq_agents = [(a.research_quality, a) for a in self.agents]
        if type == 'max':
            rq_agents.sort(reverse=True)
        elif type == 'min':
            rq_agents.sort()
        elif type == 'rnd':
            attempts = 10
        
        research_sum = 0.0
        grant_number = int(len(rq_agents) * prop)

        for i in range(attempts):
            for a in rq_agents[:grant_number]:
                research_sum += a[1].calc_research(time, True, bonus, a[0])
            for a in rq_agents[grant_number:]:
                research_sum += a[1].calc_research(time, False, bonus,a[0])

        return research_sum / attempts


    ## SIM ACTIONS ############################################################

    def produce_applications(self):

        """
        Produce applications by each agent (who is applying).
        """

        [self.funding_body.add_application(
            Application(a, self.params, self.rng), self.rng)
            for a in self.agents if a.applying]


    def evaluate_applications(self):

        """
        Evalute the submitted applications and allocate grants.
        """

        self.funding_body.rank_applications()
        successful = self.funding_body.get_grant_recipients(self.params)
        for a in successful:
            self.agents[a].grant_held = True
            self.agents[a].grant_count += 1


    def produce_research(self):

        """
        Produce research by each agent.  Return total research.
        """

        return sum([a.produce_research(self.params) for a in self.agents])


    def update_strategies(self):

        """
        Update agent strategies.
        """

        for a in self.agents:
            if self.params['learning_type'] == 'thermostat':
                a.update_strategy_self_thermostat(self.params, self.rng)
            elif self.params['learning_type'] == 'memory':
                a.update_strategy_self_memory(self.params, self.rng)
            else:
                System.exit("Unknown learning type")


    def clear_all(self):

        """
        Clear any grants currently held by agents.
        """

        for a in self.agents:
            a.grant_held = False
        self.funding_body.clear_applications()


    ## DATA ACCESS ############################################################

    def all_stats(self):

        """
        Return a table of (for the current iteration):
        [ID, rq, app, tg, g, r]
        """

        return [ 
                (a.id, a.research_quality, a.applying,
                    a.time_grant, a.grant_held, a.research)
                for a in self.agents
                ]

    def acceptance_rate(self):

        """
        Return tuple containing # grants allocated and acceptance rate:
        (# grants allocated, (# grants allocated) / (# grants submitted)).
        """

        submitted = 0
        allocated = 0
        for a in self.agents:
            if a.applying: submitted += 1
            if a.grant_held: allocated += 1
        if submitted > 0:
            return allocated, float(allocated) / submitted
        else:
            return allocated, 0.0

    
    def all_rq(self):

        "Return a list of all research quality values."

        return [a.research_quality for a in self.agents]


    def all_r(self):

        "Return a list of all research output values for current year."

        return [a.research for a in self.agents]


    def all_tg(self):

        "Return a list of all tg values."

        return [a.time_grant for a in self.agents]


    def all_apply(self):

        "Return a list of all applying values."

        return [a.applying for a in self.agents]


    def all_held(self):

        "Return a list of all grant_held values."

        return [a.grant_held for a in self.agents]


    def all_r_grant(self):

        "Return a list of tg values of agents holding grants."

        return [a.research for a in self.agents if a.grant_held]


    def all_r_fail(self):

        """
        Return a list of r values of agents who apply but fail.
        """

        return [a.research for a in self.agents if
                (a.applying and not a.grant_held)]


    def all_r_no_grant(self):

        "Return a list of tg values of agents not holding grants."

        return [a.research for a in self.agents if not a.grant_held]


    def all_tg_grant(self):

        "Return a list of tg values of agents holding grants."

        return [a.time_grant for a in self.agents if a.grant_held]


    def all_tg_fail(self):

        """
        Returns a list of tg values of agents who apply but fail.
        """

        return [a.time_grant for a in self.agents if
                (a.applying and not a.grant_held)]


    def all_tg_no_grant(self):

        "Return a list of tg values of agents not holding grants."

        return [a.time_grant for a in self.agents if not a.grant_held]


    def all_rq_grant(self):

        "Return a list of rq values of agents holding grants."

        return [a.research_quality for a in self.agents if a.grant_held]


    def all_rq_no_grant(self):

        "Return a list of rq values of agents not holding grants."

        return [a.research_quality for a in self.agents if not a.grant_held]


    def all_rq_fail(self):

        "Returns a list of rq values of agents who apply but fail."

        return [a.research_quality for a in self.agents if 
                (a.applying and not a.grant_held)]


    def all_rq_no_apply(self):
        
        "Returns a list of rq values of agents who don't apply."

        return [a.research_quality for a in self.agents if not a.applying]


    def all_grant_counts(self):

        "Returns a list of lifetime grant counts for each agent."

        return [a.grant_count for a in self.agents]

    def print_all(self):
        for a in self.agents:
            a.print_all()

###############################################################################
###############################################################################
###############################################################################
###############################################################################
###############################################################################

