/* These tools are Free Software, licensed under the MIT license. *
 *  Copyright 2007, Philip Boulain. See LICENSE.TXT for details.  */
#include <stdio.h>
#include <string.h>
#include "mediawikiformat.h"
#include "mediawikimarkup.h"

static unsigned int failures;

static void test_eq(const char* test, long expected, long actual) {
	bool pass = (expected == actual);
	printf("%s: %s", test, pass ? "pass" : "FAIL");
	if(pass) { printf(" (%ld)", actual); }
	puts("");
	if(!pass) {
		printf("\tExpected: %ld\n\t  Actual: %ld\n", expected, actual);
		failures++;
	}
}

static void test_distances(void) {
	upstr a;
	upstr b;
	upstr_init(&a);
	upstr_init(&b);

	upstr_set(&a, "kitten");
	upstr_set(&b, "sitting");
	test_eq("Lev kitten      ", 3, upstr_levenshtein(&a, &b));
	test_eq("Dst kitten      ", 3, upstr_distance   (&a, &b));
	test_eq("Lev kitten rev  ", 3, upstr_levenshtein(&b, &a));
	test_eq("Dst kitten rev  ", 3, upstr_distance   (&b, &a));
	upstr_set(&b, "kitten");
	test_eq("Lev kittens     ", 0, upstr_levenshtein(&a, &b)); // Identical
	test_eq("Dst kittens     ", 0, upstr_distance   (&a, &b));
	upstr_set(&b, "");
	test_eq("Lev solo kitten ", 6, upstr_levenshtein(&a, &b)); // Just add
	test_eq("Dst solo kitten ", 6, upstr_distance   (&a, &b));
	upstr_set(&a, "");
	test_eq("Lev blanks      ", 0, upstr_levenshtein(&a, &b)); // Null compare
	test_eq("Dst blanks      ", 0, upstr_distance   (&a, &b));
	upstr_set(&a, "kitten");
	upstr_set(&b, "puppy");
	test_eq("Lev kitten puppy", 6, upstr_levenshtein(&a, &b)); // No commonality
	test_eq("Dst kitten puppy", 6, upstr_distance   (&a, &b));
#define MOARSTRINGS(N, A, B, L, D) upstr_set(&a, (A)); upstr_set(&b, (B)); \
	test_eq("Lev " N, L, upstr_levenshtein(&a, &b)); \
	test_eq("Dst " N, D, upstr_distance(&a, &b));

	MOARSTRINGS("AD->ABCD    ", "AD", "ABCD", 2, 2)
	MOARSTRINGS("ABCD->AD    ", "ABCD", "AD", 2, 2)
	MOARSTRINGS("ABCD->APQD  ", "ABCD", "APQD", 2, 2)
	MOARSTRINGS("CDEFGHCQDQGH", "CDEFGH", "CQDQGH", 3, 3) // Self-similar
	MOARSTRINGS("CDEFGHCQGQGH", "CDEFGH", "CQGQGH", 3, 3) // Self-similar
	MOARSTRINGS("tri-overlap ", "ABYZC", "AQBXC", 3, 3)
	MOARSTRINGS("tri-overlap2", "ABC", "A123B456C", 6, 6)
	MOARSTRINGS("trapezoidal ", "ABCZD", "AZBCD", 2, 2)
	MOARSTRINGS("selfsim tail", "CDEFGH1234567890", "CQDQGH1234567890", 3, 3)
	MOARSTRINGS("long puresub",
		"ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789",
		"ABCDEfghIJKLMNOPqRSTUVWXYZ2103456789",
		6, 6)

	puts("\tNow for test cases which are algorithm bugs in distancer...");

	MOARSTRINGS("overlap ins.","ABCF","ADCDBCF",3,4)
	MOARSTRINGS("overlap sub.","APQCD","ACXYD",3,2)
	MOARSTRINGS("ol. ins. tl.","ABCF1234567890","ADCDBCF1234567890",3,4)
	MOARSTRINGS("add phrase  ",
		"Cats are the most awesome creatures known to creation.",
		"Cats are the most awesome creatures, outside of dragons, known to creation.",
		21, 33)
	MOARSTRINGS("add phrase R",
		"Cats are the most awesome creatures, outside of dragons, known to creation.",
		"Cats are the most awesome creatures known to creation.",
		21, 33)
	MOARSTRINGS("long mixture",
		"The most famous cat's are garfield and my pet fluffeh.",
		"The most famous cats are Garfield and Ceiling Cat.",
		15, 14) // 2('+g/G); 4 before 'e'; 2 before ' '; 6 before '.'

	puts("\tNow for a few real-world cases...");

	// http://en.wikipedia.org/w/index.php?title=Anarchism&diff=next&oldid=42738
	MOARSTRINGS("anarchism 6a",
		"* The [http://www.mises.org/\nLudwig von Mises Insitute]",
		"* The [http://www.mises.org/index.html\nvon Mises Insitute]",
		11, 10)
#undef MOARSTRINGS

	upstr_destroy(&a);
	upstr_destroy(&b);
}

static void test_parser(void) {
	upstr a;
	upstr b;
	upstr_init(&a);
	upstr_init(&b);
	mw_markupinfo info;
	mw_markupinfo_init(&info);

#define PRINT_BAG(LABEL, BAG) \
	{ unsigned int i; upstr* str; \
		printf("--- %s:\n", (LABEL)); \
		MD_BAG_FOREACH((BAG), i, str) \
			puts((char*) str->chars); \
		MD_BAG_FOREACH_DONE() }

	upstr_set(&a,
"=== DUBIOUS HEADING ===\n"
"This is a fake Wikipedia node which claims that 1 + 1 == 2.\n"
"'''Phil's head ''exploded.''''' <!-- haha --> Ouch---got to hurt.\n"
"  \t  Preformatted?\n"
"#REDIRECT [[Space]] EOL\n"
"[http://www.foo.com] http://www.bar.com/ [file:/dev/null] EOL\n"
"[[Foxes]] [[Dragons|TROGDOR]] [[Wolves|doctor=present|Badwolf]] EOL\n"
"The keyword __dllexport is __magical__ in MSVC++. _Seriously_.\n"
"{{#expr 1 + 1}} {{Transclude}} {|wibble!Table contents| contain+ funky! characters|} EOL\n"
"----\n"
"[[Category:Gibberish]] EOL\n"
	);
	mw_stripMarkupRoughly(&a, &b, &info);
	printf("--- Wikitext:\n%s--- Plaintext:\n%s", a.chars, b.chars);
	PRINT_BAG("Templates",  info.templates)
	PRINT_BAG("Categories", info.categories)
	PRINT_BAG("Page links", info.pagelinks)
	PRINT_BAG("URL links",  info.urllinks)
	puts("--- End");
#undef PRINT_BAG

	upstr_destroy(&a);
	upstr_destroy(&b);
	mw_markupinfo_destroy(&info);
}

bool initFilter(int argc, char** argv) {
	failures = 0;
	
	test_distances(); puts("");

	test_parser(); puts("");

	if(failures) { printf("!!! %u TESTS FAILED\n", failures); }
	puts("Tests over; bailing now (superflous invalid arguments error)...");
	return false;
}

void killFilter(void) {}

bool generatesXMLOutput(void) { return false; }

bool processPage(const mw_page* page) {
	return false;
}

void processRevision(const mw_page* page, const mw_revision* revision) {}

