/* Adapted bag class from Mandala, licensed under the MIT license.   *
 * Also available under the CPL at:                                  *
 *   http://www.ecs.soton.ac.uk/~prb/projects/mandala/               *
 * Copyright 2005-2007, Philip Boulain. See LICENSE.TXT for details. */

/* As well as removing Mandala-environment-isms, this version is less inclined
 * to abort(), and more inclined to complain to STDERR and return error states.
 * Some function signatures have been changed to accommodate this; some outright
 * removed as they are unsafe without greater modification.
 * Memory allocation (OptimizeSize()) has been rewritten, as the old one was
 * broken when it wanted to shrink a bag, and instead enlarged it (!). */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include "bag.h"

struct MD_Bag_s {
	void** start;        /* Pointer to start of array.       */
	unsigned int length; /* Number of elements in the array. */
	unsigned int size;   /* Space allocated. Internal use.   */
};

static bool MD_Bag_OptimiseSize(MD_Bag* b);


MD_Bag*  MD_Bag_New(void) {
	MD_Bag* self = malloc(sizeof(MD_Bag));
	if(self) {
		self->start  = 0;
		self->length = 0;
		self->size   = 0;
	}
	return self;
}

bool  MD_Bag_Add(MD_Bag* b, void* item) {
	if(b->length == UINT_MAX) { return false; } /* Can't address it! */
	/* Make room */
	b->length++;
	if(MD_Bag_OptimiseSize(b)) {
		/* Tack it on the end */
		b->start[b->length - 1] = item;
		return true;
	} else {
		return false;
	}
}

/*
bool  MD_Bag_AddUniq(MD_Bag* b, void* item) {
	if(MD_Bag_Contains(b, item)) {
		return true;
	} else {
		// Potential silent failure!
		MD_Bag_Add(b, item);
		return false;
	}
}

void  MD_Bag_AddBag(MD_Bag* self, MD_Bag* other) {
	unsigned int i;
	unsigned int offset = self->length; // Start for data write
	// Increase our length by that of the other bag and make room
	self->length += other->length;
	MD_Bag_OptimiseSize(self);
	// Copy across data
	for(i = 0; i < other->length; i++) {
		self->start[i + offset] = other->start[i];
	}
}
*/

void* MD_Bag_Remove(MD_Bag* b, void* item) {
	unsigned int index;
	for(index = 0; index < b->length; index++) {
		if(b->start[index] == item)
			return MD_Bag_RemoveI(b, index);
	}
	return 0;
}

void* MD_Bag_RemoveI(MD_Bag* b, unsigned int index) {
	void* item;
	if(index < b->length) {
		fprintf(stderr, "Requested removal from item (%u) out of bag bounds (%u); giving NULL!\n", index, b->length);
		return 0;
	}
	item = b->start[index];
	b->length--;
	if(b->length > 0) /* Overwrite the deleted item with the tail */
		b->start[index] = b->start[b->length];
	/* Do NOT OptimiseSize here, because that may realloc() the array and
	 * screw up iteration through it by breaking the cached start. It will
	 * just have to wait until the next addition, clear, or such. */
	return item;
}

void **MD_Bag_Contents(const MD_Bag *b) {
	return b->start;
}

unsigned int MD_Bag_Length(const MD_Bag *b) {
	return b->length;
}

bool  MD_Bag_Contains(MD_Bag* b, void* item) {
	unsigned int index;
	void* idx;
	MD_BAG_FOREACH(b, index, idx)
		if(idx == item) { return true; }
	MD_BAG_FOREACH_DONE()
	return false;
}

void  MD_Bag_Clear(MD_Bag* b) {
	b->length = 0;
	/* Cannot try to allocate more memory, thus cannot fail badly, unless
	 * bag was already empty with zero bytes, in which case the status
	 * quo is merely maintained. */
	MD_Bag_OptimiseSize(b);
}

void  MD_Bag_Destroy(MD_Bag* b) {
	if(b->start)
		{ free(b->start); }
	free(b);
}

/* Make 'size' some sane amount larger than 'length'. */
static bool MD_Bag_OptimiseSize(MD_Bag* b) {
	/* Must always have space to add one more thing */
	if(b->length + 1 > b->size) {
		void** newlump;
		size_t newsize = b->size * 2;
		if(!newsize) { newsize = 1; }
		/* At first, b->start == 0, so this just acts like malloc() */
		newlump = (void**)realloc(b->start, sizeof(void*) * newsize);
		/* Check memory was availible (will also fail if size == 0) */
		if(newlump) {
			b->start = newlump;
			b->size  = newsize;
		} else {
			fprintf(stderr, "Cannot expand full bag\n");
			return false;
		}
	}
	/* Try not to waste too much space */
	else if((b->length + 1) < (b->size / 2)) {
		void** newlump;
		size_t newsize = b->size / 2;
		newlump = (void**)realloc(b->start, sizeof(void*) * newsize);
		if(newlump) {
			b->start = newlump;
			b->size  = newsize;
		} /* Else, too bad. This isn't a big problem. */
	}
	return true;
}

