/* Adapted bag class from Mandala, licensed under the MIT license.   *
 * NOT SUITABLE FOR BACKPORTING INTO MANDALA. See implementation.    *
 * Also available under the CPL at:                                  *
 *   http://www.ecs.soton.ac.uk/~prb/projects/mandala/               *
 * Copyright 2005-2007, Philip Boulain. See LICENSE.TXT for details. */
#ifndef MD_MANDALA_BAG_H_
#define MD_MANDALA_BAG_H_

#include <stdbool.h>

/** Bag ************************************************************************
 *  A 'bag' is a collection whose order is not guaranteed.
 *  This implementation stores void pointers and is designed to be as fast as possible. */
typedef struct MD_Bag_s MD_Bag;
/** May return NULL if insufficient memory. */
MD_Bag* MD_Bag_New(void);
/** Destroy the bag. As for clear(), no deallocation of contents is made. */
void  MD_Bag_Destroy(MD_Bag* self);
/** Get a pointer to the start of the bag's contents.
 *  Use this and the length to access them, for speed.
 *  If you go out of the bounds contents[0] to contents[length - 1], you will
 *  segfault. This would be Bad. Do not do it.
 *  
 *  \sa MD_BAG_FOREACH() */
void** MD_Bag_Contents(const MD_Bag* self);
/** Get the number of items in the bag. */
unsigned int MD_Bag_Length(const MD_Bag* self);
/** Add an item to the bag. Returns true if this succeeded.
 *  O(1), although constant may include a realloc() occasionally.
 *  WARNING: API change from Mandala's bags, which returned the item. */
bool  MD_Bag_Add(MD_Bag* self, void* item);

#if 0
/** Add an item to the bag if not already in it.
 *  O(N). Returns true if the item was already in the bag. */
bool  MD_Bag_AddUniq(MD_Bag* self, void* item);
/** Add the contents of another bag to this one.
 *  This is more efficient than looping and calling Add(). */
void  MD_Bag_AddBag(MD_Bag* self, MD_Bag* other);
#endif

/** Remove an item from the bag by its pointer value.
 *  This requires a pass over the bag contents, so is O(N).
 *  If the same time has been added multiple times, only the first instance will be removed.
 *  Returns the item removed, or NULL if no such item exists. */
void* MD_Bag_Remove(MD_Bag* self, void* item);
/** Removes a specific index from the bag.
 *  O(1), because you've already found the item while iterating.
 *  Returns the item removed. Will \b abort if the index isn't valid.
 *  
 *  WARNING: Removals may re-order the set. If removing during iteration,
 *  reprocess the index you just removed or the tail of the internal array will
 *  be missed (i.e. decrement your loop iterator). Removals should not otherwise
 *  interfere with iteration, or further removals. */
void* MD_Bag_RemoveI(MD_Bag* self, unsigned int index);
/** Test if the bag contains a particular item. O(N). */
bool  MD_Bag_Contains(MD_Bag* self, void* item);
/** Remove all items from a bag. Makes no attempt to deallocate them. */
void  MD_Bag_Clear(MD_Bag* self);

#if defined(_MSC_VER)
	#define MD_BAG_SETPTR(BAG,ITER,PTR) \
		(*(void**)&(PTR)) = MD_Bag_ForEachContents[(ITER)];
#else // Not MSVC (assume GCC, or at least some other sane compiler)
	#define MD_BAG_SETPTR(BAG,ITER,PTR) \
		           (PTR)  = MD_Bag_ForEachContents[(ITER)];
#endif
/* The crazy "(*(void**)&(PTR))" stuff is an ugly version of "(PTR)".
 * The poor, innocent macro has been corrupted in this fashion to keep non-C99
 * compilers (e.g. MSVC) from throwing an error at the concept of actually
 * implicitly casting a void pointer to something useful, despite the fact
 * that this in no way affects the output binary.
 * BUT IT GETS EVEN MORE FUN. gcc 4.0.3, with optimisation on, will break this
 * code and spew a load of type-punning warnings, causing a pretty segfault.
 * Hence the two versions. Sigh. */
/** Macro to set up a for loop which iterates through the bag. This is a filthy
 *  hack to make C look like a far more elegant language than it really is.
 *  (Damn, I wish C99 had closures.)
 *
 *  DO NOT MODIFY THE BAG IN ANY WAY DURING THIS LOOP.
 *  If you do, Little Miss Segfault is likely to pay you a visit.
 *  (Modifying the actual items in the bag themselves is, of course, fine.)
 *
 *  Arguments: bag, unsigned integer for index, pointer to set.
 *  If using C++, the pointer must be void*, and you'll have to cast it to
 *  something more useful yourself in the loop body. */
#define MD_BAG_FOREACH(BAG,ITER,PTR) \
	{ void** MD_Bag_ForEachContents = MD_Bag_Contents((BAG)); \
	unsigned int MD_Bag_ForEachLength = MD_Bag_Length((BAG)); \
	for((ITER) = 0; (ITER) < MD_Bag_ForEachLength; (ITER)++) { \
		MD_BAG_SETPTR((BAG),(ITER),(PTR))
#define MD_BAG_FOREACH_DONE() }}

// Here we go again...
#if defined(_MSC_VER)
	#define MD_BAG_SETPTR_SAFE(BAG,ITER,PTR) \
		(*(void**)&(PTR)) = MD_Bag_Contents((BAG))[(ITER)];
#else
	#define MD_BAG_SETPTR_SAFE(BAG,ITER,PTR) \
		           (PTR)  = MD_Bag_Contents((BAG))[(ITER)];
#endif
/** Version of FOREACH macro which tolerates bag operations during iteration.
 *  This is, resultingly, marginally slower.
 *
 *  REMEMBER: If you remove during an interation, decrement your iterator! */
#define MD_BAG_FOREACH_SAFE(BAG,ITER,PTR) \
	for((ITER) = 0; (ITER) < MD_Bag_Length((BAG)); (ITER)++) { \
		MD_BAG_SETPTR_SAFE((BAG),(ITER),(PTR))
#define MD_BAG_FOREACH_SAFE_DONE() }

#endif

