/* These tools are Free Software, licensed under the MIT license. *
 *  Copyright 2007, Philip Boulain. See LICENSE.TXT for details.  */
#ifndef UPSTR_H
#define UPSTR_H

#include <stdbool.h>
#include <libxml/parser.h> /* For xmlChar */

// Unicode Pascal-ish string. Used for efficiency, especially as this is how
// libxml throws characters at you. Guaranteed chars[len] = '\0'.
typedef struct {
	size_t len; // Does NOT include the null
	xmlChar* chars;
	size_t alloc; // Don't play with this, children
} upstr;

// Theroetically, could use the restrict keyword in places here:
//   http://developers.sun.com/sunstudio/articles/cc_restrict.html
// But I don't fancy debugging the horrible optimisation errors it might cause

// UPStr empty string init (zeros fields); mostly internal.
void upstr_init(upstr* str);
// UPStr comparison function; useful externally.
bool upstr_eq(const upstr* one, const upstr* two);
// Set an UPStr to the same content as a C string (copy)
void upstr_set(upstr* str, const char* cstr);
// Set an UPStr to the same content as another UPStr (copy)
void upstr_setu(upstr* dest, const upstr* src);
// UPStr expansion function; basically for internal use.
// Tries to ensure that there is room for 'required' bytes;
// if returns false, string is still of old size.
bool upstr_expand(upstr* str, size_t required);
// Append a string (as expressed by libxml2) to an UPStr.
// Returns true if succeeded, false if still just old string.
bool upstr_append(upstr* str, const xmlChar* ch, int len);
// Compute the Levenshtein distance between two UPStrings.
// Memory requirement is double that of the shorter string.
size_t upstr_levenshtein(const upstr* one, const upstr* two);
// Compute the distance between two UPStrings using a simpler, computationally
// cheaper heuristic, which is O(n) on identical strings, and may perform
// limited "rewinding" on differing strings (technically O(n^2), but with
// rare worst-case). This generally returns a result similar to Levenshtein, but
// focuses on local changes; it can over or underestimate by up to about 10% for
// certain types of difference.
size_t upstr_distance(const upstr* one, const upstr* two);
// Destroy an UPStr
void upstr_destroy(upstr* str);

#endif

