% Copyright (C) 2013  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your
% option) any later version.

% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.

% Calculate the average unary codeword length
% distribution is a string specifying the name of a distribution function, e.g. 'distribution_zeta'
% parameter is the parameter of that distribution
% threshold optionally respecifies the threshold that must be reached before terminating the infinite sum
% l is the average unary codeword length
function l = average_unary_codeword_length(distribution, parameter, threshold)

    switch distribution

        case 'distribution_geometric'
            p1 = parameter;

            % See Eq (6) in (Maunder et al., 2013)
            l = 1./p1;

        case 'distribution_zeta'
            s = parameter;

            % See Eq (7) in (Maunder et al., 2013)
            l = zeta(s-1)./zeta(s);

        otherwise

            if nargin < 3
                threshold = 1e-10;
            end

            symbol_count = 1000000;
            l = 0;
            keepgoing = 1;
            symbol_indices = 1:symbol_count;

            % See Eq (5) in (Maunder et al., 2013)
            while keepgoing

                weighted_unary_codeword_lengths = feval(distribution, symbol_indices, parameter).*symbol_indices;

                l = l + sum(weighted_unary_codeword_lengths);

                if(weighted_unary_codeword_lengths(end) < threshold)
                    keepgoing = 0;
                else
                    symbol_indices = symbol_indices + symbol_count;
                end
            end
    end
end