% Measure the mutual information of some LLRs using the histogram method
% Copyright (C) 2013  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your
% option) any later version.

% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.



% Use the histogram method to measure the mutual information of some LLRs
% ztilde is a vector of LLRs, where LLR=ln[Pr(bit=1)/Pr(bit=0)]
% z is the corresponding vector of bits
% I is the mutual information in the range 0 to 1
function I = measure_mutual_information_histogram(ztilde, z)

z = reshape(z,1,numel(z));
ztilde = reshape(ztilde,1,numel(ztilde));


if(length(ztilde) ~= length(z))
    error('Must have same number of llrs and bits!');
end


bit_1_count = sum(z);
bit_0_count = length(z) - bit_1_count;
if(bit_0_count == 0 || bit_1_count == 0)
    I = 0.0;
else
    
    llr_0_noninfinite_count = 0;
    llr_1_noninfinite_count = 0;
    llr_0_max = -Inf;
    llr_0_min = Inf;
    llr_1_max = -Inf;
    llr_1_min = Inf;
    for bit_index = 1:length(z)
        if(ztilde(bit_index) ~= -Inf && ztilde(bit_index) ~= Inf)
            if(z(bit_index) == 0)
                llr_0_noninfinite_count = llr_0_noninfinite_count+1;
                
                if(ztilde(bit_index) > llr_0_max)
                    llr_0_max = ztilde(bit_index);
                end
                if(ztilde(bit_index) < llr_0_min)
                    llr_0_min = ztilde(bit_index);
                end
            else
                llr_1_noninfinite_count = llr_1_noninfinite_count+1;
                
                if(ztilde(bit_index) > llr_1_max)
                    llr_1_max = ztilde(bit_index);
                end
                if(ztilde(bit_index) < llr_1_min)
                    llr_1_min = ztilde(bit_index);
                end
            end
        end
    end
    
    if(llr_0_noninfinite_count > 0 && llr_1_noninfinite_count > 0 && llr_0_min <= llr_1_max && llr_1_min <= llr_0_max)
        llr_0_mean = 0.0;
        llr_1_mean = 0.0;
        for bit_index = 1:length(z)
            if(ztilde(bit_index) ~= -Inf && ztilde(bit_index) ~= Inf)
                if(z(bit_index) == 0)
                    llr_0_mean = llr_0_mean+ztilde(bit_index);
                else
                    llr_1_mean = llr_1_mean+ztilde(bit_index);
                end
            end
            
        end
        llr_0_mean = llr_0_mean/llr_0_noninfinite_count;
        llr_1_mean = llr_1_mean/llr_1_noninfinite_count;
        
        llr_0_variance = 0.0;
        llr_1_variance = 0.0;
        for bit_index = 1:length(z)
            if(ztilde(bit_index) ~= -Inf && ztilde(bit_index) ~= Inf)
                
                if(z(bit_index) == 0)
                    
                    llr_0_variance = llr_0_variance + (ztilde(bit_index) - llr_0_mean)^2;
                else
                    
                    llr_1_variance = llr_1_variance + (ztilde(bit_index) - llr_1_mean)^2;
                end
            end
        end
        llr_0_variance = llr_0_variance/llr_0_noninfinite_count;
        llr_1_variance = llr_1_variance/llr_1_noninfinite_count;
        
        bin_width = 0.5*(3.49*sqrt(llr_0_variance)*(llr_0_noninfinite_count^(-1.0/3.0)) + 3.49*sqrt(llr_1_variance)*(llr_1_noninfinite_count^(-1.0/3.0)));
        if(bin_width > 0.0)
            
            bin_offset = floor(min(llr_0_min, llr_1_min)/bin_width)-1;
            temp = max(llr_0_max, llr_1_max)/bin_width-bin_offset+1;
            bin_count = ceil(temp);
            if(bin_count == temp)
                bin_count = bin_count+1;
            end
            
        else
            
            bin_offset = -1;
            bin_count = 3;
        end
        lots_of_bins = true;
        
    else
        lots_of_bins = false;
        bin_count = 4;
    end
    
    histogram = zeros(2,bin_count);
    
    for bit_index = 1:length(z)
        if(ztilde(bit_index) == -Inf)
            histogram(z(bit_index)+1,1) = histogram(z(bit_index)+1,1)+1;
        elseif(ztilde(bit_index) == Inf)
            histogram(z(bit_index)+1,bin_count) = histogram(z(bit_index)+1,bin_count)+1;
        else
            if(lots_of_bins == true)
                if(bin_width > 0.0)
                    histogram(z(bit_index)+1,floor(ztilde(bit_index)/bin_width)-bin_offset+1) = histogram(z(bit_index)+1,floor(ztilde(bit_index)/bin_width)-bin_offset+1)+1;
                else
                    histogram(z(bit_index)+1,2) = histogram(z(bit_index)+1,2)+1;
                end
            else
                histogram(z(bit_index)+1,z(bit_index)+2) = histogram(z(bit_index)+1,z(bit_index)+2)+1;
            end
        end
    end
    
    prob_y_given_x = zeros(2,bin_count);
    prob_y_given_x(1,:) = histogram(1,:)/bit_0_count;
    prob_y_given_x(2,:) = histogram(2,:)/bit_1_count;
    
    prob_y = (histogram(1,:)+histogram(2,:))/numel(z);
    
    prob_x = zeros(2,1);
    prob_x(1) = bit_0_count/numel(z);
    prob_x(2) = bit_1_count/numel(z);
    
    I = 0.0;
    for bit_value = 0:1
        for bin_index = 1:bin_count
            if(histogram(bit_value+1,bin_index) > 0)
                I = I + prob_y_given_x(bit_value+1,bin_index)*prob_x(bit_value+1)*log2(prob_y_given_x(bit_value+1,bin_index)/prob_y(bin_index));
            end
        end
    end
    
end
