% Copyright (C) 2013  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your
% option) any later version.

% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.



%Script for plotting the inverted EXIT function of a UEC code

% Select a source distribution
distribution = 'distribution_zeta'; parameter = zeta_p1_to_s(0.797);
%distribution = 'distribution_H264'; parameter = 0;

% Select a UEC codeword set
C = [0 1; 1 1]; % r=4
%C = [0 1; 1 1; 1 1]; % r=6
%C = [0 1; 1 1; 1 1; 1 1]; % r=8
%C = [0 1; 1 1; 1 1; 1 1; 1 1]; % r=10

% Select a target length for the UEC-encoded bit vector z
bn = 10000;

% Select the number of frames to simulate
frame_count = 10;

% Select the number of points to plot in the EXIT function
IA_count = 11;

% Select 0 for Log-MAP or 1 for Max-Log-MAP
global approx_maxstar
approx_maxstar = 0;

% Select 0 for AWGN a priori LLRs or 1 for BEC a priori LLRs
bec = 0;






% Determine the UEC codeword length
n = size(C,2);

% Determine the number of states in the UEC trellis
r = 2*size(C,1);

% Determine the average unary codeword length
l = average_unary_codeword_length(distribution,parameter);

% Determine the probabilities of the first r/2-1 symbol values
Px_vector = feval(distribution,1:r/2-1,parameter);

% Determine the length of the symbol vector x to use
a = round(bn/n/l);

% A priori mutual informations to consider
IAs = 0:1/(IA_count-1):1;

% Initialise results
IEs_hist_mean=zeros(size(IAs));
IEs_hist_std=zeros(size(IAs));
IEs_av_mean=zeros(size(IAs));
IEs_av_std=zeros(size(IAs));

% Consider each a priori mutual information
parfor IA_index = 1:IA_count
    IEs_hist = zeros(1,frame_count);
    IEs_av = zeros(1,frame_count);
    
    % For each frame
    for frame_index = 1:frame_count
        
        % Generate some random symbols
        x  = generate_random_symbols(a, distribution, parameter);
        
        % UEC encode the symbols
        z = UEC_encoder(x, C);
        
        % Generate the a priori LLRs having the a priori mutual information considered
        if bec
            ztildea = generate_bec_llrs(z, IAs(IA_index));
        else
            ztildea = generate_awgn_llrs(z, IAs(IA_index));
        end
        
        % UEC decode the LLRs
        ztildee = UEC_decoder(ztildea, C, Px_vector, l, a);
        
        % Measure the mutual information of the extrinsic LLRs
        IEs_hist(frame_index) = measure_mutual_information_histogram(ztildee, z);
        IEs_av(frame_index) = measure_mutual_information_averaging(ztildee);
        
    end
    
    % Determine the mean and standard deviation of the extrinisic mutual informations
    IEs_hist_mean(IA_index) = mean(IEs_hist);
    IEs_hist_std(IA_index) = std(IEs_hist);
    IEs_av_mean(IA_index) = mean(IEs_av);
    IEs_av_std(IA_index) = std(IEs_av);
end

% Measure the area beneath the inverted EXIT function
area_hist = 1-sum((IEs_hist_mean(2:end)+IEs_hist_mean(1:end-1)).*(IAs(2:end)-IAs(1:end-1)))/2;
area_av = 1-sum((IEs_av_mean(2:end)+IEs_av_mean(1:end-1)).*(IAs(2:end)-IAs(1:end-1)))/2;



% Plot the inverted EXIT function
figure
xlim([0 1]);
ylim([0 1]);
axis square
hold on
plot(IEs_hist_mean,IAs,'r-');
plot((IEs_hist_mean+IEs_hist_std),IAs,'r--');
plot((IEs_hist_mean-IEs_hist_std),IAs,'r--');
plot(IEs_av_mean,IAs,'b-');
plot((IEs_av_mean+IEs_av_std),IAs,'b--');
plot((IEs_av_mean-IEs_av_std),IAs,'b--');
xlabel('I_e');
ylabel('I_a');
title('Inverted 2state EXIT function');

% Display the area beneath the inverted EXIT function
annotation('textbox','String',{[distribution, ', param = ', num2str(parameter)]},'LineStyle','none','Position',[0.5 0.30 0.5 0.1]);
annotation('textbox','String',{['Measured hist area frac = ', num2str(area_hist)]},'LineStyle','none','Position',[0.5 0.25 0.5 0.1]);
annotation('textbox','String',{['Measured av area frac = ', num2str(area_av)]},'LineStyle','none','Position',[0.5 0.20 0.5 0.1]);
annotation('textbox','String',{['Predicted area = ', num2str(UEC_exit_area(Px_vector, l, n))]},'LineStyle','none','Position',[0.5 0.15 0.5 0.1]);
annotation('textbox','String',{['Coding rate = ', num2str(UEC_coding_rate(distribution, parameter, n))]},'LineStyle','none','Position',[0.5 0.10 0.5 0.1]);

% Save the results in a file
results = [IAs', IEs_hist_mean', (IEs_hist_mean'+IEs_hist_std'), (IEs_hist_mean'-IEs_hist_std'), IEs_av_mean', (IEs_av_mean'+IEs_av_std'), (IEs_av_mean'-IEs_av_std')];
filename = ['results_UEC_exit_',distribution,'_',num2str(parameter),'_',num2str(size(C,1)*2),'_',num2str(a),'_',num2str(frame_count),'_',num2str(IA_count),'_',num2str(approx_maxstar),'_',num2str(bec),'.txt'];
save(filename, 'results', '-ASCII', '-DOUBLE', '-TABS');



