% Copyright (C) 2013  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your
% option) any later version.

% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.

% Calculate the symbol entropy
% distribution is a string specifying the name of a distribution function, e.g. 'distribution_zeta'
% parameter is the parameter of that distribution
% threshold optionally respecifies the threshold that must be reached before terminating the infinite sum
% HX is the symbol entropy
function HX = symbol_entropy(distribution, parameter, threshold)

    switch distribution

        case 'distribution_geometric'
            p1 = parameter;

            % See Eq (2) in (Maunder et al., 2013)
            HX = (information(p1) + information(1-p1))./p1;            

        case 'distribution_zeta'
            s = parameter;
            zeta_s = zeta(s);

            % See Eq (4) in (Maunder et al., 2013)
            HX = log2(zeta_s) - s.*zeta(1,s)./zeta_s/log(2);

        otherwise

            if nargin < 3
                threshold = 1e-10;
            end

            symbol_count = 1000000;
            HX = 0;
            keepgoing = 1;
            symbol_indices = 1:symbol_count;

            % See paragraph above Eq (1) in (Maunder et al., 2013)
            while keepgoing

                Px_vector = feval(distribution, symbol_indices, parameter);

                HX = HX + sum(information(Px_vector(Px_vector>0)));
                
                if(Px_vector(end) < threshold)
                    keepgoing = 0;
                else
                    symbol_indices = symbol_indices + symbol_count;
                end
            end
    end
end

