% Copyright (C) 2013  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your
% option) any later version.

% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.


% Select simulation parameters
x_max = 4; % Max symbol value for which FLC decoder is invoked
a = 20000; % Source symbol vector length
p_1 = 0.9; % Probability of symbol values having the value of 1
n_2 = 2; % Number of encoded bits to use in CC code - 2 or 3
max_bottom_left_it = 10; % Maximum number of iterations between FLC and CC decoders
approx_maxstar = 0; % Use the Max-Log-MAP algorithm?

% Calculate some dependent parameters
s = zeta_p1_to_s(p_1);
l_2 = get_l_2(s);
P_t_given_x_vector = get_P_t_given_x_vector(x_max,s);
R_o_2 = get_R_2(s,n_2)
A_o_2 = get_A_2(s,n_2,x_max)




% Select the number of frames to simulate
frame_count = 10;

% Select the number of points to plot in the EXIT function
IA_count = 11;

% A priori mutual informations to consider
IAs = 0:1/(IA_count-1):1;

% Initialise results
IEs_hist_mean=zeros(size(IAs));
IEs_hist_std=zeros(size(IAs));
IEs_av_mean=zeros(size(IAs));
IEs_av_std=zeros(size(IAs));

% Consider each a priori mutual information
parfor IA_index = 1:length(IAs)
    IEs_hist = zeros(1,frame_count);
    IEs_av = zeros(1,frame_count);
    
    % For each frame
    for frame_index = 1:frame_count
        
        % Generate random source
        d = generate_zeta_symbols(a, s);

        % Splitter
        [x,t] = splitter(d);

        % Bottom-left encoder
        u = FLC_encoder(x,t);
        pi_3 = randperm(length(u));
        v = u(pi_3);
        switch n_2
            case 2
                w = CC2_encoder(v);
            case 3
                w = CC3_encoder(v);
        end
        
        xhat = x;

        
        
        
        wtildea = generate_awgn_llrs(w, IAs(IA_index));
        
        vtildea = zeros(size(v)); 

        % Iteration between FLC and CC decoders
        for bottom_left_it = 1:max_bottom_left_it
            % Bottom-left decoder
            switch n_2
                case 2
                    [vtildee, wtildee] = CC2_decoder_bcjr(vtildea, wtildea);
                case 3
                    [vtildee, wtildee] = CC3_decoder_bcjr(vtildea, wtildea);
            end
            utildea = zeros(size(u));
            utildea(pi_3) = vtildee;
            [utildee, that] = FLC_decoder(utildea, xhat, P_t_given_x_vector);
            vtildea = utildee(pi_3);
        end
        
        
        
        
        
        % Measure the mutual information of the extrinsic LLRs
        IEs_hist(frame_index) = measure_mutual_information_histogram(wtildee, w);
        IEs_av(frame_index) = measure_mutual_information_averaging(wtildee);
        
    end
    
    % Determine the mean and standard deviation of the extrinisic mutual informations
    IEs_hist_mean(IA_index) = mean(IEs_hist);
    IEs_hist_std(IA_index) = std(IEs_hist);
    IEs_av_mean(IA_index) = mean(IEs_av);
    IEs_av_std(IA_index) = std(IEs_av);
end

% Measure the area beneath the inverted EXIT function
area_hist = 1-sum((IEs_hist_mean(2:end)+IEs_hist_mean(1:end-1)).*(IAs(2:end)-IAs(1:end-1)))/2;
area_av = 1-sum((IEs_av_mean(2:end)+IEs_av_mean(1:end-1)).*(IAs(2:end)-IAs(1:end-1)))/2;



% Plot the inverted EXIT function
figure
xlim([0 1]);
ylim([0 1]);
axis square
hold on
plot(IEs_hist_mean,IAs,'r-');
plot((IEs_hist_mean+IEs_hist_std),IAs,'r--');
plot((IEs_hist_mean-IEs_hist_std),IAs,'r--');
plot(IEs_av_mean,IAs,'b-');
plot((IEs_av_mean+IEs_av_std),IAs,'b--');
plot((IEs_av_mean-IEs_av_std),IAs,'b--');
xlabel('I_e');
ylabel('I_a');
title(['bottom-left p_1 = ',num2str(p_1),' x_{max} = ',num2str(x_max),' n_2 = ',num2str(n_2),' maxit = ',num2str(max_bottom_left_it)]);

% Display the area beneath the inverted EXIT function
annotation('textbox','String',{['Measured hist area frac = ', num2str(area_hist)]},'LineStyle','none','Position',[0.5 0.25 0.5 0.1]);
annotation('textbox','String',{['Measured av area frac = ', num2str(area_av)]},'LineStyle','none','Position',[0.5 0.20 0.5 0.1]);
annotation('textbox','String',{['Predicted area = ', num2str(A_o_2)]},'LineStyle','none','Position',[0.5 0.15 0.5 0.1]);
annotation('textbox','String',{['Coding rate = ', num2str(R_o_2)]},'LineStyle','none','Position',[0.5 0.10 0.5 0.1]);

% Save the results in a file
results = [IAs', IEs_hist_mean', (IEs_hist_mean'+IEs_hist_std'), (IEs_hist_mean'-IEs_hist_std'), IEs_av_mean', (IEs_av_mean'+IEs_av_std'), (IEs_av_mean'-IEs_av_std')];
filename = ['results_bottom_left_exit','.txt'];
save(filename, 'results', '-ASCII', '-DOUBLE', '-TABS');



