% Script for drawing the BER plot of the LTE turbo code, using either the BCJR algorithm or the fully-parallel algorithm.
% Copyright (C) 2013  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your
% option) any later version.

% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.



% x_max - Max symbol value for which FLC decoder is invoked
% p_1 - Probability of symbol values having the value of 1
% R_i_1 - Coding rate of top-right punctured URC code
% R_i_2 - Coding rate of bottom-right punctured URC code
% r_URC - Number of states to use in URC code - 2, 4 or 8
% n_2 - Number of encoded bits to use in CC code - 2 or 3
% max_top_it - Maximum number of iterations between UEC and URC decoders
% max_bottom_it - Maximum number of iterations between FLC-CC and URC decoders
% target_frame_errors Choose how many frame errors must be observed before plotting - 10 will give a fast simulation, 100 will give smooth plots
% max_symbols_to_simulate Choose the maximum number of symbols that will be simulated for each SNR value - 1e5 will give a fast simulation, 1e9 will give plots that reach lower BERs
function plot_ser(p_1, x_max, n_2, r_URC, R_i_1, R_i_2, SNR_start, SNR_delta, SNR_end, max_top_it, max_bottom_it, target_frame_errors, max_symbols_to_simulate)

if nargin == 0
    p_1 = 0.9; % Probability of symbol values having the value of 1
    x_max = 4; % Max symbol value for which FLC decoder is invoked
    n_2 = 2; % Number of encoded bits to use in CC code - 2 or 3
    r_URC = 2; % Number of states to use in URC code - 2, 4 or 8
    R_i_1 = 1.0578; % Coding rate of top-right punctured URC code
    R_i_2 = 1.0578; % Coding rate of bottom-right punctured URC code
    SNR_start = -5;
    SNR_delta = 0.5;
    SNR_end = 10;
    max_top_it = 100; % Maximum number of iterations between UEC and URC decoders
    max_bottom_it = 100; % Maximum number of iterations between FLC-CC and URC decoders
    target_frame_errors = 100;
    max_symbols_to_simulate = 1e9;
elseif isdeployed
     p_1 = str2double(p_1);
     x_max = str2double(x_max);
     n_2 = str2double(n_2);
     r_URC = str2double(r_URC);
     R_i_1 = str2double(R_i_1);
     R_i_2 = str2double(R_i_2);
     SNR_start = str2double(SNR_start);
     SNR_delta = str2double(SNR_delta);
     SNR_end = str2double(SNR_end);
     max_top_it = str2double(max_top_it);
     max_bottom_it = str2double(max_bottom_it);
     target_frame_errors = str2double(target_frame_errors);
     max_symbols_to_simulate = str2double(max_symbols_to_simulate);
end


global approx_maxstar;





% Select simulation parameters
C = [0 1; 1 1]; % UEC codebook
a = 20000; % Source symbol vector length
%max_bottom_left_it = 1; % Maximum number of iterations between FLC and CC decoders
approx_maxstar = 0; % Use the Max-Log-MAP algorithm?
chances = 3;

% Calculate some dependent parameters
r_1 = 2*size(C,1);
s = zeta_p1_to_s(p_1);
l_1 = get_l_1(s);
Px_vector = get_Px_vector(r_1,s);
P_t_given_x_vector = get_P_t_given_x_vector(x_max,s);


tic


% Choose a file to save the results into.
filename = ['results_ser_',num2str(p_1),'_',num2str(x_max),'_',num2str(n_2),'_',num2str(r_URC),'_',num2str(R_i_1),'_',num2str(R_i_2),'_',num2str(SNR_start),'_',num2str(SNR_delta),'_',num2str(SNR_end),'_',num2str(max_top_it),'_',num2str(max_bottom_it),'_',num2str(target_frame_errors),'_',num2str(max_symbols_to_simulate),'.mat'];


% Choose parameters for the SER simulation
SNRs = (SNR_start:SNR_delta:SNR_end)'; % Choose which Eb/N0 values (in dB) to consider for the Rayleigh fading channel


% Create a figure to plot the results.
figure
axes1 = axes('YScale','log');
ylabel('SER');
xlabel('SNR (in dB)');
xlim([SNRs(1), SNRs(end)]);
ylim([1e-6,1]);
hold on
plots = plot(nan,'Parent',axes1);



% Counters to store the number of bits and errors simulated so far
symbol_counts=zeros(length(SNRs),1);
frame_error_counts=zeros(length(SNRs),1);
symbol_error_counts=zeros(length(SNRs),1);

% Loop over the Eb/N0 values
for SNR_index = 1:length(SNRs)
    
    % Convert from Eb/N0 (in dB) to SNR (in dB)
    SNR = SNRs(SNR_index);
    
    % Convert from SNR (in dB) to noise power spectral density
    N0 = 1/(10^(SNR/10));
    
    % Continue the simulation until enough frame errors or bits have been simulated
    while frame_error_counts(SNR_index) < target_frame_errors && symbol_counts(SNR_index) < max_symbols_to_simulate

        % Generate random source
        d = generate_zeta_symbols(a, s);

        % Splitter
        [x,t] = splitter(d);

        % Unary encoder
        y = unary_encoder(x);

        % Top-left encoder
        z = UEC_trellis_encoder(y,C);

        % Interleaver 1
        z2 = reshape(z,[1,numel(z)]);
        pi_1 = randperm(length(z2));
        m = z2(pi_1);

        % Top-right encoder
        switch r_URC
            case 2
                n = URC2_encoder(m);
            case 4
                n = URC4_encoder(m);
            case 8
                n = URC8_encoder(m);
        end

        % Interleaver 2, puncturer, modulator, channel, demodulator, deinterleaver and depuncturer
        pi_2 = randperm(length(n));
        o = n(pi_2);
        
        length_o2 = 2*round(length(o)/R_i_1/2);
        if length_o2 > length(o)
            length_o2 = length_o2 - 2;
        end
        
        o2 = o(1:length_o2);
        o3 = reshape(o2,[2,length(o2)/2]);
        tx = modulate(o3);
        channel = sqrt(1/2) * (randn(size(tx)) + 1i*randn(size(tx)));
        rx = channel.*tx + sqrt(N0/2) * (randn(size(tx)) + 1i*randn(size(tx)));
        o3tilde = soft_demodulate(rx, channel, N0);
        o2tilde = reshape(o3tilde,size(o2));
        otilde = [o2tilde,zeros(1,length(o) - length(o2))];
        ntilde = zeros(size(n));
        ntilde(pi_2) = otilde;

        chances_taken = 0;
        best_MI = -1;
        
        % Iteration between UEC and URC decoders
        mtildea = zeros(size(m));
        for top_it = 1:max_top_it

            % Top-right decoder
            switch r_URC
                case 2
                    mtildee = URC2_decoder_bcjr(mtildea,ntilde);
                case 4
                    mtildee = URC4_decoder_bcjr(mtildea,ntilde);
                case 8
                    mtildee = URC8_decoder_bcjr(mtildea,ntilde);
            end

            % Deinterleaver 1
            z2tildea = zeros(size(z2));
            z2tildea(pi_1) = mtildee;
            ztildea = reshape(z2tildea,size(z));

            % Top-left decoder
            [ztildee, ytildep] = UEC_trellis_decoder(ztildea,C,Px_vector,l_1,a);
            

            MI = measure_mutual_information_averaging(ytildep);
            if MI > best_MI
                best_MI = MI;
                
                % Check if decoding has been successful
                xhat = unary_decoder_soft(ytildep, a);    
                if isequal(xhat,x)
                    break;
                end
            else
                chances_taken = chances_taken + 1;
                if chances_taken == chances
                    break;
                end
            end
                
            
            % Interleaver 1
            z2tildee = reshape(ztildee,[1,numel(z)]);
            mtildea = z2tildee(pi_1);

        end

        % Bottom-left encoder
        u = FLC_encoder(x,t);
        pi_3 = randperm(length(u));
        v = u(pi_3);
        switch n_2
            case 2
                w = CC2_encoder(v);
            case 3
                w = CC3_encoder(v);
        end

        % Interleaver 4
        w2 = reshape(w,[1,numel(w)]);
        pi_4 = randperm(length(w2));
        p = w2(pi_4);

        % Bottom-right encoder
        switch r_URC
            case 2
                q = URC2_encoder(p);
            case 4
                q = URC4_encoder(p);
            case 8
                q = URC8_encoder(p);
        end

        % Interleaver 5, puncturer, modulator, channel, demodulator, deinterleaver and depuncturer
        pi_5 = randperm(length(q));
        r = q(pi_5);
        
        length_r2 = 2*round(length(r)/R_i_2/2);
        if length_r2 > length(r)
            length_r2 = length_r2-2;
        end
        
        r2 = r(1:length_r2);
        r3 = reshape(r2,[2,length(r2)/2]);
        tx = modulate(r3);
        channel = sqrt(1/2) * (randn(size(tx)) + 1i*randn(size(tx)));
        rx = channel.*tx + sqrt(N0/2) * (randn(size(tx)) + 1i*randn(size(tx)));
        r3tilde = soft_demodulate(rx, channel, N0);
        r2tilde = reshape(r3tilde,size(r2));
        rtilde = [r2tilde,zeros(1,length(r) - length(r2))];
        qtilde = zeros(size(q));
        qtilde(pi_5) = rtilde;


        chances_taken = 0;
        best_MI = -1;
        
        
        % Iteration between FLC-CC and URC decoders
        ptildea = zeros(size(p));
        vtildea = zeros(size(v));
        for bottom_it = 1:max_bottom_it

            % Bottom-right decoder
            switch r_URC
                case 2
                    ptildee = URC2_decoder_bcjr(ptildea,qtilde);
                case 4
                    ptildee = URC4_decoder_bcjr(ptildea,qtilde);
                case 8
                    ptildee = URC8_decoder_bcjr(ptildea,qtilde);
            end




            % Deinterleaver 4
            w2tildea = zeros(size(w2));
            w2tildea(pi_4) = ptildee;
            wtildea = reshape(w2tildea,size(w));



%            % Iteration between FLC and CC decoders
%            for bottom_left_it = 1:max_bottom_left_it
                % Bottom-left decoder
                switch n_2
                    case 2
                        [vtildee, wtildee] = CC2_decoder_bcjr(vtildea, wtildea);
                    case 3
                        [vtildee, wtildee] = CC3_decoder_bcjr(vtildea, wtildea);
                end
                utildea = zeros(size(u));
                utildea(pi_3) = vtildee;
                [utildee, temp_that] = FLC_decoder(utildea, xhat, P_t_given_x_vector);
                vtildea = utildee(pi_3);

%                 % Check if decoding has been successful
%                 if isequal(that,t)
%                     break;
%                 end
%             end


            MI = measure_mutual_information_averaging(utildea+utildee);
            if MI > best_MI
                best_MI = MI;
                
                % Check if decoding has been successful
                that = temp_that;    
                if isequal(that,t)
                    break;
                end
            else
                chances_taken = chances_taken + 1;
                if chances_taken == chances
                    break;
                end
            end




            % Interleaver 4
            w2tildee = reshape(wtildee,[1,numel(w)]);
            ptildea = w2tildee(pi_4);

        end

        % Inverse splitter
        dhat = splitter_inv(xhat,that);
        
        symbol_errors = sum(dhat ~= d);
        
        
        
        if symbol_errors > 0
            % Accumulate the number of errors that have been simulated so far
            symbol_error_counts(SNR_index) = symbol_error_counts(SNR_index) + symbol_errors;
            frame_error_counts(SNR_index) = frame_error_counts(SNR_index) + 1;
        end
                
        % Accumulate the number of bits that have been simulated so far
        symbol_counts(SNR_index) = symbol_counts(SNR_index) + a;
        
        % Store the SNR and BERs in a matrix and display it.
        results = [SNRs(1:SNR_index,:), symbol_counts(1:SNR_index,:), symbol_error_counts(1:SNR_index,:), frame_error_counts(1:SNR_index,:)];
        
        if toc > 60
            % Save the results into a binary file. This avoids the loss of precision that is associated with ASCII files.
            save(filename, 'results', '-MAT');
            tic
        end
        
    end
    
    % Plot the BER vs Eb/N0 results
    set(plots,'XData',SNRs);
    set(plots,'YData',symbol_error_counts./symbol_counts);
    drawnow
    
end

