% Copyright (C) 2013  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your
% option) any later version.

% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.



% Select simulation parameters
C = [0 1; 1 1]; % UEC codebook
a = 20000; % Source symbol vector length
p_1 = 0.9; % Probability of symbol values having the value of 1
R_i_1 = 1.0578; % Coding rate of top-right punctured URC code
R_i_2 = 1.0578; % Coding rate of bottom-right punctured URC code
r_URC = 2; % Number of states to use in URC code - 2, 4 or 8
SNR = -0.3802; % Channel SNR in dB
approx_maxstar = 0; % Use the Max-Log-MAP algorithm?

% Calculate some dependent parameters
r_1 = 2*size(C,1);
s = zeta_p1_to_s(p_1);
N0 = 1/(10^(SNR/10));
R_o_1 = get_R_1(s,2)
R_o_2 = get_R_2(s,n_2)
eta = get_H_D(s)*2/(l_1*2/R_i_1 + l_2*n_2/R_i_2)
EbN0 = SNR - 10*log10(eta)





% Select the number of frames to simulate
frame_count = 10;

% Select the number of points to plot in the EXIT function
IA_count = 11;

% A priori mutual informations to consider
IAs = 0:1/(IA_count-1):1;

% Initialise results
IEs_hist_mean=zeros(size(IAs));
IEs_hist_std=zeros(size(IAs));
IEs_av_mean=zeros(size(IAs));
IEs_av_std=zeros(size(IAs));

% Consider each a priori mutual information
parfor IA_index = 1:length(IAs)
    IEs_hist = zeros(1,frame_count);
    IEs_av = zeros(1,frame_count);
    
    % For each frame
    for frame_index = 1:frame_count
        
        % Generate random source
        d = generate_zeta_symbols(a, s);

        % Splitter
        [x,t] = splitter(d);

        % Unary encoder
        y = unary_encoder(x);

        % Top-left encoder
        z = UEC_trellis_encoder(y,C);

        % Interleaver 1
        z2 = reshape(z,[1,numel(z)]);
        pi_1 = randperm(length(z2));
        m = z2(pi_1);

        % Top-right encoder
        switch r_URC
            case 2
                n = URC2_encoder(m);
            case 4
                n = URC4_encoder(m);
            case 8
                n = URC8_encoder(m);
        end

        % Interleaver 2, puncturer, modulator, channel, demodulator, deinterleaver and depuncturer
        pi_2 = randperm(length(n));
        o = n(pi_2);
        o2 = o(1:2*round(length(o)/R_i_1/2));
        o3 = reshape(o2,[2,length(o2)/2]);
        tx = modulate(o3);
        channel = sqrt(1/2) * (randn(size(tx)) + 1i*randn(size(tx)));
        rx = channel.*tx + sqrt(N0/2) * (randn(size(tx)) + 1i*randn(size(tx)));
        o3tilde = soft_demodulate(rx, channel, N0);
        o2tilde = reshape(o3tilde,size(o2));
        otilde = [o2tilde,zeros(1,length(o) - length(o2))];
        ntilde = zeros(size(n));
        ntilde(pi_2) = otilde;

        
        
        mtildea = generate_awgn_llrs(m, IAs(IA_index));
        
        % Top-right decoder
        switch r_URC
            case 2
                mtildee = URC2_decoder_bcjr(mtildea,ntilde);
            case 4
                mtildee = URC4_decoder_bcjr(mtildea,ntilde);
            case 8
                mtildee = URC8_decoder_bcjr(mtildea,ntilde);
        end
        
        
        % Measure the mutual information of the extrinsic LLRs
        IEs_hist(frame_index) = measure_mutual_information_histogram(mtildee, m);
        IEs_av(frame_index) = measure_mutual_information_averaging(mtildee);
        
    end
    
    % Determine the mean and standard deviation of the extrinisic mutual informations
    IEs_hist_mean(IA_index) = mean(IEs_hist);
    IEs_hist_std(IA_index) = std(IEs_hist);
    IEs_av_mean(IA_index) = mean(IEs_av);
    IEs_av_std(IA_index) = std(IEs_av);
end

% Measure the area beneath the EXIT function
area_hist = sum((IEs_hist_mean(2:end)+IEs_hist_mean(1:end-1)).*(IAs(2:end)-IAs(1:end-1)))/2;
area_av = sum((IEs_av_mean(2:end)+IEs_av_mean(1:end-1)).*(IAs(2:end)-IAs(1:end-1)))/2;



% Plot the EXIT function
figure
xlim([0 1]);
ylim([0 1]);
axis square
hold on
plot(IAs,IEs_hist_mean,'r-');
plot(IAs,(IEs_hist_mean+IEs_hist_std),'r--');
plot(IAs,(IEs_hist_mean-IEs_hist_std),'r--');
plot(IAs,IEs_av_mean,'b-');
plot(IAs,(IEs_av_mean+IEs_av_std),'b--');
plot(IAs,(IEs_av_mean-IEs_av_std),'b--');
xlabel('I_a');
ylabel('I_e');
title(['top-right p_1 = ',num2str(p_1),' r_1 = ',num2str(r_1),' r_{URC} = ',num2str(r_URC),' R^i_1 = ',num2str(R_i_1),' R^i_2 = ',num2str(R_i_2),' eta = ',num2str(eta),' E_b/N_0 = ',num2str(EbN0)]);

% Display the area beneath the inverted EXIT function
annotation('textbox','String',{['Measured hist area frac = ', num2str(area_hist)]},'LineStyle','none','Position',[0.5 0.25 0.5 0.1]);
annotation('textbox','String',{['Measured av area frac = ', num2str(area_av)]},'LineStyle','none','Position',[0.5 0.20 0.5 0.1]);

% Save the results in a file
results = [IAs', IEs_hist_mean', (IEs_hist_mean'+IEs_hist_std'), (IEs_hist_mean'-IEs_hist_std'), IEs_av_mean', (IEs_av_mean'+IEs_av_std'), (IEs_av_mean'-IEs_av_std')];
filename = ['results_top_right_exit','.txt'];
save(filename, 'results', '-ASCII', '-DOUBLE', '-TABS');



