% Script for drawing the BER plot of the WiMAX turbo code, using either the BCJR algorithm or the fully-parallel algorithm.
% Copyright (C) 2013  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it 
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your 
% option) any later version.

% This program is distributed in the hope that it will be useful, but 
% WITHOUT ANY WARRANTY; without even the implied warranty of 
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General 
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.

% Setup all variables
clear all
global approx_maxstar;

% Choose parameters for the turbo decoder
algorithm = 1; % Choose 0 for BCJR algorithm and 1 for fully-parallel algorithm
approx_maxstar = 0; % Choose 0 for Log-MAP and 1 for Max-Log-MAP versions of the BCJR and fully parallel algorithms
bits_per_frame = 2400; % Choose the frame length N

% Choose parameters for the BER simulation
EbN0s = 0:0.5:7; % Choose which Eb/N0 values (in dB) to consider for the Rayleigh fading channel
iterations_to_plot = [1,2,4,8,16,32,64,128]; % Choose the number of decoding iterations after which the BER is plotted
min_frame_errors_to_plot = 10; % Choose how many frame errors must be observed before plotting - 10 will give a fast simulation, 100 will give smooth plots
max_bits_to_simulate = 1e5; % Choose the maximum number of bits that will be simulated for each Eb/N0 value - 1e5 will give a fast simulation, 1e9 will give plots that reach lower BERs

% Initalise the interleaver
interleaver = get_WiMAX_interleaver(bits_per_frame);

% Calculate the number of bits per symbol
bits_per_symbol = 1/3;

% Create a figure to plot the results.
figure
axes1 = axes('YScale','log');
title(['N = ',num2str(bits_per_frame),'-bit WiMAX turbo decoder']);
ylabel('BER');
xlabel('E_b/N_0 (in dB)');
xlim([EbN0s(1), EbN0s(end)]);
ylim([1e-6,1]);
hold on
plots = zeros(size(iterations_to_plot));
for iteration_index = 1:length(iterations_to_plot)
    plots(iteration_index) = plot(nan,'Parent',axes1);
end

% Counters to store the number of bits and errors simulated so far
bit_counts=zeros(length(EbN0s),1);
frame_error_counts=zeros(length(EbN0s),length(iterations_to_plot));
bit_error_counts=zeros(length(EbN0s),length(iterations_to_plot));

% Loop over the Eb/N0 values
for EbN0_index = 1:length(EbN0s)
    
    % Convert from Eb/N0 (in dB) to SNR (in dB)
    SNR = EbN0s(EbN0_index) + 10*log10(bits_per_symbol);
    
    % Convert from SNR (in dB) to noise power spectral density
    N0 = 1/(10^(SNR/10));
    
    % Continue the simulation until enough frame errors or bits have been simulated
    while frame_error_counts(EbN0_index, end) < min_frame_errors_to_plot && bit_counts(EbN0_index) < max_bits_to_simulate
        
        % Initially only simulate the transmission of the systematic bits -
        % if these are recovered without error then there is no need to
        % simulate the transmission of the parity bits
        
        % Generate the random frames of bits
        a = round(rand(1,bits_per_frame));
        b = round(rand(1,bits_per_frame));
        
        % BPSK modulate the systematic bits
        a_tx = 2*a-1;
        b_tx = 2*b-1;
        
        % Generate some random Rayleigh fading channel coefficients
        h_a = sqrt(1/2)*(randn(size(a_tx))+1i*randn(size(a_tx)));
        h_b = sqrt(1/2)*(randn(size(b_tx))+1i*randn(size(b_tx)));
        
        % Generate some random AWGN
        n_a = sqrt(N0/2)*(randn(size(a_tx))+1i*randn(size(a_tx)));
        n_b = sqrt(N0/2)*(randn(size(b_tx))+1i*randn(size(b_tx)));
        
        % Simulate transmission over a Rayleigh fading channel
        a_rx = h_a.*a_tx + n_a;
        b_rx = h_b.*b_tx + n_b;
        
        % Use soft BPSK demodulation to obtain the systematic LLRs
        a_c = 4*real(a_rx.*conj(h_a))/N0;
        b_c = 4*real(b_rx.*conj(h_b))/N0;
        
        % Make a hard decision and see how many bit errors we have
        bit_errors = sum(([a_c,b_c] > 0) ~= [a,b]);
        
        % If we have any bit errors then we need to simulate the transmission of the parity bits
        if bit_errors > 0
            
            % Interleave the random frames of bits
            ab = reshape([a;b],1,2*bits_per_frame);
            cd = ab(interleaver);
            c = cd(1:2:end);
            d = cd(2:2:end);
            
            % Generate the parity bits
            [e,f] = convolutional_encoder(a,b);
            [g,h] = convolutional_encoder(c,d);
            
            % BPSK modulate the parity bits
            e_tx = 2*e-1;
            f_tx = 2*f-1;
            g_tx = 2*g-1;
            h_tx = 2*h-1;
            
            % Generate some random Rayleigh fading channel coefficients
            h_e = sqrt(1/2)*(randn(size(e_tx))+1i*randn(size(e_tx)));
            h_f = sqrt(1/2)*(randn(size(f_tx))+1i*randn(size(f_tx)));
            h_g = sqrt(1/2)*(randn(size(g_tx))+1i*randn(size(g_tx)));
            h_h = sqrt(1/2)*(randn(size(h_tx))+1i*randn(size(h_tx)));
            
            % Generate some random AWGN
            n_e = sqrt(N0/2)*(randn(size(e_tx))+1i*randn(size(e_tx)));
            n_f = sqrt(N0/2)*(randn(size(f_tx))+1i*randn(size(f_tx)));
            n_g = sqrt(N0/2)*(randn(size(g_tx))+1i*randn(size(g_tx)));
            n_h = sqrt(N0/2)*(randn(size(h_tx))+1i*randn(size(h_tx)));
            
            % Simulate transmission over a Rayleigh fading channel
            e_rx = h_e.*e_tx + n_e;
            f_rx = h_f.*f_tx + n_f;
            g_rx = h_g.*g_tx + n_g;
            h_rx = h_h.*h_tx + n_h;
            
            % Use soft BPSK demodulation to obtain the parity LLRs
            e_c = 4*real(e_rx.*conj(h_e))/N0;
            f_c = 4*real(f_rx.*conj(h_f))/N0;
            g_c = 4*real(g_rx.*conj(h_g))/N0;
            h_c = 4*real(h_rx.*conj(h_h))/N0;
            
            % Interleave the systematic LLRs
            ab_c = reshape([a_c;b_c],1,2*bits_per_frame);
            cd_c = ab_c(interleaver);
            c_c = cd_c(1:2:end);
            d_c = cd_c(2:2:end);
            
            % Initialise a priori and extrinsic LLRs to zero
            a_e = zeros(size(a));
            a_a = zeros(size(a));
            b_e = zeros(size(b));
            b_a = zeros(size(b));
            ab_e = zeros(size(ab));
            ab_a = zeros(size(ab));
            c_e = zeros(size(c));
            c_a = zeros(size(c));
            d_e = zeros(size(d));
            d_a = zeros(size(d));
            cd_e = zeros(size(cd));
            cd_a = zeros(size(cd));
            
            if algorithm % use the fully-parallel algorithm
                
                % Inialise extrinsic alphas and betas
                upper_extrinsic_betas=zeros(8,length(a_c)+1);
                upper_extrinsic_alphas=zeros(8,length(a_c)+1);
                
                lower_extrinsic_betas=zeros(8,length(a_c)+1);
                lower_extrinsic_alphas=zeros(8,length(a_c)+1);
                                
                % Perform iterative decoding
                for iteration_index = 1:iterations_to_plot(end)
                    
                    % Update a priori alphas and betas
                    upper_apriori_alphas = upper_extrinsic_alphas;
                    upper_apriori_alphas(:,1) = upper_extrinsic_alphas(:,end);
                    upper_apriori_betas = upper_extrinsic_betas;
                    upper_apriori_betas(:,end) = upper_extrinsic_betas(:,1);
                    lower_apriori_alphas = lower_extrinsic_alphas;
                    lower_apriori_alphas(:,1) = lower_extrinsic_alphas(:,end);
                    lower_apriori_betas = lower_extrinsic_betas;
                    lower_apriori_betas(:,end) = lower_extrinsic_betas(:,1);
                    
                    % Interleave the extrinsic LLRs
                    ab_e = reshape([a_e;b_e],1,2*bits_per_frame);
                    cd_a = ab_e(interleaver);
                    c_a = cd_a(1:2:end);
                    d_a = cd_a(2:2:end);
                                        
                    % Operate the upper block decoders having odd indices and the lower block decoders having even indices in parallel
                    parfor bit_index = 1:length(a_a)
                        if mod(bit_index,2)==1
                            [a_e(bit_index), b_e(bit_index), upper_extrinsic_betas(:,bit_index), upper_extrinsic_alphas(:,bit_index+1)] = block_decoder(a_a(bit_index)+a_c(bit_index), b_a(bit_index)+b_c(bit_index), upper_apriori_alphas(:,bit_index), upper_apriori_betas(:,bit_index+1), e_c(bit_index), f_c(bit_index));
                        else
                            [c_e(bit_index), d_e(bit_index), lower_extrinsic_betas(:,bit_index), lower_extrinsic_alphas(:,bit_index+1)] = block_decoder(c_a(bit_index)+c_c(bit_index), d_a(bit_index)+d_c(bit_index), lower_apriori_alphas(:,bit_index), lower_apriori_betas(:,bit_index+1), g_c(bit_index), h_c(bit_index));
                        end
                    end
                    
                    % Deinterleave the extrinsic LLRs
                    cd_e = reshape([c_e;d_e],1,2*bits_per_frame);
                    ab_a(interleaver) = cd_e;
                    a_a = ab_a(1:2:end);
                    b_a = ab_a(2:2:end);
                    
                    
                    
                    
                    
                    % Update a priori alphas and betas
                    upper_apriori_alphas = upper_extrinsic_alphas;
                    upper_apriori_alphas(:,1) = upper_extrinsic_alphas(:,end);
                    upper_apriori_betas = upper_extrinsic_betas;
                    upper_apriori_betas(:,end) = upper_extrinsic_betas(:,1);
                    lower_apriori_alphas = lower_extrinsic_alphas;
                    lower_apriori_alphas(:,1) = lower_extrinsic_alphas(:,end);
                    lower_apriori_betas = lower_extrinsic_betas;
                    lower_apriori_betas(:,end) = lower_extrinsic_betas(:,1);

                    % Interleave the extrinsic LLRs
                    ab_e = reshape([a_e;b_e],1,2*bits_per_frame);
                    cd_a = ab_e(interleaver);
                    c_a = cd_a(1:2:end);
                    d_a = cd_a(2:2:end);
                    
                    % Operate the upper block decoders having even indices and the lower block decoders having odd indices in parallel
                    parfor bit_index = 1:length(a_a)
                        if mod(bit_index,2)==0
                            [a_e(bit_index), b_e(bit_index), upper_extrinsic_betas(:,bit_index), upper_extrinsic_alphas(:,bit_index+1)] = block_decoder(a_a(bit_index)+a_c(bit_index), b_a(bit_index)+b_c(bit_index), upper_apriori_alphas(:,bit_index), upper_apriori_betas(:,bit_index+1), e_c(bit_index), f_c(bit_index));
                        else
                            [c_e(bit_index), d_e(bit_index), lower_extrinsic_betas(:,bit_index), lower_extrinsic_alphas(:,bit_index+1)] = block_decoder(c_a(bit_index)+c_c(bit_index), d_a(bit_index)+d_c(bit_index), lower_apriori_alphas(:,bit_index), lower_apriori_betas(:,bit_index+1), g_c(bit_index), h_c(bit_index));
                        end
                    end
                    
                    % Deinterleave the extrinsic LLRs
                    cd_e = reshape([c_e;d_e],1,2*bits_per_frame);
                    ab_a(interleaver) = cd_e;
                    a_a = ab_a(1:2:end);
                    b_a = ab_a(2:2:end);
                    
                    
                    
                    
                    
                    % Obtain the a posteriori LLRs
                    a_p = a_a + a_c + a_e;
                    b_p = b_a + b_c + b_e;
                    
                    % Make a hard decision and see how many bit errors we have
                    bit_errors = sum(([a_p,b_p] > 0) ~= [a,b]);                   
                    
                    if bit_errors > 0
                        % Accumulate the number of errors that have been simulated so far
                        bit_error_counts(EbN0_index, iterations_to_plot == iteration_index) = bit_error_counts(EbN0_index, iterations_to_plot == iteration_index) + bit_errors;
                        frame_error_counts(EbN0_index, iterations_to_plot == iteration_index) = frame_error_counts(EbN0_index, iterations_to_plot == iteration_index) + 1;
                    else
                        % No need to continue iterating if all the errors have been removed - it is assumed that a CRC is used to detect that this has happened
                        break;
                    end
                end
                
            else % use the BCJR algorithm              
                
                alpha_0_upper = 0;
                beta_N_upper = 0;
                alpha_0_lower = 0;
                beta_N_lower = 0;
                
                % Perform iterative decoding
                for iteration_index = 1:iterations_to_plot(end)
                
                
                    % Perform the BCJR algorithm
                    [a_e, b_e, alpha_0_upper, beta_N_upper] = bcjr_decoder(a_a+a_c, b_a+b_c, e_c, f_c, alpha_0_upper, beta_N_upper);
                                
                    % Interleave the extrinsic LLRs
                    ab_e = reshape([a_e;b_e],1,2*bits_per_frame);
                    cd_a = ab_e(interleaver);
                    c_a = cd_a(1:2:end);
                    d_a = cd_a(2:2:end);
                                
                    % Perform the BCJR algorithm
                    [c_e, d_e, alpha_0_lower, beta_N_lower] = bcjr_decoder(c_a+c_c, d_a+d_c, g_c, h_c, alpha_0_lower, beta_N_lower);
                    
                    % Deinterleave the extrinsic LLRs
                    cd_e = reshape([c_e;d_e],1,2*bits_per_frame);
                    ab_a(interleaver) = cd_e;
                    a_a = ab_a(1:2:end);
                    b_a = ab_a(2:2:end);
               
                    % Obtain the a posteriori LLRs
                    a_p = a_a + a_c + a_e;
                    b_p = b_a + b_c + b_e;
                    
                    % Make a hard decision and see how many bit errors we have
                    bit_errors = sum(([a_p,b_p] > 0) ~= [a,b]);                   
                    
                    if bit_errors > 0
                        % Accumulate the number of errors that have been simulated so far
                        bit_error_counts(EbN0_index, iterations_to_plot == iteration_index) = bit_error_counts(EbN0_index, iterations_to_plot == iteration_index) + bit_errors;
                        frame_error_counts(EbN0_index, iterations_to_plot == iteration_index) = frame_error_counts(EbN0_index, iterations_to_plot == iteration_index) + 1;
                    else
                        % No need to continue iterating if all the errors have been removed - it is assumed that a CRC is used to detect that this has happened
                        break;
                    end
                end
            end
        end
        
        % Accumulate the number of bits that have been simulated so far
        bit_counts(EbN0_index) = bit_counts(EbN0_index) + length(a) + length(b);
        
    end
    
    % Plot the BER vs Eb/N0 results
    for iteration_index = 1:length(iterations_to_plot)
        set(plots(iteration_index),'XData',EbN0s(frame_error_counts(:,iteration_index)>=min_frame_errors_to_plot));
        set(plots(iteration_index),'YData',bit_error_counts(frame_error_counts(:,iteration_index)>=min_frame_errors_to_plot,iteration_index)./bit_counts(frame_error_counts(:,iteration_index)>=min_frame_errors_to_plot));
    end
    drawnow
    
end

