% Encoder function for a tail-biting unity-rate recursive convolutional code
% having 3 memory elements. This is as used in the WiMAX turbo code, as specified in IEEE Std 802.16-2012.
% Copyright (C) 2013  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it 
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your 
% option) any later version.

% This program is distributed in the hope that it will be useful, but 
% WITHOUT ANY WARRANTY; without even the implied warranty of 
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General 
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.

% uncoded_bits_A is a 1xN vector of uncoded bits
% uncoded_bits_B is a 1xN vector of uncoded bits
% encoded_bits_A is a 1xN vector of encoded bits
% encoded_bits_B is a 1xN vector of encoded bits
function [encoded_bits_A, encoded_bits_B] = convolutional_encoder(uncoded_bits_A, uncoded_bits_B)

if(length(uncoded_bits_A) ~= length(uncoded_bits_B))
    error('Bit sequences must have the same length');
end

% Initialise our output bit vectors
encoded_bits_A = zeros(1,length(uncoded_bits_A));
encoded_bits_B = zeros(1,length(uncoded_bits_B));

for s1_start = 0:1
   for s2_start = 0:1
       for s3_start = 0:1
            
            % We start in the start state
            s1 = s1_start;
            s2 = s2_start;
            s3 = s3_start;
            
            % Encode the uncoded bit sequence
            for bit_index = 1:length(uncoded_bits_A)
                
                % Determine the next state
                s1_plus = mod(uncoded_bits_A(bit_index)+uncoded_bits_B(bit_index)+s1+s3, 2); % This uses the feedback polynomial
                s2_plus = mod(s1+uncoded_bits_B(bit_index), 2);
                s3_plus = mod(s2+uncoded_bits_B(bit_index), 2);
                
                % Determine the encoded bit
                encoded_bits_A(bit_index) = mod(s1_plus+s2+s3, 2); % This uses the generator polynomial
                encoded_bits_B(bit_index) = mod(s1_plus+s3, 2); % This uses the generator polynomial
                
                % Enter the next state
                s1 = s1_plus;
                s2 = s2_plus;
                s3 = s3_plus;
            end
            
            if s1 == s1_start && s2 == s2_start && s3 == s3_start
                return
            end
            
        end
    end
end


end