% BCJR algorithm for a tail-biting unity-rate recursive convolutional code
% having 3 memory elements. This is as used in the WiMAX turbo code, as specified in IEEE Std 802.16-2012.
% Copyright (C) 2013  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it 
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your 
% option) any later version.

% This program is distributed in the hope that it will be useful, but 
% WITHOUT ANY WARRANTY; without even the implied warranty of 
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General 
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.

% apriori_uncoded_A is a 1xN vector of a priori uncoded LLRs
% apriori_uncoded_B is a 1xN vector of a priori uncoded LLRs
% apriori_encoded_A is a 1xN vector of a priori encoded LLRs
% apriori_encoded_B is a 1xN vector of a priori encoded LLRs
% alpha_0 is a vector of alphas
% beta_N is a vector of betas
% extrinsic_uncoded_A is a 1xN vector of extrinsic encoded LLRs
% extrinsic_uncoded_B is a 1xN vector of extrinsic encoded LLRs
% alpha_N is a vector of alphas which can be used for alpha_0 in the next iteration
% beta_0 is a vector of betas which can be used for beta_N in the next iteration
function [extrinsic_uncoded_A, extrinsic_uncoded_B, alpha_N, beta_0] = bcjr_decoder(apriori_uncoded_A, apriori_uncoded_B, apriori_encoded_A, apriori_encoded_B, alpha_0, beta_N)


    if(length(apriori_uncoded_A) ~= length(apriori_encoded_A) || length(apriori_uncoded_B) ~= length(apriori_encoded_B) || length(apriori_uncoded_A) ~= length(apriori_uncoded_B))
        error('LLR sequences must have the same length');
    end

    % Matrix to describe the trellis
    % Each row describes one transition in the trellis
    % Each state is allocated an index 1,2,3,... Note that this list starts 
    % from 1 rather than 0.
    %               FromState,  ToState,    UncodedA,   UncodedB,   EncodedA,   EncodedB
    transitions =  [1,          1,          0,          0,          0,          0; 
                    2,          5,          0,          0,          0,          0; 
                    3,          2,          0,          0,          1,          0; 
                    4,          6,          0,          0,          1,          0; 
                    5,          7,          0,          0,          1,          1; 
                    6,          3,          0,          0,          1,          1; 
                    7,          8,          0,          0,          0,          1; 
                    8,          4,          0,          0,          0,          1; 
                    1,          8,          0,          1,          1,          1; 
                    2,          4,          0,          1,          1,          1; 
                    3,          7,          0,          1,          0,          1; 
                    4,          3,          0,          1,          0,          1; 
                    5,          2,          0,          1,          0,          0; 
                    6,          6,          0,          1,          0,          0; 
                    7,          1,          0,          1,          1,          0; 
                    8,          5,          0,          1,          1,          0;
                    1,          5,          1,          0,          1,          1; 
                    2,          1,          1,          0,          1,          1; 
                    3,          6,          1,          0,          0,          1; 
                    4,          2,          1,          0,          0,          1; 
                    5,          3,          1,          0,          0,          0; 
                    6,          7,          1,          0,          0,          0; 
                    7,          4,          1,          0,          1,          0; 
                    8,          8,          1,          0,          1,          0; 
                    1,          4,          1,          1,          0,          0; 
                    2,          8,          1,          1,          0,          0; 
                    3,          3,          1,          1,          1,          0; 
                    4,          7,          1,          1,          1,          0; 
                    5,          6,          1,          1,          1,          1; 
                    6,          2,          1,          1,          1,          1; 
                    7,          5,          1,          1,          0,          1; 
                    8,          1,          1,          1,          0,          1];
               
    % Find the largest state index in the transitions matrix           
    % In this example, we have eight states since the code has three memory elements
    state_count = max(max(transitions(:,1)),max(transitions(:,2)));

    if ~exist('alpha_0','var') || ~exist('beta_N','var') || ~isequal(size(alpha_0),[state_count,1]) || ~isequal(size(beta_N),[state_count,1])
        alpha_0 = zeros(state_count,1);
        beta_N = zeros(state_count,1);
    end

    
    % Calculate the possible a priori transition log-probabilities
    a = apriori_uncoded_A;
    b = apriori_uncoded_B;
    c = apriori_encoded_A;
    d = apriori_encoded_B;
    ab = a+b; % 1 addition
    ac = a+c; % 1 addition
    ad = a+d; % 1 addition
    bc = b+c; % 1 addition
    bd = b+d; % 1 addition
    cd = c+d; % 1 addition
    abc = ab+c; % 1 addition
    abd = ab+d; % 1 addition
    acd = ac+d; % 1 addition
    bcd = bc+d; % 1 addition
    abcd = ab+cd; % 1 addition
    
    
    % Calculate the a priori transition log-probabilities
    gammas = zeros(size(transitions,1),length(apriori_uncoded_A));
    gammas(transitions(:,3)==0 & transitions(:,4)==0 & transitions(:,5)==0 & transitions(:,6)==1,:) = repmat(d   , sum(transitions(:,3)==0 & transitions(:,4)==0 & transitions(:,5)==0 & transitions(:,6)==1),1);
    gammas(transitions(:,3)==0 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==0,:) = repmat(c   , sum(transitions(:,3)==0 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==0),1);
    gammas(transitions(:,3)==0 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==1,:) = repmat(cd  , sum(transitions(:,3)==0 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==1),1);
    gammas(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==0,:) = repmat(b   , sum(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==0),1);
    gammas(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==1,:) = repmat(bd  , sum(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==1),1);
    gammas(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==0,:) = repmat(bc  , sum(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==0),1);
    gammas(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==1,:) = repmat(bcd , sum(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==1),1);
    gammas(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==0 & transitions(:,6)==0,:) = repmat(a   , sum(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==0 & transitions(:,6)==0),1);
    gammas(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==0 & transitions(:,6)==1,:) = repmat(ad  , sum(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==0 & transitions(:,6)==1),1);
    gammas(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==0,:) = repmat(ac  , sum(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==0),1);
    gammas(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==1,:) = repmat(acd , sum(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==1),1);
    gammas(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==0,:) = repmat(ab  , sum(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==0),1);
    gammas(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==1,:) = repmat(abd , sum(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==1),1);
    gammas(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==0,:) = repmat(abc , sum(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==0),1);
    gammas(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==1,:) = repmat(abcd, sum(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==1),1);

    % Recursion to calculate forward state log-probabilities
    alphas=zeros(state_count,length(apriori_uncoded_A)+1);
    alphas(:,1)=alpha_0;
    for bit_index = 2:length(apriori_uncoded_A)+1        
        temp = alphas(transitions(:,1),bit_index-1)+gammas(:,bit_index-1); % 30 additions since two gammas have a value of 0
        for state_index = 1:state_count
            alphas(state_index,bit_index) = maxstar(temp(transitions(:,2) == state_index)); % 3 maxstar per iteration of for loop = 24 maxstars
        end
    end
    alpha_N = alphas(:,end);
    
    % Recursion to calculate backward state log-probabilities
    betas=zeros(state_count,length(apriori_uncoded_A)+1);
    betas(:,end)=beta_N;
    for bit_index = length(apriori_uncoded_A):-1:1
        temp = betas(transitions(:,2),bit_index+1)+gammas(:,bit_index); % 30 additions since two gammas have a value of 0
        for state_index = 1:state_count
            betas(state_index,bit_index) = maxstar(temp(transitions(:,1) == state_index)); % 3 maxstar per iteration of for loop = 24 maxstars
        end
    end
    beta_0 = betas(:,1);

    % Calculate a posteriori transition log-probabilities
    deltas = alphas(transitions(:,1),1:end-1) + betas(transitions(:,2),2:end) + gammas; % 62 additions since two gammas have a value of 0
        
    % Calculate the uncoded extrinsic LLRs
    log_p00=maxstar(deltas(transitions(:,3) == 0 & transitions(:,4) == 0,:)); % 7 maxstars
    log_p01=maxstar(deltas(transitions(:,3) == 0 & transitions(:,4) == 1,:)); % 7 maxstars
    log_p10=maxstar(deltas(transitions(:,3) == 1 & transitions(:,4) == 0,:)); % 7 maxstars
    log_p11=maxstar(deltas(transitions(:,3) == 1 & transitions(:,4) == 1,:)); % 7 maxstars
    extrinsic_uncoded_A = maxstar([log_p10;log_p11])-maxstar([log_p00;log_p01])-apriori_uncoded_A; % 2 additions and 2 maxstars
    extrinsic_uncoded_B = maxstar([log_p01;log_p11])-maxstar([log_p00;log_p10])-apriori_uncoded_B; % 2 additions and 2 maxstars
  