% Fully-parallel algorithm for a unity-rate recursive convolutional code
% having 3 memory elements, a generator polynomial of [1,1,0,1] and a feedback
% polynomial of [1,0,1,1]. This is as used in the LTE turbo code, as specified in ETSI TS 136 212.
% Copyright (C) 2013  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it 
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your 
% option) any later version.

% This program is distributed in the hope that it will be useful, but 
% WITHOUT ANY WARRANTY; without even the implied warranty of 
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General 
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.

% apriori_uncoded is an a priori uncoded LLR
% apriori_encoded is an a priori encoded LLR
% apriori_alphas is a vector of 8 a priori forward state log-probabilities
% apriori_betas is a vector of 8 a priori backward state log-probabilities
% extrinsic_uncoded is an extrinsic encoded LLR
% extrinsic_betas is a vector of 8 extrinsic backward state log-probabilities
% extrinsic_alphas is a vector of 8 extrinsic forward state log-probabilities
function [extrinsic_uncoded_A, extrinsic_uncoded_B, extrinsic_betas, extrinsic_alphas] = block_decoder(apriori_uncoded_A, apriori_uncoded_B, apriori_alphas, apriori_betas, apriori_encoded_A, apriori_encoded_B)

    % Matrix to describe the trellis
    % Each row describes one transition in the trellis
    % Each state is allocated an index 1,2,3,... Note that this list starts 
    % from 1 rather than 0.
    %               FromState,  ToState,    UncodedA,   UncodedB,   EncodedA,   EncodedB
    transitions =  [1,          1,          0,          0,          0,          0; 
                    2,          5,          0,          0,          0,          0; 
                    3,          2,          0,          0,          1,          0; 
                    4,          6,          0,          0,          1,          0; 
                    5,          7,          0,          0,          1,          1; 
                    6,          3,          0,          0,          1,          1; 
                    7,          8,          0,          0,          0,          1; 
                    8,          4,          0,          0,          0,          1; 
                    1,          8,          0,          1,          1,          1; 
                    2,          4,          0,          1,          1,          1; 
                    3,          7,          0,          1,          0,          1; 
                    4,          3,          0,          1,          0,          1; 
                    5,          2,          0,          1,          0,          0; 
                    6,          6,          0,          1,          0,          0; 
                    7,          1,          0,          1,          1,          0; 
                    8,          5,          0,          1,          1,          0;
                    1,          5,          1,          0,          1,          1; 
                    2,          1,          1,          0,          1,          1; 
                    3,          6,          1,          0,          0,          1; 
                    4,          2,          1,          0,          0,          1; 
                    5,          3,          1,          0,          0,          0; 
                    6,          7,          1,          0,          0,          0; 
                    7,          4,          1,          0,          1,          0; 
                    8,          8,          1,          0,          1,          0; 
                    1,          4,          1,          1,          0,          0; 
                    2,          8,          1,          1,          0,          0; 
                    3,          3,          1,          1,          1,          0; 
                    4,          7,          1,          1,          1,          0; 
                    5,          6,          1,          1,          1,          1; 
                    6,          2,          1,          1,          1,          1; 
                    7,          5,          1,          1,          0,          1; 
                    8,          1,          1,          1,          0,          1];

    % Find the largest state index in the transitions matrix
    state_count = max(max(transitions(:,1)),max(transitions(:,2)));

    if length(apriori_alphas) ~= state_count
        error('The length of apriori_alphas is not correct');
    end

    if length(apriori_betas) ~= state_count
        error('The length of apriori_betas is not correct');
    end

    % Calculate the possible a priori transition log-probabilities
    a = apriori_uncoded_A;
    b = apriori_uncoded_B;
    c = apriori_encoded_A;
    d = apriori_encoded_B;
    ab = a+b; % 1 addition
    ac = a+c; % 1 addition
    ad = a+d; % 1 addition
    bc = b+c; % 1 addition
    bd = b+d; % 1 addition
    cd = c+d; % 1 addition
    abc = ab+c; % 1 addition
    abd = ab+d; % 1 addition
    acd = ac+d; % 1 addition
    bcd = bc+d; % 1 addition
    abcd = ab+cd; % 1 addition

    % Calculate a posteriori transition log-probabilities
    deltas = apriori_alphas(transitions(:,1)) + apriori_betas(transitions(:,2)); % 32 additions
    deltas(transitions(:,3)==0 & transitions(:,4)==0 & transitions(:,5)==0 & transitions(:,6)==1) = deltas(transitions(:,3)==0 & transitions(:,4)==0 & transitions(:,5)==0 & transitions(:,6)==1) + d; % 2 additions
    deltas(transitions(:,3)==0 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==0) = deltas(transitions(:,3)==0 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==0) + c; % 2 additions
    deltas(transitions(:,3)==0 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==1) = deltas(transitions(:,3)==0 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==1) + cd; % 2 additions
    deltas(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==0) = deltas(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==0) + b; % 2 additions
    deltas(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==1) = deltas(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==1) + bd; % 2 additions
    deltas(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==0) = deltas(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==0) + bc; % 2 additions
    deltas(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==1) = deltas(transitions(:,3)==0 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==1) + bcd; % 2 additions
    deltas(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==0 & transitions(:,6)==0) = deltas(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==0 & transitions(:,6)==0) + a; % 2 additions
    deltas(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==0 & transitions(:,6)==1) = deltas(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==0 & transitions(:,6)==1) + ad; % 2 additions
    deltas(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==0) = deltas(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==0) + ac; % 2 additions
    deltas(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==1) = deltas(transitions(:,3)==1 & transitions(:,4)==0 & transitions(:,5)==1 & transitions(:,6)==1) + acd; % 2 additions
    deltas(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==0) = deltas(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==0) + ab; % 2 additions
    deltas(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==1) = deltas(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==0 & transitions(:,6)==1) + abd; % 2 additions
    deltas(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==0) = deltas(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==0) + abc; % 2 additions
    deltas(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==1) = deltas(transitions(:,3)==1 & transitions(:,4)==1 & transitions(:,5)==1 & transitions(:,6)==1) + abcd; % 2 additions

    % Calculate the uncoded extrinsic LLRs
    log_p00=maxstar(deltas(transitions(:,3) == 0 & transitions(:,4) == 0,:)); % 7 maxstars
    log_p01=maxstar(deltas(transitions(:,3) == 0 & transitions(:,4) == 1,:)); % 7 maxstars
    log_p10=maxstar(deltas(transitions(:,3) == 1 & transitions(:,4) == 0,:)); % 7 maxstars
    log_p11=maxstar(deltas(transitions(:,3) == 1 & transitions(:,4) == 1,:)); % 7 maxstars
    extrinsic_uncoded_A = maxstar([log_p10;log_p11])-maxstar([log_p00;log_p01])-apriori_uncoded_A; % 2 additions and 2 maxstars
    extrinsic_uncoded_B = maxstar([log_p01;log_p11])-maxstar([log_p00;log_p10])-apriori_uncoded_B; % 2 additions and 2 maxstars

    % Calculate the extrinsic backward state log-probabilities
    log_p1 = zeros(size(apriori_alphas));
    for state_index = 1:state_count
        log_p1(state_index) = maxstar(deltas(transitions(:,1) == state_index)); % 1 maxstar per iteration of for loop = 8 maxstars
    end
    extrinsic_betas = log_p1 - apriori_alphas; % 8 additions

    % Calculate the extrinsic forward state log-probabilities
    log_p1 = zeros(size(apriori_betas));
    for state_index = 1:state_count
        log_p1(state_index) = maxstar(deltas(transitions(:,2) == state_index)); % 1 maxstar per iteration of for loop = 8 maxstars
    end
    extrinsic_alphas = log_p1 - apriori_betas; % 8 additions

end


