%% Importing and Organising Data.

%% Read the data and store it into the tabular format
clear;
load('mpiv_vecdata.mat');

%We are interested in statistical parameters from all sequences. Each
%sequence contains a single mpiv 2D velocity vector field. A velocity
%vector field is a result of mpiv computation of either two consecutive
%frames (images) or two or more arbitrarily chosen frames.

% Here we define a set of variables we are interested in

Vars = {'SEQN','INDX','COLMEAN','COLMEDIAN','COLSTD','ROWMEAN','ROWMEDIAN',...
    'ROWSTD','COLMEAN_AD','COLMEDIAN_AD','ROWMEAN_AD','ROWMEDIAN_AD'};

VarUnits = {'#','#','[pixel]','[pixel]','[pixel]','[pixel]',...
    '[pixel]','[pixel]','[pixel]','[pixel]','[pixel]','[pixel]'};

VarDesc = {'Seq. number', 'Index', 'Col. mean', 'Col. median',...
    'Col. STD', 'Row mean','Row median','Row STD','Col. mean abs. dev.',...
    'Col median abs. dev.','Row mean abs. dev.','Row median abs. dev.'};


%% Read number of sequences (velocity fields) in the data, determine the size of a single vector field

seqN    = size(iuData,3);       %get the number of sequences
vSize   = size(iuData,2);       %get the size of a single velocity vect. field
vSizeIn = (1:vSize)';           %create a vector for indexing within a seqence
seqSize = ones(vSize,1);        %initialise sequence indexing vector

%transpose the velocity field matrixes so its coordinates reflect the
%coordinates of the analysed images (y axis in reverse direction)
iuData = permute(iuData,[2 1 3]);
ivData = permute(ivData,[2 1 3]);
iu_ipData = permute(iu_ipData,[2 1 3]);
iv_ipData = permute(iv_ipData,[2 1 3]);

%initialise an empty table
tmpzeros = zeros(vSize, size(Vars,2));
tmpdata = array2table(tmpzeros,'VariableNames',Vars);
tmpdata.Properties.VariableUnits = VarUnits;
tmpdata.Properties.VariableDescriptions = VarDesc;
%tmpdata.Properties.Description = 'Statistical analysis of velocity vector fields';

%raw mpiv data tables
dataU = tmpdata;                    % Vx component of the velocity vector
dataV = tmpdata;                    % Vy component of the velocity vector

%interpolated mpiv data tables
dataUip = tmpdata;                  % Vx component of the velocity vector
dataVip = tmpdata;                  % Vy component of the velocity vector

%% %%%% Build the dataU table %%%%%%
i=1;
%statistics for each column (along the 'y' axis) of the x component of the velocity vector
tmpdata{:,'COLMEAN'}      = nanmean(iuData(:,:,i),1)';
tmpdata{:,'COLMEDIAN'}    = nanmedian(iuData(:,:,i),1)';
tmpdata{:,'COLSTD'}       = nanstd(iuData(:,:,i),0,1)';
tmpdata{:,'COLMEAN_AD'}   = mad(iuData(:,:,i),0,1)';
tmpdata{:,'COLMEDIAN_AD'} = mad(iuData(:,:,i),1,1)';

%statistics for each row (along the 'x' axis) of the x component of the velocity vector
tmpdata{:,'ROWMEAN'}      = nanmean(iuData(:,:,i),2);
tmpdata{:,'ROWMEDIAN'}    = nanmedian(iuData(:,:,i),2);
tmpdata{:,'ROWSTD'}       = nanstd(iuData(:,:,i),0,2);
tmpdata{:,'ROWMEAN_AD'}   = mad(iuData(:,:,i),0,2);
tmpdata{:,'ROWMEDIAN_AD'} = mad(iuData(:,:,i),1,2);

%write index and sequence number into the table
tmpdata{:,'INDX'}      = vSizeIn;
tmpdata{:,'SEQN'}      = seqSize*i;

dataU = tmpdata;
for i = 2:seqN
    %statistics for each column (along the 'y' axis) of the x component of the velocity vector
    tmpdata{:,'COLMEAN'}      = nanmean(iuData(:,:,i),1)';
    tmpdata{:,'COLMEDIAN'}    = nanmedian(iuData(:,:,i),1)';
    tmpdata{:,'COLSTD'}       = nanstd(iuData(:,:,i),0,1)';
    tmpdata{:,'COLMEAN_AD'}   = mad(iuData(:,:,i),0,1)';
    tmpdata{:,'COLMEDIAN_AD'} = mad(iuData(:,:,i),1,1)';

    %statistics for each row (along the 'x' axis) of the x component of the velocity vector
    tmpdata{:,'ROWMEAN'}      = nanmean(iuData(:,:,i),2);
    tmpdata{:,'ROWMEDIAN'}    = nanmedian(iuData(:,:,i),2);
    tmpdata{:,'ROWSTD'}       = nanstd(iuData(:,:,i),0,2);
    tmpdata{:,'ROWMEAN_AD'}   = mad(iuData(:,:,i),0,2);
    tmpdata{:,'ROWMEDIAN_AD'} = mad(iuData(:,:,i),1,2);
    %write index and sequence number into the table
    tmpdata{:,'INDX'}      = vSizeIn;
    tmpdata{:,'SEQN'}      = seqSize*i;
    
    dataU = [dataU;tmpdata];
end
dataU.Properties.Description = 'Vx(x,y) stats of Velocity vector V(x,y)';

%%  Build the dataV table %%%

%statistics for each column (along the 'y' axis) of the y component of the velocity vector
tmpdata{:,'COLMEAN'}      = nanmean(ivData(:,:,i),1)';
tmpdata{:,'COLMEDIAN'}    = nanmedian(ivData(:,:,i),1)';
tmpdata{:,'COLSTD'}       = nanstd(ivData(:,:,i),0,1)';
tmpdata{:,'COLMEAN_AD'}   = mad(ivData(:,:,i),0,1)';
tmpdata{:,'COLMEDIAN_AD'} = mad(ivData(:,:,i),1,1)';

%statistics for each row (along the 'x' axis) of the y component of the velocity vector
tmpdata{:,'ROWMEAN'}      = nanmean(ivData(:,:,i),2);
tmpdata{:,'ROWMEDIAN'}    = nanmedian(ivData(:,:,i),2);
tmpdata{:,'ROWSTD'}       = nanstd(ivData(:,:,i),0,2);
tmpdata{:,'ROWMEAN_AD'}   = mad(ivData(:,:,i),0,2);
tmpdata{:,'ROWMEDIAN_AD'} = mad(ivData(:,:,i),1,2);

%write index and sequence number into the table
tmpdata{:,'INDX'}      = vSizeIn;
tmpdata{:,'SEQN'}      = seqSize*i;

dataV = tmpdata;
for i = 2:seqN
    %statistics for each column (along the 'y' axis) of the y component of the velocity vector
    tmpdata{:,'COLMEAN'}      = nanmean(ivData(:,:,i),1)';
    tmpdata{:,'COLMEDIAN'}    = nanmedian(ivData(:,:,i),1)';
    tmpdata{:,'COLSTD'}       = nanstd(ivData(:,:,i),0,1)';
    tmpdata{:,'COLMEAN_AD'}   = mad(ivData(:,:,i),0,1)';
    tmpdata{:,'COLMEDIAN_AD'} = mad(ivData(:,:,i),1,1)';

    %statistics for each row (along the 'x' axis) of the y component of the velocity vector
    tmpdata{:,'ROWMEAN'}      = nanmean(ivData(:,:,i),2);
    tmpdata{:,'ROWMEDIAN'}    = nanmedian(ivData(:,:,i),2);
    tmpdata{:,'ROWSTD'}       = nanstd(ivData(:,:,i),0,2);
    tmpdata{:,'ROWMEAN_AD'}   = mad(ivData(:,:,i),0,2);
    tmpdata{:,'ROWMEDIAN_AD'} = mad(ivData(:,:,i),1,2);
    %write index and sequence number into the table
    tmpdata{:,'INDX'}      = vSizeIn;
    tmpdata{:,'SEQN'}      = seqSize*i;
    
    dataV = [dataV;tmpdata];
end
dataV.Properties.Description = 'Vy(x,y) stats of Velocity vector V(x,y)';

%% %%%% Build the dataUip table %%%%
i=1;
%statistics for each column (along the 'y' axis) of the x component of the velocity vector
tmpdata{:,'COLMEAN'}      = nanmean(iu_ipData(:,:,i),1)';
tmpdata{:,'COLMEDIAN'}    = nanmedian(iu_ipData(:,:,i),1)';
tmpdata{:,'COLSTD'}       = nanstd(iu_ipData(:,:,i),0,1)';
tmpdata{:,'COLMEAN_AD'}   = mad(iu_ipData(:,:,i),0,1)';
tmpdata{:,'COLMEDIAN_AD'} = mad(iu_ipData(:,:,i),1,1)';

%statistics for each row (along the 'x' axis) of the x component of the velocity vector
tmpdata{:,'ROWMEAN'}      = nanmean(iu_ipData(:,:,i),2);
tmpdata{:,'ROWMEDIAN'}    = nanmedian(iu_ipData(:,:,i),2);
tmpdata{:,'ROWSTD'}       = nanstd(iu_ipData(:,:,i),0,2);
tmpdata{:,'ROWMEAN_AD'}   = mad(iu_ipData(:,:,i),0,2);
tmpdata{:,'ROWMEDIAN_AD'} = mad(iu_ipData(:,:,i),1,2);

%write index and sequence number into the table
tmpdata{:,'INDX'}      = vSizeIn;
tmpdata{:,'SEQN'}      = seqSize*i;

dataUip = tmpdata;
for i = 2:seqN
    %statistics for each column (along the 'y' axis) of the x component of the velocity vector
    tmpdata{:,'COLMEAN'}      = nanmean(iu_ipData(:,:,i),1)';
    tmpdata{:,'COLMEDIAN'}    = nanmedian(iu_ipData(:,:,i),1)';
    tmpdata{:,'COLSTD'}       = nanstd(iu_ipData(:,:,i),0,1)';
    tmpdata{:,'COLMEAN_AD'}   = mad(iu_ipData(:,:,i),0,1)';
    tmpdata{:,'COLMEDIAN_AD'} = mad(iu_ipData(:,:,i),1,1)';

    %statistics for each row (along the 'x' axis) of the x component of the velocity vector
    tmpdata{:,'ROWMEAN'}      = nanmean(iu_ipData(:,:,i),2);
    tmpdata{:,'ROWMEDIAN'}    = nanmedian(iu_ipData(:,:,i),2);
    tmpdata{:,'ROWSTD'}       = nanstd(iu_ipData(:,:,i),0,2);
    tmpdata{:,'ROWMEAN_AD'}   = mad(iu_ipData(:,:,i),0,2);
    tmpdata{:,'ROWMEDIAN_AD'} = mad(iu_ipData(:,:,i),1,2);
    %write index and sequence number into the table
    tmpdata{:,'INDX'}      = vSizeIn;
    tmpdata{:,'SEQN'}      = seqSize*i;
    
    dataUip = [dataUip;tmpdata];
end
dataUip.Properties.Description = 'Vx(x,y) of interpolated Velocity vector V(x,y)';


%% %%%% Build the dataVip table %%%%%
i=1;
%statistics for each column (along the 'y' axis) of the y component of the velocity vector
tmpdata{:,'COLMEAN'}      = nanmean(iv_ipData(:,:,i),1)';
tmpdata{:,'COLMEDIAN'}    = nanmedian(iv_ipData(:,:,i),1)';
tmpdata{:,'COLSTD'}       = nanstd(iv_ipData(:,:,i),0,1)';
tmpdata{:,'COLMEAN_AD'}   = mad(iv_ipData(:,:,i),0,1)';
tmpdata{:,'COLMEDIAN_AD'} = mad(iv_ipData(:,:,i),1,1)';

%statistics for each row (along the 'x' axis) of the y component of the velocity vector
tmpdata{:,'ROWMEAN'}      = nanmean(iv_ipData(:,:,i),2);
tmpdata{:,'ROWMEDIAN'}    = nanmedian(iv_ipData(:,:,i),2);
tmpdata{:,'ROWSTD'}       = nanstd(iv_ipData(:,:,i),0,2);
tmpdata{:,'ROWMEAN_AD'}   = mad(iv_ipData(:,:,i),0,2);
tmpdata{:,'ROWMEDIAN_AD'} = mad(iv_ipData(:,:,i),1,2);

%write index and sequence number into the table
tmpdata{:,'INDX'}      = vSizeIn;
tmpdata{:,'SEQN'}      = seqSize*i;

dataVip = tmpdata;
for i = 2:seqN
    %statistics for each column (along the 'y' axis) of the y component of the velocity vector
    tmpdata{:,'COLMEAN'}      = nanmean(iv_ipData(:,:,i),1)';
    tmpdata{:,'COLMEDIAN'}    = nanmedian(iv_ipData(:,:,i),1)';
    tmpdata{:,'COLSTD'}       = nanstd(iv_ipData(:,:,i),0,1)';
    tmpdata{:,'COLMEAN_AD'}   = mad(iv_ipData(:,:,i),0,1)';
    tmpdata{:,'COLMEDIAN_AD'} = mad(iv_ipData(:,:,i),1,1)';

    %statistics for each row (along the 'x' axis) of the y component of the velocity vector
    tmpdata{:,'ROWMEAN'}      = nanmean(iv_ipData(:,:,i),2);
    tmpdata{:,'ROWMEDIAN'}    = nanmedian(iv_ipData(:,:,i),2);
    tmpdata{:,'ROWSTD'}       = nanstd(iv_ipData(:,:,i),0,2);
    tmpdata{:,'ROWMEAN_AD'}   = mad(iv_ipData(:,:,i),0,2);
    tmpdata{:,'ROWMEDIAN_AD'} = mad(iv_ipData(:,:,i),1,2);
    %write index and sequence number into the table
    tmpdata{:,'INDX'}      = vSizeIn;
    tmpdata{:,'SEQN'}      = seqSize*i;
    
    dataVip = [dataVip;tmpdata];
end
dataVip.Properties.Description = 'Vy(x,y) of interpolated Velocity vector V(x,y)';

%% Calculate and organise cumulative statistics for each sequence

% for x component of the velocity vector
for i = 1:seqN
    tmp = iuData(:,:,i);
    iuDataMean(i)      = mean(tmp(~isnan(tmp)));
    iuDataMed(i)       = median(tmp(~isnan(tmp)));
    iuDataStd(i)       = std(tmp(~isnan(tmp)));
    iuDataAdMean(i)    = mad(tmp(~isnan(tmp)),0);
    iuDataAdMed(i)     = mad(tmp(~isnan(tmp)),1);
    
end

%for y component of the velocity vector
for i = 1:seqN
    tmp = ivData(:,:,i);
    ivDataMean(i)      = mean(tmp(~isnan(tmp)));
    ivDataMed(i)       = median(tmp(~isnan(tmp)));
    ivDataStd(i)       = std(tmp(~isnan(tmp)));
    ivDataAdMean(i)    = mad(tmp(~isnan(tmp)),0);
    ivDataAdMed(i)     = mad(tmp(~isnan(tmp)),1);
    
end

%for x component of the interpolated velocity vector
for i = 1:seqN
    tmp = iu_ipData(:,:,i);
    iu_ipDataMean(i)      = mean(tmp(~isnan(tmp)));
    iu_ipDataMed(i)       = median(tmp(~isnan(tmp)));
    iu_ipDataStd(i)       = std(tmp(~isnan(tmp)));
    iu_ipDataAdMean(i)    = mad(tmp(~isnan(tmp)),0);
    iu_ipDataAdMed(i)     = mad(tmp(~isnan(tmp)),1);
    
end

%for y component of the interpolated velocity vector
for i = 1:seqN
    tmp = iv_ipData(:,:,i);
    iv_ipDataMean(i)      = mean(tmp(~isnan(tmp)));
    iv_ipDataMed(i)       = median(tmp(~isnan(tmp)));
    iv_ipDataStd(i)       = std(tmp(~isnan(tmp)));
    iv_ipDataAdMean(i)    = mad(tmp(~isnan(tmp)),0);
    iv_ipDataAdMed(i)     = mad(tmp(~isnan(tmp)),1);
    
end

%% Write the data into the tables

Vars2 = {'SEQN','MEAN','MEDIAN','STD','MEAN_AD','MEDIAN_AD'};
VarDesc2 = {'Seq. number','Mean','Median','STD','Abs. dev. about mean','Abs. dev. about median'};
VarUnits2 = {'seq #','[pixel]','[pixel]','[pixel]','[pixel]','[pixel]'};

seqIn   = (1:seqN)';            %create a vector for indexing sequences

%initialise an empty table
tmpzeros = zeros(seqN, size(Vars2,2));
tmpdata = array2table(tmpzeros,'VariableNames',Vars2);
tmpdata.Properties.Description = 'Statistics of each sequence';
tmpdata.Properties.VariableUnits = VarUnits2;
tmpdata.Properties.VariableDescriptions = VarDesc2;

tmpdata{:,'SEQN'} = seqIn;

%raw mpiv data tables
dataSeqU = tmpdata;                    % x component of the velocity vector
dataSeqU.Properties.Description = 'Vx of velocity V(x,y)';

dataSeqV = tmpdata;                    % y component of the velocity vector
dataSeqV.Properties.Description = 'Vy of velocity V(x,y)';

%interpolated mpiv data tables
dataSeqUip = tmpdata;                  % x component of the velocity vector
dataSeqUip.Properties.Description = 'Vx of interpolated velocity V(x,y)';

dataSeqVip = tmpdata;                  % y component of the velocity vector
dataSeqVip.Properties.Description = 'Vy of interpolated velocity V(x,y)';

% x component of the velocity vector
dataSeqU{:,'MEAN'}          = iuDataMean';
dataSeqU{:,'MEDIAN'}        = iuDataMed';
dataSeqU{:,'STD'}           = iuDataStd';
dataSeqU{:,'MEAN_AD'}       = iuDataAdMean';
dataSeqU{:,'MEDIAN_AD'}     = iuDataAdMed';

% y component of the velocity vector
dataSeqV{:,'MEAN'}          = ivDataMean';
dataSeqV{:,'MEDIAN'}        = ivDataMed';
dataSeqV{:,'STD'}           = ivDataStd';
dataSeqV{:,'MEAN_AD'}       = ivDataAdMean';
dataSeqV{:,'MEDIAN_AD'}     = ivDataAdMed';

% x interpolated component of the velocity vector
dataSeqUip{:,'MEAN'}        = iu_ipDataMean';
dataSeqUip{:,'MEDIAN'}      = iu_ipDataMed';
dataSeqUip{:,'STD'}         = iu_ipDataStd';
dataSeqUip{:,'MEAN_AD'}     = iu_ipDataAdMean';
dataSeqUip{:,'MEDIAN_AD'}   = iu_ipDataAdMed';

% y inerpolated component of the velocity vector
dataSeqVip{:,'MEAN'}        = iv_ipDataMean';
dataSeqVip{:,'MEDIAN'}      = iv_ipDataMed';
dataSeqVip{:,'STD'}         = iv_ipDataStd';
dataSeqVip{:,'MEAN_AD'}     = iv_ipDataAdMean';
dataSeqVip{:,'MEDIAN_AD'}   = iv_ipDataAdMed';

