function data_analysis_gui
% SIMPLE_GUI2 Select a data set from the pop-up menu, then
% click one of the plot-type push buttons. Clicking the button
% plots the selected data in the axes.
 
   
%% Read the data
        filterspec={'*.mat'};
        [filename, pathname, filterindex] = uigetfile(filterspec, 'Pick a single data file');
        if isequal(filename,0) || isequal(pathname,0)
            msgbox(sprintf('Error'),'Error', 'Error'); 
            disp('User pressed cancel')
        else
            disp(['User selected ', fullfile(pathname, filename)])
            dataFilename = fullfile(pathname, filename);
            mydata = load(dataFilename);

        end
iu_ipData   = mydata.iu_ipData;
iuData      = mydata.iuData;
iv_ipData   = mydata.iv_ipData;
ivData      = mydata.ivData;
xi          = mydata.xi;
yi          = mydata.yi;

clear mydata;


        %% Organise Data.

        %% Store it into the tabular format


        %We are interested in statistical parameters from all sequences. Each
        %sequence contains a single mpiv 2D velocity vector field. A velocity
        %vector field is a result of mpiv computation of either two consecutive
        %frames (images) or two or more arbitrarily chosen frames.

        % Here we define a set of variables we are interested in

        Vars = {'SEQN','INDX','COLMEAN','COLMEDIAN','COLSTD','ROWMEAN','ROWMEDIAN',...
        'ROWSTD','COLMEAN_AD','COLMEDIAN_AD','ROWMEAN_AD','ROWMEDIAN_AD'};

        VarUnits = {'#','#','[pixel]','[pixel]','[pixel]','[pixel]',...
        '[pixel]','[pixel]','[pixel]','[pixel]','[pixel]','[pixel]'};

        VarDesc = {'Seq. number', 'Index', 'Col. mean', 'Col. median',...
        'Col. STD', 'Row mean','Row median','Row STD','Col. mean abs. dev.',...
        'Col median abs. dev.','Row mean abs. dev.','Row median abs. dev.'};


        %% Read number of sequences (velocity fields) in the data, determine the size of a single vector field

        seqN    = size(iuData,3);       %get the number of sequences
        vSize   = size(iuData,2);       %get the size of a single velocity vect. field
        vSizeIn = (1:vSize)';           %create a vector for indexing within a seqence
        seqSize = ones(vSize,1);        %initialise sequence indexing vector

        %transpose the velocity field matrixes so its coordinates reflect the
        %coordinates of the analysed images (y axis in reverse direction)
        iuData = permute(iuData,[2 1 3]);
        ivData = permute(ivData,[2 1 3]);
        iu_ipData = permute(iu_ipData,[2 1 3]);
        iv_ipData = permute(iv_ipData,[2 1 3]);
        
        %initialise an empty table
        tmpzeros = zeros(vSize, size(Vars,2));
        tmpdata = array2table(tmpzeros,'VariableNames',Vars);
        tmpdata.Properties.VariableUnits = VarUnits;
        tmpdata.Properties.VariableDescriptions = VarDesc;
        %tmpdata.Properties.Description = 'Statistical analysis of velocity vector fields';

        %raw mpiv data tables
        dataU = tmpdata;                    % Vx component of the velocity vector
        dataV = tmpdata;                    % Vy component of the velocity vector

        %interpolated mpiv data tables
        dataUip = tmpdata;                  % Vx component of the velocity vector
        dataVip = tmpdata;                  % Vy component of the velocity vector

        %% %%%% Build the dataU table %%%%%%
        i=1;
        %statistics for each column (along the 'y' axis) of the x component of the velocity vector
        tmpdata{:,'COLMEAN'}      = nanmean(iuData(:,:,i),1)';
        tmpdata{:,'COLMEDIAN'}    = nanmedian(iuData(:,:,i),1)';
        tmpdata{:,'COLSTD'}       = nanstd(iuData(:,:,i),0,1)';
        tmpdata{:,'COLMEAN_AD'}   = mad(iuData(:,:,i),0,1)';
        tmpdata{:,'COLMEDIAN_AD'} = mad(iuData(:,:,i),1,1)';

        %statistics for each row (along the 'x' axis) of the x component of the velocity vector
        tmpdata{:,'ROWMEAN'}      = nanmean(iuData(:,:,i),2);
        tmpdata{:,'ROWMEDIAN'}    = nanmedian(iuData(:,:,i),2);
        tmpdata{:,'ROWSTD'}       = nanstd(iuData(:,:,i),0,2);
        tmpdata{:,'ROWMEAN_AD'}   = mad(iuData(:,:,i),0,2);
        tmpdata{:,'ROWMEDIAN_AD'} = mad(iuData(:,:,i),1,2);

        %write index and sequence number into the table
        tmpdata{:,'INDX'}      = vSizeIn;
        tmpdata{:,'SEQN'}      = seqSize*i;

        dataU = tmpdata;
        for i = 2:seqN
        %statistics for each column (along the 'y' axis) of the x component of the velocity vector
        tmpdata{:,'COLMEAN'}      = nanmean(iuData(:,:,i),1)';
        tmpdata{:,'COLMEDIAN'}    = nanmedian(iuData(:,:,i),1)';
        tmpdata{:,'COLSTD'}       = nanstd(iuData(:,:,i),0,1)';
        tmpdata{:,'COLMEAN_AD'}   = mad(iuData(:,:,i),0,1)';
        tmpdata{:,'COLMEDIAN_AD'} = mad(iuData(:,:,i),1,1)';

        %statistics for each row (along the 'x' axis) of the x component of the velocity vector
        tmpdata{:,'ROWMEAN'}      = nanmean(iuData(:,:,i),2);
        tmpdata{:,'ROWMEDIAN'}    = nanmedian(iuData(:,:,i),2);
        tmpdata{:,'ROWSTD'}       = nanstd(iuData(:,:,i),0,2);
        tmpdata{:,'ROWMEAN_AD'}   = mad(iuData(:,:,i),0,2);
        tmpdata{:,'ROWMEDIAN_AD'} = mad(iuData(:,:,i),1,2);
        %write index and sequence number into the table
        tmpdata{:,'INDX'}      = vSizeIn;
        tmpdata{:,'SEQN'}      = seqSize*i;

        dataU = [dataU;tmpdata];
        end
        dataU.Properties.Description = 'Vx(x,y) stats of Velocity vector V(x,y)';

        %%  Build the dataV table %%%
        i=1;
        %statistics for each column (along the 'y' axis) of the y component of the velocity vector
        tmpdata{:,'COLMEAN'}      = nanmean(ivData(:,:,i),1)';
        tmpdata{:,'COLMEDIAN'}    = nanmedian(ivData(:,:,i),1)';
        tmpdata{:,'COLSTD'}       = nanstd(ivData(:,:,i),0,1)';
        tmpdata{:,'COLMEAN_AD'}   = mad(ivData(:,:,i),0,1)';
        tmpdata{:,'COLMEDIAN_AD'} = mad(ivData(:,:,i),1,1)';

        %statistics for each row (along the 'x' axis) of the y component of the velocity vector
        tmpdata{:,'ROWMEAN'}      = nanmean(ivData(:,:,i),2);
        tmpdata{:,'ROWMEDIAN'}    = nanmedian(ivData(:,:,i),2);
        tmpdata{:,'ROWSTD'}       = nanstd(ivData(:,:,i),0,2);
        tmpdata{:,'ROWMEAN_AD'}   = mad(ivData(:,:,i),0,2);
        tmpdata{:,'ROWMEDIAN_AD'} = mad(ivData(:,:,i),1,2);

        %write index and sequence number into the table
        tmpdata{:,'INDX'}      = vSizeIn;
        tmpdata{:,'SEQN'}      = seqSize*i;

        dataV = tmpdata;
        for i = 2:seqN
        %statistics for each column (along the 'y' axis) of the y component of the velocity vector
        tmpdata{:,'COLMEAN'}      = nanmean(ivData(:,:,i),1)';
        tmpdata{:,'COLMEDIAN'}    = nanmedian(ivData(:,:,i),1)';
        tmpdata{:,'COLSTD'}       = nanstd(ivData(:,:,i),0,1)';
        tmpdata{:,'COLMEAN_AD'}   = mad(ivData(:,:,i),0,1)';
        tmpdata{:,'COLMEDIAN_AD'} = mad(ivData(:,:,i),1,1)';

        %statistics for each row (along the 'x' axis) of the y component of the velocity vector
        tmpdata{:,'ROWMEAN'}      = nanmean(ivData(:,:,i),2);
        tmpdata{:,'ROWMEDIAN'}    = nanmedian(ivData(:,:,i),2);
        tmpdata{:,'ROWSTD'}       = nanstd(ivData(:,:,i),0,2);
        tmpdata{:,'ROWMEAN_AD'}   = mad(ivData(:,:,i),0,2);
        tmpdata{:,'ROWMEDIAN_AD'} = mad(ivData(:,:,i),1,2);
        %write index and sequence number into the table
        tmpdata{:,'INDX'}      = vSizeIn;
        tmpdata{:,'SEQN'}      = seqSize*i;

        dataV = [dataV;tmpdata];
        end
        dataV.Properties.Description = 'Vy(x,y) stats of Velocity vector V(x,y)';

        %% %%%% Build the dataUip table %%%%
        i=1;
        %statistics for each column (along the 'y' axis) of the x component of the velocity vector
        tmpdata{:,'COLMEAN'}      = nanmean(iu_ipData(:,:,i),1)';
        tmpdata{:,'COLMEDIAN'}    = nanmedian(iu_ipData(:,:,i),1)';
        tmpdata{:,'COLSTD'}       = nanstd(iu_ipData(:,:,i),0,1)';
        tmpdata{:,'COLMEAN_AD'}   = mad(iu_ipData(:,:,i),0,1)';
        tmpdata{:,'COLMEDIAN_AD'} = mad(iu_ipData(:,:,i),1,1)';

        %statistics for each row (along the 'x' axis) of the x component of the velocity vector
        tmpdata{:,'ROWMEAN'}      = nanmean(iu_ipData(:,:,i),2);
        tmpdata{:,'ROWMEDIAN'}    = nanmedian(iu_ipData(:,:,i),2);
        tmpdata{:,'ROWSTD'}       = nanstd(iu_ipData(:,:,i),0,2);
        tmpdata{:,'ROWMEAN_AD'}   = mad(iu_ipData(:,:,i),0,2);
        tmpdata{:,'ROWMEDIAN_AD'} = mad(iu_ipData(:,:,i),1,2);

        %write index and sequence number into the table
        tmpdata{:,'INDX'}      = vSizeIn;
        tmpdata{:,'SEQN'}      = seqSize*i;

        dataUip = tmpdata;
        for i = 2:seqN
        %statistics for each column (along the 'y' axis) of the x component of the velocity vector
        tmpdata{:,'COLMEAN'}      = nanmean(iu_ipData(:,:,i),1)';
        tmpdata{:,'COLMEDIAN'}    = nanmedian(iu_ipData(:,:,i),1)';
        tmpdata{:,'COLSTD'}       = nanstd(iu_ipData(:,:,i),0,1)';
        tmpdata{:,'COLMEAN_AD'}   = mad(iu_ipData(:,:,i),0,1)';
        tmpdata{:,'COLMEDIAN_AD'} = mad(iu_ipData(:,:,i),1,1)';

        %statistics for each row (along the 'x' axis) of the x component of the velocity vector
        tmpdata{:,'ROWMEAN'}      = nanmean(iu_ipData(:,:,i),2);
        tmpdata{:,'ROWMEDIAN'}    = nanmedian(iu_ipData(:,:,i),2);
        tmpdata{:,'ROWSTD'}       = nanstd(iu_ipData(:,:,i),0,2);
        tmpdata{:,'ROWMEAN_AD'}   = mad(iu_ipData(:,:,i),0,2);
        tmpdata{:,'ROWMEDIAN_AD'} = mad(iu_ipData(:,:,i),1,2);
        %write index and sequence number into the table
        tmpdata{:,'INDX'}      = vSizeIn;
        tmpdata{:,'SEQN'}      = seqSize*i;

        dataUip = [dataUip;tmpdata];
        end
        dataUip.Properties.Description = 'Vx(x,y) of interpolated Velocity vector V(x,y)';


        %% %%%% Build the dataVip table %%%%%
        i=1;
        %statistics for each column (along the 'y' axis) of the y component of the velocity vector
        tmpdata{:,'COLMEAN'}      = nanmean(iv_ipData(:,:,i),1)';
        tmpdata{:,'COLMEDIAN'}    = nanmedian(iv_ipData(:,:,i),1)';
        tmpdata{:,'COLSTD'}       = nanstd(iv_ipData(:,:,i),0,1)';
        tmpdata{:,'COLMEAN_AD'}   = mad(iv_ipData(:,:,i),0,1)';
        tmpdata{:,'COLMEDIAN_AD'} = mad(iv_ipData(:,:,i),1,1)';

        %statistics for each row (along the 'x' axis) of the y component of the velocity vector
        tmpdata{:,'ROWMEAN'}      = nanmean(iv_ipData(:,:,i),2);
        tmpdata{:,'ROWMEDIAN'}    = nanmedian(iv_ipData(:,:,i),2);
        tmpdata{:,'ROWSTD'}       = nanstd(iv_ipData(:,:,i),0,2);
        tmpdata{:,'ROWMEAN_AD'}   = mad(iv_ipData(:,:,i),0,2);
        tmpdata{:,'ROWMEDIAN_AD'} = mad(iv_ipData(:,:,i),1,2);

        %write index and sequence number into the table
        tmpdata{:,'INDX'}      = vSizeIn;
        tmpdata{:,'SEQN'}      = seqSize*i;

        dataVip = tmpdata;
        for i = 2:seqN
        %statistics for each column (along the 'y' axis) of the y component of the velocity vector
        tmpdata{:,'COLMEAN'}      = nanmean(iv_ipData(:,:,i),1)';
        tmpdata{:,'COLMEDIAN'}    = nanmedian(iv_ipData(:,:,i),1)';
        tmpdata{:,'COLSTD'}       = nanstd(iv_ipData(:,:,i),0,1)';
        tmpdata{:,'COLMEAN_AD'}   = mad(iv_ipData(:,:,i),0,1)';
        tmpdata{:,'COLMEDIAN_AD'} = mad(iv_ipData(:,:,i),1,1)';

        %statistics for each row (along the 'x' axis) of the y component of the velocity vector
        tmpdata{:,'ROWMEAN'}      = nanmean(iv_ipData(:,:,i),2);
        tmpdata{:,'ROWMEDIAN'}    = nanmedian(iv_ipData(:,:,i),2);
        tmpdata{:,'ROWSTD'}       = nanstd(iv_ipData(:,:,i),0,2);
        tmpdata{:,'ROWMEAN_AD'}   = mad(iv_ipData(:,:,i),0,2);
        tmpdata{:,'ROWMEDIAN_AD'} = mad(iv_ipData(:,:,i),1,2);
        %write index and sequence number into the table
        tmpdata{:,'INDX'}      = vSizeIn;
        tmpdata{:,'SEQN'}      = seqSize*i;

        dataVip = [dataVip;tmpdata];
        end
        dataVip.Properties.Description = 'Vy(x,y) of interpolated Velocity vector V(x,y)';

        %% Calculate and organise cumulative statistics for each sequence

        % for x component of the velocity vector
        for i = 1:seqN
        tmp = iuData(:,:,i);
        iuDataMean(i)      = mean(tmp(~isnan(tmp)));
        iuDataMed(i)       = median(tmp(~isnan(tmp)));
        iuDataStd(i)       = std(tmp(~isnan(tmp)));
        iuDataAdMean(i)    = mad(tmp(~isnan(tmp)),0);
        iuDataAdMed(i)     = mad(tmp(~isnan(tmp)),1);

        end

        %for y component of the velocity vector
        for i = 1:seqN
        tmp = ivData(:,:,i);
        ivDataMean(i)      = mean(tmp(~isnan(tmp)));
        ivDataMed(i)       = median(tmp(~isnan(tmp)));
        ivDataStd(i)       = std(tmp(~isnan(tmp)));
        ivDataAdMean(i)    = mad(tmp(~isnan(tmp)),0);
        ivDataAdMed(i)     = mad(tmp(~isnan(tmp)),1);

        end

        %for x component of the interpolated velocity vector
        for i = 1:seqN
        tmp = iu_ipData(:,:,i);
        iu_ipDataMean(i)      = mean(tmp(~isnan(tmp)));
        iu_ipDataMed(i)       = median(tmp(~isnan(tmp)));
        iu_ipDataStd(i)       = std(tmp(~isnan(tmp)));
        iu_ipDataAdMean(i)    = mad(tmp(~isnan(tmp)),0);
        iu_ipDataAdMed(i)     = mad(tmp(~isnan(tmp)),1);

        end

        %for y component of the interpolated velocity vector
        for i = 1:seqN
        tmp = iv_ipData(:,:,i);
        iv_ipDataMean(i)      = mean(tmp(~isnan(tmp)));
        iv_ipDataMed(i)       = median(tmp(~isnan(tmp)));
        iv_ipDataStd(i)       = std(tmp(~isnan(tmp)));
        iv_ipDataAdMean(i)    = mad(tmp(~isnan(tmp)),0);
        iv_ipDataAdMed(i)     = mad(tmp(~isnan(tmp)),1);

        end

        %% Write the data into the tables

        Vars2 = {'SEQN','MEAN','MEDIAN','STD','MEAN_AD','MEDIAN_AD'};
        VarDesc2 = {'Seq. number','Mean','Median','STD','Abs. dev. about mean','Abs. dev. about median'};
        VarUnits2 = {'seq #','[pixel]','[pixel]','[pixel]','[pixel]','[pixel]'};

        seqIn   = (1:seqN)';            %create a vector for indexing sequences

        %initialise an empty table
        tmpzeros = zeros(seqN, size(Vars2,2));
        tmpdata = array2table(tmpzeros,'VariableNames',Vars2);
        tmpdata.Properties.Description = 'Statistics of each sequence';
        tmpdata.Properties.VariableUnits = VarUnits2;
        tmpdata.Properties.VariableDescriptions = VarDesc2;

        tmpdata{:,'SEQN'} = seqIn;

        %raw mpiv data tables
        dataSeqU = tmpdata;                    % x component of the velocity vector
        dataSeqU.Properties.Description = 'Vx of velocity V(x,y)';

        dataSeqV = tmpdata;                    % y component of the velocity vector
        dataSeqV.Properties.Description = 'Vy of velocity V(x,y)';

        %interpolated mpiv data tables
        dataSeqUip = tmpdata;                  % x component of the velocity vector
        dataSeqUip.Properties.Description = 'Vx of interpolated velocity V(x,y)';

        dataSeqVip = tmpdata;                  % y component of the velocity vector
        dataSeqVip.Properties.Description = 'Vy of interpolated velocity V(x,y)';

        % x component of the velocity vector
        dataSeqU{:,'MEAN'}          = iuDataMean';
        dataSeqU{:,'MEDIAN'}        = iuDataMed';
        dataSeqU{:,'STD'}           = iuDataStd';
        dataSeqU{:,'MEAN_AD'}       = iuDataAdMean';
        dataSeqU{:,'MEDIAN_AD'}     = iuDataAdMed';

        % y component of the velocity vector
        dataSeqV{:,'MEAN'}          = ivDataMean';
        dataSeqV{:,'MEDIAN'}        = ivDataMed';
        dataSeqV{:,'STD'}           = ivDataStd';
        dataSeqV{:,'MEAN_AD'}       = ivDataAdMean';
        dataSeqV{:,'MEDIAN_AD'}     = ivDataAdMed';

        % x interpolated component of the velocity vector
        dataSeqUip{:,'MEAN'}        = iu_ipDataMean';
        dataSeqUip{:,'MEDIAN'}      = iu_ipDataMed';
        dataSeqUip{:,'STD'}         = iu_ipDataStd';
        dataSeqUip{:,'MEAN_AD'}     = iu_ipDataAdMean';
        dataSeqUip{:,'MEDIAN_AD'}   = iu_ipDataAdMed';

        % y inerpolated component of the velocity vector
        dataSeqVip{:,'MEAN'}        = iv_ipDataMean';
        dataSeqVip{:,'MEDIAN'}      = iv_ipDataMed';
        dataSeqVip{:,'STD'}         = iv_ipDataStd';
        dataSeqVip{:,'MEAN_AD'}     = iv_ipDataAdMean';
        dataSeqVip{:,'MEDIAN_AD'}   = iv_ipDataAdMed';
        
%% Create the new data set with mean velocities subtracted
        iuDatam = iuData;
        ivDatam = ivData;
        iu_ipDatam = iu_ipData;
        iv_ipDatam = iv_ipData;
        
        for k=1:seqN
            iuDatam(:,:,k) = iuDatam(:,:,k) - dataSeqU.MEAN(k);
            ivDatam(:,:,k) = ivDatam(:,:,k) - dataSeqV.MEAN(k);
            iu_ipDatam(:,:,k) = iu_ipDatam(:,:,k) - dataSeqUip.MEAN(k);
            iv_ipDatam(:,:,k) = iv_ipDatam(:,:,k) - dataSeqVip.MEAN(k);
        end

%% Create the new set of stats data from the set above

                        %initialise an empty table
                        tmpzeros = zeros(vSize, size(Vars,2));
                        tmpdata = array2table(tmpzeros,'VariableNames',Vars);
                        tmpdata.Properties.VariableUnits = VarUnits;
                        tmpdata.Properties.VariableDescriptions = VarDesc;
                        %tmpdata.Properties.Description = 'Statistical analysis of velocity vector fields';

                        %raw mpiv data tables
                        dataUm = tmpdata;                    % Vx component of the velocity vector
                        dataVm = tmpdata;                    % Vy component of the velocity vector

                        %interpolated mpiv data tables
                        dataUipm = tmpdata;                  % Vx component of the velocity vector
                        dataVipm = tmpdata;                  % Vy component of the velocity vector

                        %% %%%% Build the dataU table %%%%%%
                        i=1;
                        %statistics for each column (along the 'y' axis) of the x component of the velocity vector
                        tmpdata{:,'COLMEAN'}      = nanmean(iuDatam(:,:,i),1)';
                        tmpdata{:,'COLMEDIAN'}    = nanmedian(iuDatam(:,:,i),1)';
                        tmpdata{:,'COLSTD'}       = nanstd(iuDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEAN_AD'}   = mad(iuDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEDIAN_AD'} = mad(iuDatam(:,:,i),1,1)';

                        %statistics for each row (along the 'x' axis) of the x component of the velocity vector
                        tmpdata{:,'ROWMEAN'}      = nanmean(iuDatam(:,:,i),2);
                        tmpdata{:,'ROWMEDIAN'}    = nanmedian(iuDatam(:,:,i),2);
                        tmpdata{:,'ROWSTD'}       = nanstd(iuDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEAN_AD'}   = mad(iuDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEDIAN_AD'} = mad(iuDatam(:,:,i),1,2);

                        %write index and sequence number into the table
                        tmpdata{:,'INDX'}      = vSizeIn;
                        tmpdata{:,'SEQN'}      = seqSize*i;

                        dataUm = tmpdata;
                        for i = 2:seqN
                        %statistics for each column (along the 'y' axis) of the x component of the velocity vector
                        tmpdata{:,'COLMEAN'}      = nanmean(iuDatam(:,:,i),1)';
                        tmpdata{:,'COLMEDIAN'}    = nanmedian(iuDatam(:,:,i),1)';
                        tmpdata{:,'COLSTD'}       = nanstd(iuDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEAN_AD'}   = mad(iuDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEDIAN_AD'} = mad(iuDatam(:,:,i),1,1)';

                        %statistics for each row (along the 'x' axis) of the x component of the velocity vector
                        tmpdata{:,'ROWMEAN'}      = nanmean(iuDatam(:,:,i),2);
                        tmpdata{:,'ROWMEDIAN'}    = nanmedian(iuDatam(:,:,i),2);
                        tmpdata{:,'ROWSTD'}       = nanstd(iuDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEAN_AD'}   = mad(iuDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEDIAN_AD'} = mad(iuDatam(:,:,i),1,2);
                        %write index and sequence number into the table
                        tmpdata{:,'INDX'}      = vSizeIn;
                        tmpdata{:,'SEQN'}      = seqSize*i;

                        dataUm = [dataUm;tmpdata];
                        end
                        dataUm.Properties.Description = 'Vx(x,y) stats of Velocity vector V(x,y)';

                        %%  Build the dataV table %%%
                        i=1;
                        %statistics for each column (along the 'y' axis) of the y component of the velocity vector
                        tmpdata{:,'COLMEAN'}      = nanmean(ivDatam(:,:,i),1)';
                        tmpdata{:,'COLMEDIAN'}    = nanmedian(ivDatam(:,:,i),1)';
                        tmpdata{:,'COLSTD'}       = nanstd(ivDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEAN_AD'}   = mad(ivDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEDIAN_AD'} = mad(ivDatam(:,:,i),1,1)';

                        %statistics for each row (along the 'x' axis) of the y component of the velocity vector
                        tmpdata{:,'ROWMEAN'}      = nanmean(ivDatam(:,:,i),2);
                        tmpdata{:,'ROWMEDIAN'}    = nanmedian(ivDatam(:,:,i),2);
                        tmpdata{:,'ROWSTD'}       = nanstd(ivDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEAN_AD'}   = mad(ivDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEDIAN_AD'} = mad(ivDatam(:,:,i),1,2);

                        %write index and sequence number into the table
                        tmpdata{:,'INDX'}      = vSizeIn;
                        tmpdata{:,'SEQN'}      = seqSize*i;

                        dataVm = tmpdata;
                        for i = 2:seqN
                        %statistics for each column (along the 'y' axis) of the y component of the velocity vector
                        tmpdata{:,'COLMEAN'}      = nanmean(ivDatam(:,:,i),1)';
                        tmpdata{:,'COLMEDIAN'}    = nanmedian(ivDatam(:,:,i),1)';
                        tmpdata{:,'COLSTD'}       = nanstd(ivDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEAN_AD'}   = mad(ivDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEDIAN_AD'} = mad(ivDatam(:,:,i),1,1)';

                        %statistics for each row (along the 'x' axis) of the y component of the velocity vector
                        tmpdata{:,'ROWMEAN'}      = nanmean(ivDatam(:,:,i),2);
                        tmpdata{:,'ROWMEDIAN'}    = nanmedian(ivDatam(:,:,i),2);
                        tmpdata{:,'ROWSTD'}       = nanstd(ivDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEAN_AD'}   = mad(ivDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEDIAN_AD'} = mad(ivDatam(:,:,i),1,2);
                        %write index and sequence number into the table
                        tmpdata{:,'INDX'}      = vSizeIn;
                        tmpdata{:,'SEQN'}      = seqSize*i;

                        dataVm = [dataVm;tmpdata];
                        end
                        dataVm.Properties.Description = 'Vy(x,y) stats of Velocity vector V(x,y)';

                        %% %%%% Build the dataUip table %%%%
                        i=1;
                        %statistics for each column (along the 'y' axis) of the x component of the velocity vector
                        tmpdata{:,'COLMEAN'}      = nanmean(iu_ipDatam(:,:,i),1)';
                        tmpdata{:,'COLMEDIAN'}    = nanmedian(iu_ipDatam(:,:,i),1)';
                        tmpdata{:,'COLSTD'}       = nanstd(iu_ipDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEAN_AD'}   = mad(iu_ipDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEDIAN_AD'} = mad(iu_ipDatam(:,:,i),1,1)';

                        %statistics for each row (along the 'x' axis) of the x component of the velocity vector
                        tmpdata{:,'ROWMEAN'}      = nanmean(iu_ipDatam(:,:,i),2);
                        tmpdata{:,'ROWMEDIAN'}    = nanmedian(iu_ipDatam(:,:,i),2);
                        tmpdata{:,'ROWSTD'}       = nanstd(iu_ipDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEAN_AD'}   = mad(iu_ipDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEDIAN_AD'} = mad(iu_ipDatam(:,:,i),1,2);

                        %write index and sequence number into the table
                        tmpdata{:,'INDX'}      = vSizeIn;
                        tmpdata{:,'SEQN'}      = seqSize*i;

                        dataUipm = tmpdata;
                        for i = 2:seqN
                        %statistics for each column (along the 'y' axis) of the x component of the velocity vector
                        tmpdata{:,'COLMEAN'}      = nanmean(iu_ipDatam(:,:,i),1)';
                        tmpdata{:,'COLMEDIAN'}    = nanmedian(iu_ipDatam(:,:,i),1)';
                        tmpdata{:,'COLSTD'}       = nanstd(iu_ipDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEAN_AD'}   = mad(iu_ipDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEDIAN_AD'} = mad(iu_ipDatam(:,:,i),1,1)';

                        %statistics for each row (along the 'x' axis) of the x component of the velocity vector
                        tmpdata{:,'ROWMEAN'}      = nanmean(iu_ipDatam(:,:,i),2);
                        tmpdata{:,'ROWMEDIAN'}    = nanmedian(iu_ipDatam(:,:,i),2);
                        tmpdata{:,'ROWSTD'}       = nanstd(iu_ipDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEAN_AD'}   = mad(iu_ipDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEDIAN_AD'} = mad(iu_ipDatam(:,:,i),1,2);
                        %write index and sequence number into the table
                        tmpdata{:,'INDX'}      = vSizeIn;
                        tmpdata{:,'SEQN'}      = seqSize*i;

                        dataUipm = [dataUipm;tmpdata];
                        end
                        dataUipm.Properties.Description = 'Vx(x,y) of interpolated Velocity vector V(x,y)';


                        %% %%%% Build the dataVip table %%%%%
                        i=1;
                        %statistics for each column (along the 'y' axis) of the y component of the velocity vector
                        tmpdata{:,'COLMEAN'}      = nanmean(iv_ipDatam(:,:,i),1)';
                        tmpdata{:,'COLMEDIAN'}    = nanmedian(iv_ipDatam(:,:,i),1)';
                        tmpdata{:,'COLSTD'}       = nanstd(iv_ipDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEAN_AD'}   = mad(iv_ipDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEDIAN_AD'} = mad(iv_ipDatam(:,:,i),1,1)';

                        %statistics for each row (along the 'x' axis) of the y component of the velocity vector
                        tmpdata{:,'ROWMEAN'}      = nanmean(iv_ipDatam(:,:,i),2);
                        tmpdata{:,'ROWMEDIAN'}    = nanmedian(iv_ipDatam(:,:,i),2);
                        tmpdata{:,'ROWSTD'}       = nanstd(iv_ipDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEAN_AD'}   = mad(iv_ipDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEDIAN_AD'} = mad(iv_ipDatam(:,:,i),1,2);

                        %write index and sequence number into the table
                        tmpdata{:,'INDX'}      = vSizeIn;
                        tmpdata{:,'SEQN'}      = seqSize*i;

                        dataVipm = tmpdata;
                        for i = 2:seqN
                        %statistics for each column (along the 'y' axis) of the y component of the velocity vector
                        tmpdata{:,'COLMEAN'}      = nanmean(iv_ipDatam(:,:,i),1)';
                        tmpdata{:,'COLMEDIAN'}    = nanmedian(iv_ipDatam(:,:,i),1)';
                        tmpdata{:,'COLSTD'}       = nanstd(iv_ipDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEAN_AD'}   = mad(iv_ipDatam(:,:,i),0,1)';
                        tmpdata{:,'COLMEDIAN_AD'} = mad(iv_ipDatam(:,:,i),1,1)';

                        %statistics for each row (along the 'x' axis) of the y component of the velocity vector
                        tmpdata{:,'ROWMEAN'}      = nanmean(iv_ipDatam(:,:,i),2);
                        tmpdata{:,'ROWMEDIAN'}    = nanmedian(iv_ipDatam(:,:,i),2);
                        tmpdata{:,'ROWSTD'}       = nanstd(iv_ipDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEAN_AD'}   = mad(iv_ipDatam(:,:,i),0,2);
                        tmpdata{:,'ROWMEDIAN_AD'} = mad(iv_ipDatam(:,:,i),1,2);
                        %write index and sequence number into the table
                        tmpdata{:,'INDX'}      = vSizeIn;
                        tmpdata{:,'SEQN'}      = seqSize*i;

                        dataVipm = [dataVipm;tmpdata];
                        end
                        dataVipm.Properties.Description = 'Vy(x,y) of interpolated Velocity vector V(x,y)';


        
%% Prepare starting data set for the GUI
        currentSeqX     = dataSeqU;
        currentSeqY     = dataSeqV;
        currentDataX	= dataU;
        currentDataY	= dataV;
        currentData     = dataU;
        current_X_data	= iuData;
        current_Y_data	= ivData;
        % and the data for the set with means subtracted
        currentDataXm	= dataUm;
        currentDataYm	= dataVm;
        currentDatam     = dataUm;
        current_X_datam	= iuDatam;
        current_Y_datam	= ivDatam;
        
        
%% Here we construct the GUI

   %  Create and then hide the GUI as it is being constructed.
	set(0,'Units','pixels');
	scr  = get(0,'ScreenSize');
	main = figure('Visible','off','Position',[360,500,620,750]);
    boxp = figure('Visible','off','Position',[360,500,550,385]);
    figplot2 = figure('Visible','off','Position',[360,500,550,385]);
%    boxp = figure('Visible','off','Position',[360,500,550,385]);
%    row = figure('Visible','off','Position',[360,500,550,385]);
%    col = figure('Visible','off','Position',[360,500,550,385]);
%    main = figure('Visible','off','Position',[360,500,550,385]);
    
	set(main,'Renderer','OpenGL');
    position = get(main,'Position');
    outerpos = get(main,'OuterPosition');
    borders = outerpos - position;
    edge = -borders(1)/2;
    %mainpos = [0, 0, scr(3)/3, scr(4)];
    %set(main,'OuterPosition',mainpos);

   %  Construct the components of the main window.
        set(0,'CurrentFigure',main);
        hboxp = uicontrol('Style','pushbutton','String','Boxplot',...
          'Position',[500,220,70,25],...
          'Callback',{@boxplot_Callback});
        hfigplot2 = uicontrol('Style','pushbutton','String','Stat-plot',...
          'Position',[500,180,70,25],...
          'Callback',{@slider_call});
        hcontour = uicontrol('Style','pushbutton',...
          'String','Countour',...
          'Position',[500,135,70,25],...
          'Callback',{@contourbutton_Callback}); 
        htext = uicontrol('Style','text','String','Select Data',...
          'Position',[505,90,60,15]);
        hpopup = uicontrol('Style','popupmenu',...
          'String',{'X-vel','X-vel-interp','Y-vel','Y-vel-interp'},...
          'Position',[505,50,100,25],...
          'Callback',{@popup_menu_Callback});

        ha2 = axes('Units','Pixels','Position',[50,380,380,300]);
        ha = axes('Units','Pixels','Position',[50,30,380,300]);
        
  	if seqN > 1
           slider = uicontrol('Parent',main,'Style','slider',...
                'position',[360 260 90 20],...
                'min',1,'max',seqN,'val',1,...
                'sliderstep',[1 1]./(seqN - 1),...
                'callback',{@slider_call});
    end
    % create slider text
        slider_text = uicontrol('Style','text','String','Current sqn.',...
            'Position',[360 300 60 15]);
    %create slider value box
        slider_val = uicontrol('Style','text','String','1',...
            'Position',[360 285 60 15]);
    
	align([slider,hboxp,hfigplot2,hcontour,htext,hpopup,slider_text,slider_val]...
                ,'Center','None');
      
   % Initialize the GUI.
   % Change units to normalized so components resize 
   % automatically.
   set([main,ha,ha2,hboxp hfigplot2,hcontour,htext,hpopup,slider,slider_text,...
       slider_val], 'Units','normalized');
   
   %Create a plot in the axes.

   [XV YV] = meshgrid(xi, yi);
   quiver( XV, YV, current_X_data(:,:,1), current_Y_data(:,:,1), 'g' );
   grid on;
   axes(ha2)
   quiver( XV, YV, current_X_data(:,:,1), current_Y_data(:,:,1), 'g' );
   grid on;
   % Assign the GUI a name to appear in the window title.
   set(main,'Name','CTC MPIV data viewer')
   % Move the GUI to the center of the screen.
   %movegui(main,'center')
   % Make the GUI visible.
   set(main,'Visible','on');
   set(ha, 'Ydir','reverse');
   set(ha, 'XAxisLocation', 'Top');
   set(ha2, 'Ydir','reverse');
   set(ha2, 'XAxisLocation', 'Top');
   
   set(boxp,'Visible','on');
   set(figplot2,'Visible','on');
        
   %  Callbacks for simple_gui. These callbacks automatically
   %  have access to component handles and initialized data 
   %  because they are nested at a lower level.
 
   %  Pop-up menu callback. Read the pop-up menu Value property
   %  to determine which item is currently displayed and make it
   %  the current data.
      function popup_menu_Callback(src,eventdata) 
         % Determine the selected data set.
         get(src);
         str = get(src, 'String');
         val = get(src, 'Value');
         % Set current data to the selected data set.
         switch str{val};
         case 'X-vel' % User selects X velocity real data set.
            currentSeqX     = dataSeqU;
            currentSeqY     = dataSeqV;
            current_X_data  = iuData;
            current_Y_data  = ivData;
            currentDataX    = dataU;
            currentDataY    = dataV;
            currentData     = dataU;
            current_X_datam	= iuDatam;
            current_Y_datam	= ivDatam;
            currentDataXm	= dataUm;
            currentDataYm	= dataVm;
            currentDatam	= dataUm;            
         case 'X-vel-interp' % User selects interpolated data set.
            currentSeqX     = dataSeqUip;
            currentSeqY     = dataSeqVip;
            current_X_data  = iu_ipData;
            current_Y_data  = iv_ipData;
            currentDataX	= dataUip;
            currentDataY	= dataVip;
            currentData     = dataUip;
            current_X_datam	= iu_ipDatam;
            current_Y_datam	= iv_ipDatam;
            currentDataXm	= dataUipm;
            currentDataYm	= dataVipm;
            currentDatam	= dataUipm;            
         case 'Y-vel' % User selects Y velocity real data set
            currentSeqX     = dataSeqU;
            currentSeqY     = dataSeqV;
            current_X_data  = iuData;
            current_Y_data  = ivData;
            currentDataX	= dataU;
            currentDataY	= dataV;
            currentData     = dataV;
            current_X_datam  = iuDatam;
            current_Y_datam  = ivDatam;
            currentDataXm	= dataUm;
            currentDataYm	= dataVm;
            currentDatam     = dataVm;            
         case 'Y-vel-interp' % User selects interpolated data set.
            currentSeqX     = dataSeqUip;
            currentSeqY     = dataSeqVip;
            current_X_data  = iu_ipData;
            current_Y_data  = iv_ipData;
            currentDataX    = dataUip;
            currentDataY	= dataVip;
            currentData     = dataVip;
            current_X_datam  = iu_ipDatam;
            current_Y_datam  = iv_ipDatam;
            currentDataXm    = dataUipm;
            currentDataYm	= dataVipm;
            currentDatam     = dataVipm;            
         end
      end
  
   % Push button callbacks. Each callback plots current_data in
   % the specified plot type.
 
   function boxplot_Callback(src,eventdata) 
   % Boxplot X and Y velocity statistics
        for j=1:seqN
            tmpx = current_X_data(:,:,j);
            tmpy = current_Y_data(:,:,j);
            boxX(:,j) = reshape(tmpx',1,numel(tmpx))';
            boxY(:,j) = reshape(tmpy',1,numel(tmpy))';
            dataSeqVip{:,'MEDIAN'}
            %tmpx2 = current_X_data(:,:,j) - iu_ipDataMed(j);
            %tmpy2 = current_Y_data(:,:,j) - iv_ipDataMed(j);
            tmpx2 = current_X_data(:,:,j) - currentSeqX.MEDIAN(j);
            tmpy2 = current_Y_data(:,:,j) - currentSeqY.MEDIAN(j);
            boxX2(:,j) = reshape(tmpx2',1,numel(tmpx2))';
            boxY2(:,j) = reshape(tmpy2',1,numel(tmpy2))';
        end
        set(0,'CurrentFigure',boxp);
        
        subplot(3,2,1);
        boxplot(boxX);
        title('X-displacement stats of all sequences','Color','r');
        xlabel('image pair');
        ylabel('displacement in pixels');
        
        subplot(3,2,2);
        boxplot(boxY);
        title('Y-displacement stats of all sequences','Color','r');
        xlabel('image pair');
        ylabel('displacement in pixels');
        
        subplot(3,2,3);
        boxplot(boxX2);
        title('X-displacement stats with median subtracted','Color','r');
        xlabel('image pair');
        ylabel('displacement in pixels');
        
        subplot(3,2,4)
        boxplot(boxY2);
        title('Y-displacement stats with median subtracted','Color','r');
        xlabel('image pair');
        ylabel('displacement in pixels');
        
        subplot(3,2,5);
        plot(currentSeqX.MEDIAN_AD,'ro');
        title('X-displacement avg median abs deviation','Color','r');
        xlabel('image pair');
        ylabel('avg displacement in pixels');
        
        subplot(3,2,6)
        plot(currentSeqY.MEDIAN_AD,'ro');
        title('Y-displacement avg median abs deviation','Color','r');
        xlabel('image pair');
        ylabel('avg displacement in pixels');

   end
 
 
   function contourbutton_Callback(src,eventdata) 
   % Display contour plot of the currently selected data.
        quiver( XV, YV, current_X_data(:,:,4), current_Y_data(:,:,4), 'r' );
        grid on
        set(ha, 'Ydir','reverse');
        set(ha, 'XAxisLocation', 'Top');
   end

   function slider_call(src,eventdata) 
   % Display contour plot of the currently selected data.
        set(0,'CurrentFigure',main);
        value = ceil(get(slider,'val'));
        set(slider_val, 'String', num2str(value));
        %pause(0.3)
        set(gcf,'CurrentAxes',ha);
%        quiver( XV, YV, current_X_data(:,:,value) - currentSeqX.MEAN(value),...
%            current_Y_data(:,:,value) - currentSeqY.MEAN(value), 'b' );
        %disp(current_Y_data(:,:,value) - currentSeqY.MEDIAN(value))
        quiver( XV, YV, current_X_datam(:,:,value),...
            current_Y_datam(:,:,value), 'r' );
        grid on

        set(ha, 'Ydir','reverse');
        set(ha, 'XAxisLocation', 'Top');
        set(ha, 'Xlim',[0 max(xi)+.1*max(xi)]);
        set(ha, 'Ylim',[-50 max(yi)]+.15*max(yi));
        title('MPIV Vector field, mean subtracted','Color','r');
        
        set(gcf,'CurrentAxes',ha2);
        %axes(ha2);
        quiver(XV, YV, current_X_data(:,:,value), current_Y_data(:,:,value), 'b' );
        grid on
        set(ha2, 'Ydir','reverse');
        set(ha2, 'XAxisLocation', 'Top');
        set(ha2, 'Xlim',[0 max(xi)+.1*max(xi)]);
        set(ha2, 'Ylim',[-50 max(yi)]+.15*max(yi));
        title('MPIV Vector field','Color','r');
        
        % Central values for rows and cols. for each sequence
        
        set(0,'CurrentFigure',figplot2);
        %str = get(hpopup, 'String');
        %val = get(hpopup, 'Value');
        %set(gcf, 'Name', str{val});
        % currentData.Properties.Description
        set(gcf, 'Name', currentData.Properties.Description);
        
        subplot(2,2,1);
        errorbar(currentData.COLMEDIAN(currentData.SEQN==value),...
            currentData.COLMEDIAN_AD(currentData.SEQN==value), 'bo-');
        hold on
        errorbar(currentDatam.COLMEDIAN(currentDatam.SEQN==value),...
            currentDatam.COLMEDIAN_AD(currentDatam.SEQN==value), 'ro-');
        hold off
        title('Median displacement in each column','Color','r');
        xlabel('column number');
        ylabel(currentData.Properties.VariableUnits{4});
        
        subplot(2,2,2);
        errorbar(currentData.ROWMEDIAN(currentData.SEQN==value),...
            currentData.ROWMEDIAN_AD(currentData.SEQN==value), 'bo-');
        hold on
        errorbar(currentDatam.ROWMEDIAN(currentDatam.SEQN==value),...
            currentDatam.ROWMEDIAN_AD(currentDatam.SEQN==value), 'ro-');
        hold off
        title('Median displacement in each row','Color','r');
        xlabel('row number');
        ylabel(currentData.Properties.VariableUnits{7});
        
        subplot(2,2,3);
        stem(currentData.COLMEDIAN_AD(currentData.SEQN==value),'go-');
        title(currentData.Properties.VariableDescriptions{10},'Color','r');
        xlabel('column number');
        ylabel(currentData.Properties.VariableUnits{10});
        
        subplot(2,2,4)
        stem(currentData.ROWMEDIAN_AD(currentData.SEQN==value),'go-');
        title(currentData.Properties.VariableDescriptions{12},'Color','r');
        xlabel('row number');
        ylabel(currentData.Properties.VariableUnits{12});
        
   end
end 