function dy=kineticodefile(t,y,param)

%--------------------------------------------------------------------------
% Initialization
%--------------------------------------------------------------------------
% Import parameters from main programme
Vm=param.Vm;        % [dm^3] maternal volume
Vs=param.Vs;        % [dm^3] syncytiotrophoblast volume
Vf=param.Vf;        % [dm^3] fetal volume
A=param.A;          % [dm^2] exchange area, set to 1 at the moment

naa=param.naa;      % number of amino acids
indm=param.indm;    % index for maternal amino acids
inds=param.inds;    % index for syncytiotrophoblast amino acids
indf=param.indf;    % index for fetal amino acids

ym=y(indm);         % [micro molar] maternal amino acid concentrations
ys=y(inds);         % [micro molar] syncytiotrophoblast amino acid concentrations
yf=y(indf);         % [micro molar] fetal amino acid concentrations

dym=zeros(naa,1);   % [micro molar/min] rate of change maternal amino acids
dys=zeros(naa,1);   % [micro molar/min] rate of change syncytiotrophoblast amino acids
dyf=zeros(naa,1);   % [micro molar/min] rate of change fetal amino acids

Cinm=param.Cinm;    % [micro mol/l] maternal artery concentrations fetal tracer
Fm=param.Fm;        % [dm^3/min] Constant maternal flow 


Cinf=param.Cinf;    % [micro mol/l] fetal inlet concentrations
Ff=param.Ff;        % [dm^3/min]Constant fetal flow



%--------------------------------------------------------------------------
% Transport maternal - syncytiotrophoblast (MVM)
%--------------------------------------------------------------------------

% --- Maternal flow ---
Qabs= Fm*(Cinm-ym);    % [micro mol/min] net influx during timestep  
dym=dym+Qabs/Vm;       % [micro molar/min] concentration change due to influx


% --- Accumulative transporter ---
indx=param.MVM_accum_aaindx; % amino acids to include
if indx>0
    xt=param.MVM_accum_xt;
    beta=param.MVM_accum_beta;
    pot=param.MVM_accum_pot;
    
    F=9.64853399e4;   % faraday Constant (C/mol)
    R=8.314;          % gas constant (C V/K/mol)
    T=310;            % absolute temp: RT= 298K, BT= 310K
    FRT=F/(R*T)*1e-3; % F/RT  (1/mV)
    z=0;   
    zna=1;
    
    
    k1=param.MVM_accum_k1*exp((beta-1)*(zna+z)*FRT*pot);
    k1b=param.MVM_accum_k1b*exp(beta*(zna+z)*FRT*pot);
    k2=param.MVM_accum_k2;
    k2b=param.MVM_accum_k2b;
    
    
    Ko=param.MVM_accum_Ko;
    KNao=param.MVM_accum_KNao;
    Ki=param.MVM_accum_Ki;
    KNai=param.MVM_accum_KNai;
    
    CNa1=param.MVM_accum_CNa1;
    CNa2=param.MVM_accum_CNa2;
    
    Ctot1=sum(ym(indx));
    Ctot2=sum(ys(indx));
    Denom=CNa1*CNa2*(k1*Ctot1*(Ctot2+Ki)+k1b*Ctot2*(Ctot1+Ko))+CNa1*Ki*KNai*Ctot1*(k1+k2b)+CNa2*Ko*KNao*Ctot2*(k1b+k2)+Ki*Ko*(CNa1*k2b*KNai+CNa2*k2*KNao)+(k2+k2b)*Ki*Ko*KNai*KNao;
    
    for iaa=indx
        rindx=setdiff(indx,iaa); % get list of the rest of the relevant amino acids, i.e. different from the current one
        
        Ca1=ym(iaa);
        Ca2=ys(iaa);
        if rindx~=0
            Cb1=ym(rindx);
            Cb2=ys(rindx);
        else
            Cb1=0; Cb2=0;
        end
                
        Jacc=xt/Denom*(CNa1*CNa2*k1*k1b*(Ca1*Cb2-Ca2*Cb1)+Ca1*CNa1*k1*k2b*Ki*KNai-Ca2*CNa2*k1b*k2*Ko*KNao);        
        
        dys(iaa)=dys(iaa)+ Jacc.*A./Vs;
        dym(iaa)=dym(iaa)- Jacc.*A./Vm;
    end
end


% --- Exchanger ---
indx=param.MVM_exchange_aaindx;
if indx>0
    Dxt=param.MVM_exchange_Dxt;   % [micro mol/(min dm^2)] transport rate
    Ko=param.MVM_exchange_Ko;
    Ki=param.MVM_exchange_Ki;
    
    Ctot1=sum(ym(indx));
    Ctot2=sum(ys(indx));
    
    for iaa=indx
        Ca1=ym(iaa);
        Ca2=ys(iaa);
        
        rindx=setdiff(indx,iaa);  % get list of the rest of the relevant amino acids, i.e. different from the current one
        Cr1=sum(ym(rindx));
        Cr2=sum(ys(rindx));
        
        Qhlp=Dxt*(Ca1*Cr2 - Ca2*Cr1)/( Ki(iaa)*Ctot1+Ko(iaa)*Ctot2 + 2*Ctot1*Ctot2);
        
        dym(iaa)=dym(iaa)-Qhlp*A/Vm;
        dys(iaa)=dys(iaa)+Qhlp*A/Vs;
    end
    
    
end
 
%--------------------------------------------------------------------------
% Transport syncytiotrophoblast - syncytiotrophoblast to fetal (BM)
%--------------------------------------------------------------------------

% --- Fetal flow ---
Qabs= Ff*(Cinf-yf);     % [micro mol/min] net influx during timestep 
dyf=dyf+Qabs/Vf;        % [micro molar/min] concentration change due to influx


% --- Exchanger ---
indx=param.BM_exchange_aaindx;
if indx>0
    Dxt=param.BM_exchange_Dxt;
    Ko=param.BM_exchange_Ko;
    Ki=param.BM_exchange_Ki;
    
    Ctot1=sum(ys(indx));
    Ctot2=sum(yf(indx));
    
    for iaa=indx
        Ca1=ys(iaa);
        Ca2=yf(iaa);
        
        rindx=setdiff(indx,iaa); % get list of the rest of the relevant amino acids, i.e. different from the current one
        Cr1=sum(ys(rindx));
        Cr2=sum(yf(rindx));
        
        Qhlp=Dxt*(Ca1*Cr2 - Ca2*Cr1)/( Ko(iaa)*Ctot1+Ki(iaa)*Ctot2 + 2*Ctot1*Ctot2);
        
        dys(iaa)=dys(iaa)-Qhlp*A/Vs;
        dyf(iaa)=dyf(iaa)+Qhlp*A/Vf;
    end
    
end


% --- Facilitated transporter ---
indx=param.BM_efflux_aaindx;             % amino acids to include
if indx>0
    Dxt=param.BM_efflux_Dxt;                 % [micro mol/(min dm^2)] transport rate efflux transporter
    Ko=param.BM_efflux_Ko;
    Ki=param.BM_efflux_Ki;
    
    Ctot1=sum(ys(indx));
    Ctot2=sum(yf(indx));
    
    Ca1=ys(indx);
    Ca2=yf(indx);
    Jeff= Dxt./2.*(Ca1./(Ctot1+Ki(indx)')-Ca2./(Ctot2+Ko(indx)'));
    dyf(indx)=dyf(indx)+ Jeff.*A./Vf;
    dys(indx)=dys(indx)- Jeff.*A./Vs;
end


%--------------------------------------------------------------------------
% passive paracellular diffusion, NOT USED
%--------------------------------------------------------------------------
% Rd=param.Paracellflux_Rd;
% 
% Qhlp= Rd * ( ym-yf ); % [micro mol / (min dm^2)]% works on all solutes in parallel, no competition
% 
% dym=dym-Qhlp*A/Vm;
% dyf=dyf+Qhlp*A/Vf;


% --- Collect output ---
dy=[dym;dys;dyf]; 




