%**************************************************************************
% Programme for calculating placental amino acid transfer
% -------------------------------------------------------
%           N. Panitchob & B.G. Sengers 2015
%
% A detailed description of the model is provided in the accompanying
% publication and in Panitchob (2015), Computational modelling of amino 
% acid transfer interactions in the placenta, PhD thesis, University of 
% Southampton, UK
%
%**************************************************************************
clear all

% Load problem definition
iperf=1;                        % problem number
str=['probdef',num2str(iperf)];
eval(str);  

% --- Output folder ---
%dirvar='RESULTS\'; % output folder
%[s, w] = dos(['dir ',dirvar]);
%if s==1
% str=['!mkdir ',dirvar];
% eval(str);
%end


% --- Initialize input parameters -----------------------------------------
param.Vm=Vm;
param.Vs=Vs;
param.Vf=Vf;
param.A=1;  % area set to 1, so Qmax is total mol/min instead of  mol/area per minute see probdef and kineticodefile

naa=length(aa);     % number of amino acids
indm=1:naa;         % index for maternal amino acids
inds=naa+1:2*naa;   % index for syncytiotrophoblast amino acids
indf=2*naa+1:3*naa; % index for fetal amino acids

param.naa=naa;
param.indm=indm;
param.inds=inds;
param.indf=indf;


 %--- Maternal inlet flow and concentrations ------------------------------
 Cinm=zeros(naa,1);               % maternal inlet concentrations [micro mol/l] 
 for i=1:naa                
  Cinm(i)=aa(i).cinmaternal;      % maternal inlet concentration [micro mol/l]     
 end
 param.Cinm=Cinm;
 param.Fm=Fm;                     % Constant maternal flow 
 
 
%--- Fetal inlet concentrations -------------------------------------------
Cinf=zeros(naa,1);                % fetal inlet concentration 
for i=1:naa                
 Cinf(i)=aa(i).cinfetal;          % fetal inlet concentration [micro mol/l]     
end
param.Cinf=Cinf;
param.Ff=Ff;                      % Constant fetal flow


%--- Initial concentrations ---
ym0=[];       
ys0=[];   
yf0=[];  
for i=1:naa  
 ym0=[ym0;aa(i).c0maternal];      % [micro molar] initial maternal amino acid concentrations
 ys0=[ys0;aa(i).c0syncitio];      % [micro molar] initial syncytiotrophoblast amino acid concentrations
 yf0=[yf0;aa(i).c0fetal];         % [micro molar] initial fetal amino acid concentrations
end


% --- Compute results, time integration ---
y0= [ym0;ys0;yf0]; 
tspan=[0,Tend];

odeoptions=odeset('Reltol',0.00001,'Abstol',1e-5);
[tout, yout]=ode45(@kineticodefile,tspan,y0,odeoptions,param);  



% --- PLOTTING ------------------------------------------------------------

% --- Rearrange output ---
ymtot=yout(:,indm);
ystot=yout(:,inds);
yftot=yout(:,indf);


% --- Fetal venous-arterial difference ---
Nt=length(tout)-1; 
yfN=yftot(Nt+1,:);
dVA=yfN-Cinf';


% --- Plot Time course of concentration in each compartment ---
figure
clf
fontsiz=10; 

% --- Maternal compartment ---
subplot(3,1,1)
str=['plot('];
lstr=['legend('];
aalist=1:naa; 
for i=aalist
 hlpstr=['tout,ymtot(:,',num2str(i),'),''',aa(i).linestyle,''''];
 str=[str,hlpstr]; 
 lstr=[lstr,'''',aa(i).name,''''];
 if i<aalist(length(aalist)), 
     str=[str,',']; 
     lstr=[lstr,',']; 
 end;
end
str=[str,')'];
lstr=[lstr,')'];
eval(str)
eval(lstr)
h=title('Maternal compartment');
set(h,'Fontsize',fontsiz)
h=xlabel('time (min)');
set(h,'Fontsize',fontsiz)
h=ylabel('concentration (\muM)');
set(h,'Fontsize',fontsiz)
set(gca,'Fontsize',fontsiz)    

% --- Syncytiotrophoblast compartment ---
subplot(3,1,2)
str=['plot('];
lstr=['legend('];
aalist=1:naa; 
for i=aalist
 hlpstr=['tout,ystot(:,',num2str(i),'),''',aa(i).linestyle,''''];
 str=[str,hlpstr]; 
 lstr=[lstr,'''',aa(i).name,''''];
 if i<aalist(length(aalist)), 
     str=[str,',']; 
     lstr=[lstr,',']; 
 end;
end
str=[str,')'];
lstr=[lstr,')'];
eval(str)
eval(lstr)
h=title('Syncytiotrophoblast compartment');
set(h,'Fontsize',fontsiz)
h=xlabel('time (min)');
set(h,'Fontsize',fontsiz)
h=ylabel('concentration (\muM)');
set(h,'Fontsize',fontsiz)
set(gca,'Fontsize',fontsiz)    

% --- Fetal compartment ---
subplot(3,1,3)
str=['plot('];
lstr=['legend('];
aalist=1:naa; 
for i=aalist
 hlpstr=['tout,yftot(:,',num2str(i),'),''',aa(i).linestyle,''''];
 str=[str,hlpstr]; 
 lstr=[lstr,'''',aa(i).name,''''];
 if i<aalist(length(aalist)), 
     str=[str,',']; 
     lstr=[lstr,',']; 
 end;
end
str=[str,')'];
lstr=[lstr,')'];
eval(str)
eval(lstr)
h=title('Fetal compartment');
set(h,'Fontsize',fontsiz)
h=xlabel('time [min]');
set(h,'Fontsize',fontsiz)
h=ylabel('concentration (\muM)');
set(h,'Fontsize',fontsiz)
set(gca,'Fontsize',fontsiz)    


