#include "footbot_foraging.h"
#include <argos3/core/utility/configuration/argos_configuration.h>
#include <argos3/core/utility/math/vector2.h>
#include <argos3/core/utility/logging/argos_log.h>
#include <algorithm>


/* ======================= READ PARAMETERS ================== */

CFootBotForaging::DiffusionParams::DiffusionParams() : GoStraightAngleRange(CRadians(-1.0f), CRadians(1.0f)) {}

void CFootBotForaging::DiffusionParams::Init(TConfigurationNode& t_node) {
   try {
      CRange<CDegrees> cGoStraightAngleRangeDegrees(CDegrees(-10.0f), CDegrees(10.0f));
      GetNodeAttribute(t_node, "go_straight_angle_range", cGoStraightAngleRangeDegrees);
      GoStraightAngleRange.Set(ToRadians(cGoStraightAngleRangeDegrees.GetMin()),
                               ToRadians(cGoStraightAngleRangeDegrees.GetMax()));
      GetNodeAttribute(t_node, "delta", Delta);
   }
   catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initializing controller diffusion parameters.", ex);
   }
}

void CFootBotForaging::WheelTurningParams::Init(TConfigurationNode& t_node) {
   try {
      CDegrees cAngle;
      GetNodeAttribute(t_node, "hard_turn_angle_threshold", cAngle);
      hardTurnOnAngleThreshold = ToRadians(cAngle);
      GetNodeAttribute(t_node, "soft_turn_angle_threshold", cAngle);
      softTurnOnAngleThreshold = ToRadians(cAngle);
      GetNodeAttribute(t_node, "no_turn_angle_threshold", cAngle);
      noTurnAngleThreshold = ToRadians(cAngle);
      GetNodeAttribute(t_node, "max_speed", maxSpeed);
   }
   catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initializing controller wheel turning parameters.", ex);
   }
}

void CFootBotForaging::BodyParams::Init(TConfigurationNode& t_node) {
   try {
	   uint linuxBased;
	   GetNodeAttribute(t_node, "linuxBased", linuxBased);
	   isLinuxBased = (linuxBased == 1);
	   //LOG << "Liux based body? " << isLinuxBased << std::endl;
	   if (isLinuxBased) {
		  //-- on iridis
		  groundSensorFLId=0;
		  groundSensorBLId=1;
		  groundSensorFRId=3;
		  groundSensorBRId=2;
	   } else {
		  //-- on a mac
		  groundSensorFLId=0;
		  groundSensorBLId=3;
		  groundSensorFRId=1;
		  groundSensorBRId=2;
	   }
   }
   catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initializing controller wheel turning parameters.", ex);
   }
}


/**
 * Initialisation of set / explored parameters
 */
void CFootBotForaging::Parameters::Init(TConfigurationNode& t_node) {
   try {
	   GetNodeAttribute(t_node, "doesRelEnergyEfficiency", doesRelEnergyEfficiency);
	   GetNodeAttribute(t_node, "cartCapacity", cartCapacity);
	   GetNodeAttribute(t_node, "unloadPerPellet", unloadPerPellet);
	   GetNodeAttribute(t_node, "maxInteractionTime", maxInteractionTime);
	   GetNodeAttribute(t_node, "minObserverTime", minObserverTime);
	   GetNodeAttribute(t_node, "scoutProb", scoutProb);
	   GetNodeAttribute(t_node, "maxTimeToScout", maxTimeToScout);
	   GetNodeAttribute(t_node, "searchRadius", searchRadius);
	   GetNodeAttribute(t_node, "searchTime", searchTime);
	   GetNodeAttribute(t_node, "restingProb", restingProb);
	   GetNodeAttribute(t_node, "wakingProb", wakingProb);
	   GetNodeAttribute(t_node, "signalDistance", signalDistance);
	   GetNodeAttribute(t_node, "minReactivationProb", minReactivationProb);
	   GetNodeAttribute(t_node, "maxReactivationProb", maxReactivationProb);


	   GetNodeAttribute(t_node, "maxUnloadTime", maxUnloadTime);
	   GetNodeAttribute(t_node, "stopSignalProbTrembleDancer", stopSignalProbTrembleDancer);
	   GetNodeAttribute(t_node, "stopSignalProbNonTrebleDancer", stopSignalProbNonTrebleDancer);
	   GetNodeAttribute(t_node, "unloadTimeThreshold", unloadTimeThreshold);
	   GetNodeAttribute(t_node, "trembleSignalDistance", trembleSignalDistance);

	   GetNodeAttribute(t_node, "maxEncounters", maxEncounters);
	   GetNodeAttribute(t_node, "encounterSignalDistance", encounterSignalDistance);
	   GetNodeAttribute(t_node, "samplingEncountersTime", samplingEncountersTime);
	   GetNodeAttribute(t_node, "wakeSignalTimeMultiplier", wakeSignalTimeMultiplier);

	   GetNodeAttribute(t_node, "beggingSignalDistance", beggingSignalDistance);
	   GetNodeAttribute(t_node, "beggingProb", beggingProb);


   } catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initializing parameters.", ex);
   }

}

/**
 * Initialisation of set / explored parameters
 */
void CFootBotForaging::Behaviour::Init(TConfigurationNode& t_node) {
   try {
	   GetNodeAttribute(t_node, "useModelB", useModelB);
	   GetNodeAttribute(t_node, "useModelBNonSocial", useModelBNonSocial);
	   GetNodeAttribute(t_node, "useModelC", useModelC);
	   GetNodeAttribute(t_node, "useModelD", useModelD);
   } catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initializing behaviour.", ex);
   }
}





/*================================== INITIALISATION =================== */



CFootBotForaging::CFootBotForaging() :
   wheels(NULL),
   leds(NULL),
   collisionAvoidingAngle(0),
   rangeAndBearingActuator(NULL),
   rangeAndBearingSensor(NULL),
   proximitySensor(NULL),
   lightSensor(NULL),
   groundSensor(NULL),
   randNumGen(NULL) {}


void CFootBotForaging::Init(TConfigurationNode& t_node) {
   try {
	   //-- init sensors and actuators
	   wheels    = GetActuator<CCI_DifferentialSteeringActuator>("differential_steering");
	   leds      = GetActuator<CCI_LEDsActuator                >("leds"                 );
	   rangeAndBearingActuator      = GetActuator<CCI_RangeAndBearingActuator     >("range_and_bearing"    );
	   rangeAndBearingSensor      = GetSensor  <CCI_RangeAndBearingSensor       >("range_and_bearing"    );
	   proximitySensor = GetSensor  <CCI_FootBotProximitySensor      >("footbot_proximity"    );
	   lightSensor     = GetSensor  <CCI_FootBotLightSensor          >("footbot_light"        );
	   groundSensor    = GetSensor  <CCI_FootBotMotorGroundSensor    >("footbot_motor_ground" );
	   steeringSensor    = GetSensor  <CCI_DifferentialSteeringSensor    >("differential_steering" );

	   //-- parse XML parameters
	   diffustionParams.Init(GetNode(t_node, "diffusion"));
	   wheelParams.Init(GetNode(t_node, "wheel_turning"));
	   bodyParams.Init(GetNode(t_node, "body"));
	   parameters.Init(GetNode(t_node, "parameters"));
	   behaviour.Init(GetNode(t_node, "behaviour"));
   }
   catch(CARGoSException& ex) {
	   THROW_ARGOSEXCEPTION_NESTED("Error initializing the foot-bot foraging controller for robot \"" << GetId() << "\"", ex);
   }

   //-- other stuff
   randNumGen = CRandom::CreateRNG("argos");
   gslRandNumGen = gsl_rng_alloc(gsl_rng_default);
   Reset();
}



void CFootBotForaging::Reset() {
	scoutingStatus=2;
	reportedWaggleDanceTime=-2; // void waggle dance time
	neighbourhoodSearchStatus=2;
	SetState(Memory::STATE_OBSERVING);
	memory.Reset();
	cart.Reset();
	begRelEESent = false;
	begEESent = false;
	begAngleToTargetSent = false;
	begDistanceToTargetSent = false;
	begToIdSent = false;


	/*if (this->GetId() == "robot0" || this->GetId() == "robot1" || this->GetId() == "robot2" || this->GetId() == "robot3" || this->GetId() == "robot4") {
		memory.targetDeposit.isNull = false;
		memory.targetDeposit.energyEfficiency = 100;
		memory.targetDeposit.relEnergyEfficiency = 1;
		memory.targetDeposit.relLocation = CVector2(-3,-2);
		//memory.targetDeposit.relLocation = CVector2(-5,0);
		behaviour.CalculateReactivationProb(memory.targetDeposit.relEnergyEfficiency,parameters.minReactivationProb,parameters.maxReactivationProb);
		behaviour.CalculateWaggleDaceTime(memory.targetDeposit.relEnergyEfficiency,parameters.maxInteractionTime);
		memory.timeWaggleDancing = 0;
		memory.timeTillBeggingLeft = 0;
		SetState(Memory::STATE_WAGGLE_DANCING);



		//memory.timeUnloading = 170;
		//behaviour.CalculateTrembleDaceTime(memory.timeUnloading,parameters.maxUnloadTime,parameters.unloadTimeThreshold,parameters.maxInteractionTime);
		//behaviour.UpdateWaggleDanceTimeByUnloadTime(memory.timeUnloading,parameters.maxUnloadTime,parameters.unloadTimeThreshold);
		//SetState(Memory::STATE_FORAGING);
	}*/

}

CFootBotForaging::Memory::Memory() : location(), state() {
	targetDeposit = Deposit();
	randWalkTurnCounter = 0;
	randWalkTurnLeft = false;
	timeTillNextSample = 0;
	timeScouting = 0;
	timeObserving = 0;
	numDepositsScouted = 0;
	depositSearchTimeLeft = 0;
	timeWaggleDancing = 0;
	timeCounter = 0;
	isOverExited = false;
	timeTillBeggingLeft = 0;
	overExcitementTimeLeft = 0;
	lightNavigationAngleDecided = false;
}

void CFootBotForaging::Memory::Reset() {
   randWalkTurnCounter = 0;
   targetDeposit = Deposit();
}

void CFootBotForaging::Cart::Reset() {
	amount = 0;
}

/*======================================= UPDATE LOOP ======================== */

/**
 * Main update loop function
 */
void CFootBotForaging::ControlStep() {
	ReadCurrentLocation();

	if (memory.overExcitementTimeLeft > 0) {
		memory.isOverExited = true;
		memory.overExcitementTimeLeft--;
	} else {
		memory.isOverExited = false;
	}

	switch(memory.state) {
		case Memory::STATE_LOADING:
		case Memory::STATE_LOADING_SAMPLE: {
			//PerformLoad(); //-- called from foraging_loop_functions
			break;
		}
		case Memory::STATE_UNLOADING:{
			//PerformUnload(); //-- called from foraging_loop_functions
			break;
		}
		case Memory::STATE_RESTING: {
			Rest();
			break;
		}
		case Memory::STATE_OBSERVING: {
			Observe();
			break;
		}
		case Memory::STATE_FORAGING: {
			Forage();
			break;
		}
		case Memory::STATE_SCOUTING: {
			Scout();
			break;
		}
		case Memory::STATE_RETURN_TO_BASE_AFTER_SCOUTING:
		case Memory::STATE_RETURN_TO_BASE_AFTER_FORAGING:
		case Memory::STATE_RETURN_TO_BASE_UNSUCCESSFUL: {
			ReturnToBase();
			break;
		}
		case Memory::STATE_WAGGLE_DANCING: {
			PerformWaggleDance();
			break;
		}
		case Memory::STATE_TREMBLE_DANCING: {
			PerformTrembleDance();
			break;
		}
		case Memory::STATE_SAMPLING_ENCOUNTERS: {
			SampleEncounters();
			break;
		}
		case Memory::STATE_WAKING_OTHERS: {
			PerformWaggleDance();
			break;
		}
		default: {
			//////////////LOGERR << "We can't be here, there's a bug!" << std::endl;
		}
	}

	//========== MODEL D ============
	//-- broadcast current target deposit
	if (behaviour.useModelD && memory.location <= Memory::LOCATION_DANCE_FLOOR && memory.state != Memory::STATE_FORAGING) {
		BroadcastTargetInfo();
	}
	//===============================

	//-- make sure that stays within bounds
	if (memory.relBaseLocation.Length() > 20) {
		NavigateToLight();
	}

	//-- update odometry to target
	if (!memory.targetDeposit.isNull) {
		UpdateOdometryToLocation(memory.targetDeposit.relLocation);
	}
	//-- update odomerty to base
	UpdateOdometryToLocation(memory.relBaseLocation);

	//-- set front LED color based on state
	switch (memory.state) {
		case Memory::STATE_OBSERVING:
			leds->SetAllColors(CColor::ORANGE);
			break;
		case Memory::STATE_FORAGING:
			leds->SetAllColors(CColor::GREEN);
			break;
		case Memory::STATE_SCOUTING:
			leds->SetAllColors(CColor::MAGENTA);
			break;
		case Memory::STATE_RETURN_TO_BASE_AFTER_FORAGING:
		case Memory::STATE_RETURN_TO_BASE_AFTER_SCOUTING:
		case Memory::STATE_RETURN_TO_BASE_UNSUCCESSFUL:
			leds->SetAllColors(CColor::BLUE);
			break;
		case Memory::STATE_LOADING:
		case Memory::STATE_UNLOADING:
			leds->SetAllColors(CColor::CYAN);
			break;
		case Memory::STATE_WAGGLE_DANCING:
			leds->SetAllColors(CColor::WHITE);
			if ((memory.timeCounter/10)%2 == 0 && behaviour.willBeg) {
				leds->SetAllColors(CColor::YELLOW);
			}
			break;
		case Memory::STATE_TREMBLE_DANCING:
			leds->SetAllColors(CColor::RED);
			break;
		case Memory::STATE_SAMPLING_ENCOUNTERS:
			leds->SetAllColors(CColor::CYAN);
			break;
		case Memory::STATE_WAKING_OTHERS:
			//leds->SetAllColors(CColor::MAGENTA);
			leds->SetAllColors(CColor::WHITE);
			if ((memory.timeCounter/10)%2 == 0 ) {
				leds->SetAllColors(CColor::BLACK);
			}
			break;
		default:
			leds->SetAllColors(CColor::BLACK);
			break;
	}
	if (memory.isOverExited && (memory.timeCounter/10)%2 == 0 ) {
		leds->SetAllColors(CColor::BLACK);
	}

	//-- indicate how much is in cart
	for (uint i=5; i<=8; i++) {
		if (i==5 && cart.amount >= 0.25) {
			leds->SetSingleColor(i,CColor::RED);
		} else if (i==6 && cart.amount >= 0.5) {
			leds->SetSingleColor(i,CColor::RED);
		} else if (i==7 && cart.amount >= 0.75) {
			leds->SetSingleColor(i,CColor::RED);
		} else if (i==8 && cart.amount >= 1.0) {
			leds->SetSingleColor(i,CColor::RED);
		} else {
			leds->SetSingleColor(i,CColor::BLACK);
		}
	}

	////////////////LOGERR << "STATE: " << memory.state << std::endl;
	////////////////LOGERR << "In cart: " << cart.amount << std::endl;
	memory.timeCounter++;


}

void CFootBotForaging::SetState(Memory::STATE state_, const bool& isNewState_) {
	if (isNewState_) {
		rangeAndBearingActuator->ClearData();
		rangeAndBearingActuator->SetData(0,SIGNAL_NONE);
	}
	memory.state = state_;
	switch (memory.state) {
		case Memory::STATE_UNLOADING:
			memory.timeTillNextUnload = 10;
			memory.timeUnloading = 0;
			unloadAmount = cart.amount;
			unloadNetAmount = cart.amount*cart.resourceQuality;
			if (isNewState_) {
				memory.isOverExited = false;
				behaviour.willWaggleDance = false;
			}
			break;
		case Memory::STATE_LOADING:
		case Memory::STATE_LOADING_SAMPLE:
			if (isNewState_) {
				isNewLoad = true;
			}
			break;
		case Memory::STATE_RESTING:
			memory.targetDeposit.isNull = true;
			if (isNewState_) {
				memory.isOverExited = false;
				memory.timeSpentGettingToMiddle = 0;
			}
			break;
		case Memory::STATE_SCOUTING:
			if (isNewState_) {
				memory.targetDeposit.isNull = true;
				memory.numDepositsScouted = 0;
				memory.timeTillNextSample = 0;
				memory.timeScouting = 0;
				memory.timeSinceEnteredUnloadingBay = 0;
			}
			break;
		case Memory::STATE_WAGGLE_DANCING:
			waggleToIdSent = false;
			waggleRelEESent = false;
			waggleEESent = false;
			waggleAngleToTargetSent = false;
			waggleDistanceToTargetSent = false;
			begRelEEReceived = false;
			begToIdReceived = false;
			begEEReceived = false;
			begAngleToTargetReceived = false;
			begDistanceToTargetReceived = false;
			receivingFromId = -1;
			advertisedDeposit.isNull = true;
			if (isNewState_) {
				memory.timeWaggleDancing = 0;
				memory.wasInterruptedDuringWD = false;
				if (behaviour.useModelD) {
					if (randNumGen->Uniform(CRange<Real>(0,1)) < parameters.beggingProb) {
						behaviour.willBeg = true;
					} else {
						behaviour.willBeg = false;
					}
				}
			}
			break;
		case Memory::STATE_OBSERVING:
			waggleToIdReceived = false;
			waggleRelEEReceived = false;
			waggleEEReceived = false;
			waggleAngleToTargetReceived = false;
			waggleDistanceToTargetReceived = false;
			receivingFromId = -1;
			advertisedDeposit.isNull = true;
			if (isNewState_) {
				memory.timeObserving = 0;
				reportedWaggleDanceTime=-1;
				memory.targetDeposit.isNull = true;
				if (behaviour.useModelB) {
					behaviour.CalculateStopSignalProb(memory.didTrembleDance,parameters.stopSignalProbTrembleDancer, parameters.stopSignalProbNonTrebleDancer);
					////////////////LOGERR << this->GetId() << " end of TD, SS prob: " << behaviour.stopSignalProb << std::endl;
				}
			}
			break;
		case Memory::STATE_FORAGING:
			if (isNewState_) {
				memory.depositSearchTimeLeft = 0;
				memory.timeForaging = 0;
				reportedNeighbourhoodSearchTime = 0;
				memory.timeSinceEnteredUnloadingBay = 0;
			}
			break;
		case Memory::STATE_TREMBLE_DANCING:
			if (isNewState_) {
				memory.didTrembleDance = false;
			}
			break;
		case Memory::STATE_SAMPLING_ENCOUNTERS:
			if (isNewState_) {
				memory.encounteredIds.clear();
				memory.timeSamplingEncountersLeft = parameters.samplingEncountersTime;
			}
			break;
		case Memory::STATE_WAKING_OTHERS:
			waggleToIdSent = false;
			waggleRelEESent = false;
			waggleEESent = false;
			waggleAngleToTargetSent = false;
			waggleDistanceToTargetSent = false;
			if (isNewState_) {
				memory.wakingSignalTimeLeft = behaviour.wakingSignalTime;
			}
			break;
		default:
			break;
	}

}


/* ================================ IN NEST =============================== */

/**
 * Broadcast current target EE when asked
 */
void CFootBotForaging::BroadcastTargetInfo() {
	const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
	memory.sendingBeggingResponseToId = -1;

	for(size_t i = 0; i < packets.size(); ++i) {
		if (packets[i].Data[0] == SIGNAL_BEG && packets[i].Range < parameters.beggingSignalDistance && !memory.targetDeposit.isNull && !memory.isWaggleCommunicating && !memory.isReceivingBegginResponse) {

			rangeAndBearingActuator->ClearData();
			rangeAndBearingActuator->SetData(1,id);
			memory.sendingBeggingResponseToId = packets[i].Data[1];

			SetWheelSpeeds(0,0);

			////////////LOGERR << this->GetId() << " begging responding to " << memory.sendingBeggingResponseToId << std::endl;

			Real valueToSend;
			if (!begToIdSent) {
				valueToSend = memory.sendingBeggingResponseToId;
				rangeAndBearingActuator->SetData(0, SIGNAL_BEG_RESPONSE_TO_ID);
				begToIdSent = true;
			} else if (!begEESent) {
				valueToSend = memory.targetDeposit.energyEfficiency;
				rangeAndBearingActuator->SetData(0, SIGNAL_BEG_RESPONSE_EE);
				begEESent = true;
			} else if (!begRelEESent) {
				valueToSend =  memory.targetDeposit.relEnergyEfficiency;
				rangeAndBearingActuator->SetData(0, SIGNAL_BEG_RESPONSE_RELEE);
				begRelEESent = true;
			} else if (!begAngleToTargetSent) {
				//send angle to target, adjusted by angle to the receiver
				valueToSend = (packets[i].HorizontalBearing - (CRadians::PI_OVER_TWO + memory.targetDeposit.relLocation.Angle())).UnsignedNormalize().GetValue();
				rangeAndBearingActuator->SetData(0, SIGNAL_BEG_RESPONSE_ANGLE_TO_TARGET);
				begAngleToTargetSent = true;
			} else if (!begDistanceToTargetSent) {
				//send distance to the target
				valueToSend = memory.targetDeposit.relLocation.Length();
				rangeAndBearingActuator->SetData(0, SIGNAL_BEG_RESPONSE_DISTANCE_TO_TARGET);
				begDistanceToTargetSent = true;
			}

			//-- add the encoded float into the packet data
			UInt8* encodedFloat = (UInt8*)(&valueToSend);
			for (int i=0; i<8; i++) {
				rangeAndBearingActuator->SetData(i+2,encodedFloat[i]);
			}

			//-- check if all information has been sent, in which case memory of what was sent is reset
			if (begRelEESent && begRelEESent && begAngleToTargetSent && begDistanceToTargetSent && begToIdSent) {
				begToIdSent=false;
				begEESent=false;
				begRelEESent=false;
				begAngleToTargetSent=false;
				begDistanceToTargetSent=false;
				//SetState(Memory::STATE_WAGGLE_DANCING, false);
			}

			//-- don't send begging response to any one else
			break;

		}
	}
}

/**
 * Perform a waggle dance recruiting others to a deposit. Could be in this state either purely waggle dancing on the dance
 * floor or if waking others.
 */
void CFootBotForaging::PerformWaggleDance() {
	memory.isWaggleCommunicating = false;

	//---- resolve location: waggle dance on dance floor, waking signals in resting bay
	if (memory.wakingSignalTimeLeft > 0) {
		//-- waking signals, make sure it's in the resting bay
		if (memory.location != Memory::LOCATION_RESTING_BAY) {
			NavigateToLight();
		} else {
			//-- walk randomly
			DoRandomWalk(wheelParams.maxSpeed);
		}
		memory.wakingSignalTimeLeft--;
	} else {
		memory.state = Memory::STATE_WAGGLE_DANCING;
		//-- purely waggle dancing, make sure it is on dance floor
		/*if (memory.location == Memory::LOCATION_RESTING_BAY) {
			NavigateFromLight();
		} else */
		if (memory.location > Memory::LOCATION_DANCE_FLOOR) {
			NavigateToLight();
		} else {
			//-- slow random walk across the dance floor
			DoRandomWalk(wheelParams.maxSpeed);
			memory.timeWaggleDancing++;
			//-- check if should end waggle dance
			if (memory.timeWaggleDancing >= behaviour.waggleDanceTime) {
				reportedWaggleDanceTime = behaviour.waggleDanceTime;
				////////////////LOGERR << this->GetId() << "!! Finished dancing" << std::endl;
				SetState(Memory::STATE_FORAGING);
			}
		}
	}

	//-- resolve locations: both waggle and waking on both dance floor and resting bay
	if (memory.wakingSignalTimeLeft > 0) {
			//-- waking signals, make sure it's in the resting bay
			if (memory.location != Memory::LOCATION_RESTING_BAY) {
				NavigateToLight();
			} else {
				//-- walk randomly
				DoRandomWalk(wheelParams.maxSpeed);
			}
			memory.wakingSignalTimeLeft--;
		} else {
			memory.state = Memory::STATE_WAGGLE_DANCING;
			//-- purely waggle dancing, make sure it is on dance floor
			if (memory.location == Memory::LOCATION_RESTING_BAY) {
				NavigateFromLight();
			} else if (memory.location > Memory::LOCATION_DANCE_FLOOR) {
				NavigateToLight();
			} else {
				//-- slow random walk across the dance floor
				DoRandomWalk(wheelParams.maxSpeed);
				memory.timeWaggleDancing++;
				//-- check if should end waggle dance
				if (memory.timeWaggleDancing >= behaviour.waggleDanceTime) {
					reportedWaggleDanceTime = behaviour.waggleDanceTime;
					////////////////LOGERR << this->GetId() << "!! Finished dancing" << std::endl;
					SetState(Memory::STATE_FORAGING);
				}
			}
		}


	//---- do signal send / receive
	if (memory.state == Memory::STATE_WAGGLE_DANCING || memory.state == Memory::STATE_WAKING_OTHERS) {
		rangeAndBearingActuator->ClearData();

		//========== MODEL D ========
		if (behaviour.useModelD) {
			if (memory.timeTillBeggingLeft <= 0 && behaviour.willBeg) {
				rangeAndBearingActuator->SetData(0, SIGNAL_BEG);
			} else {
				rangeAndBearingActuator->SetData(0, SIGNAL_NONE);
				memory.timeTillBeggingLeft--;
			}
		}
		//===========================

		rangeAndBearingActuator->SetData(4, SIGNAL_WAGGLE_DANCE_AVAILABLE);
		rangeAndBearingActuator->SetData(1,id);

		//-- generate random order in which packets are read
		const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
		int* indeces = new int[packets.size()];
		for(size_t i = 0; i < packets.size(); ++i) {
			int randInt;
			bool unique = true;
			int counter = 0;
			do {
				randInt = randNumGen->Uniform(CRange<int>(0,packets.size()));
				for (int j=0; j<i; j++) {
					if (indeces[j] == randInt) {
						unique = false;
						break;
					} else {
						unique = true;
					}
				}
			} while (!unique);

			indeces[i] = randInt;
		}

		//-- read signals from others
		CRadians angleToObserver = CRadians(0);
		int toId = -1;

		for(int p=0; p < packets.size(); p++) {
			int packetId = indeces[p];
			//-- find out if someone is tremble dancing - if yes, remember to stop next time not in the middle of transmission
			if (packets[packetId].Data[0] == SIGNAL_TREMBLE_DANCE && packets[packetId].Range < parameters.trembleSignalDistance) {
				memory.wasInterruptedDuringWD = true;
				//memory.overExcitementTimeLeft = 400; // set some time so that can't be, as a forager, put to sleep by the same tremble dancer.
				////////////////LOGERR << this->GetId() << "!! Interrupted during dancing by TD" << std::endl;
			} else if (packets[packetId].Data[0] == SIGNAL_STOP && packets[packetId].Data[1] == id && packets[packetId].Range < parameters.signalDistance) {
				memory.wasInterruptedDuringWD = true;
				////////////////LOGERR << this->GetId() << "!! Interrupted during dancing by SS" << std::endl;
			}


			//-- find out if someone is receiving from the robot's id - if yes, stop moving
			if (packets[packetId].Data[0] == SIGNAL_WAGGLE_DANCE_RECEIVING && packets[packetId].Data[1] == id && packets[packetId].Range < parameters.signalDistance*1) {
				////LOGERR << this->GetId() << " Waggle communicating to " << packets[packetId].Data[2] << std::endl;
				memory.isWaggleCommunicating = true;
				SetWheelSpeeds(0,0);
				angleToObserver = packets[packetId].HorizontalBearing;
				toId = packets[packetId].Data[2];
			} else {
				//-- not in the middle of transmission, check if received interruption signal
				if (memory.wasInterruptedDuringWD) {
					memory.isWaggleCommunicating = false;
					SetState(Memory::STATE_FORAGING);
				}
			}
		}

		//-- broadcast waggle dance
		SendWaggleDance(angleToObserver, toId);

		//=========== MODEL D ============
		//-- if it's not going to communicate waggle dance info, go through packages again and see if it's getting begging response
		if (behaviour.useModelD && !memory.isWaggleCommunicating && memory.timeTillBeggingLeft <= 0 && behaviour.willBeg) {
			for(int p=0; p < packets.size(); p++) {
				int packetId = indeces[p];
				if (packets[packetId].Data[0] >= SIGNAL_BEG_RESPONSE_EE && packets[packetId].Data[0] <= SIGNAL_BEG_RESPONSE_DISTANCE_TO_TARGET && packets[packetId].Range < parameters.beggingSignalDistance) {
					//-- if in the middle of receiving from another robot, ignore this
					if ((begRelEEReceived || begEEReceived || begAngleToTargetReceived || begDistanceToTargetReceived || begToIdReceived) && receivingFromId != packets[packetId].Data[1]) {
						//-- someone else's message, ignore
						//////////////LOGERR << this->GetId() << "Ignore BEG RESPN from " << packets[i].Data[1] << std::endl;
					} else {
						//-- set who currently receiving from
						receivingFromId = packets[packetId].Data[1];
						////////////////LOGERR << this->GetId() << "  sending beg response to " << memory.sendingBeggingResponseToId << std::endl;
						//-- if 2 robots are trying to communicate at the same time, the one with higher id will receive the begging response first
						if (memory.sendingBeggingResponseToId == -1 || id > memory.sendingBeggingResponseToId) {
							memory.isReceivingBegginResponse = true;

							CRadians angleToSender = packets[packetId].HorizontalBearing;
							////////////LOGERR << this->GetId() << "Receiving BEGGING RESPONSE from " << receivingFromId << std::endl;
							////////////////LOGERR << "   r " << packets[i].Range << " a " << angleToSender<< std::endl;

							SetWheelSpeeds(0,0);

							//-- decode the float
							UInt8* encodedFloat;
							encodedFloat = new UInt8[8];
							for(size_t d = 2; d < 10; ++d) {
								encodedFloat[d-2] = packets[packetId].Data[d];
							}
							Real receivedFloat = *(Real*)encodedFloat;
							delete encodedFloat;
							//-- remember it
							if (packets[packetId].Data[0] == SIGNAL_BEG_RESPONSE_TO_ID) {
								//-- reset everything if this message is to a different beggar
								if (int(receivedFloat) != this->GetNumId()) {
									////////////LOGERR << this->GetId() << "!!! Ignore BEG RESPO from " << packets[i].Data[1] << " to " << receivedFloat << std::endl;
									begToIdReceived = false;
									begRelEEReceived = false;
									begEEReceived = false;
									begAngleToTargetReceived = false;
									begDistanceToTargetReceived = false;
									memory.isReceivingBegginResponse = false;
									memory.isWaggleCommunicating = false;
								} else {
									begToIdReceived = true;
								}
							} else if (packets[packetId].Data[0] == SIGNAL_BEG_RESPONSE_RELEE) {
								advertisedDeposit.relEnergyEfficiency = receivedFloat;
								begRelEEReceived = true;
							} else if (packets[packetId].Data[0] == SIGNAL_BEG_RESPONSE_EE) {
								advertisedDeposit.energyEfficiency = receivedFloat;
								begEEReceived = true;
							} else if (packets[packetId].Data[0] == SIGNAL_BEG_RESPONSE_ANGLE_TO_TARGET) {
								//-- use the x axis of the vector to store angle for now
								advertisedDeposit.relLocation.SetX(-receivedFloat);
								begAngleToTargetReceived = true;
							} else if (packets[packetId].Data[0] == SIGNAL_BEG_RESPONSE_DISTANCE_TO_TARGET) {
								//-- use y axis of the vector to store distance for now
								advertisedDeposit.relLocation.SetY(receivedFloat);
								begDistanceToTargetReceived = true;
							}


							//-- check if received everything about the advertised deposit
							if (begRelEEReceived && begEEReceived && begAngleToTargetReceived && begDistanceToTargetReceived && begToIdReceived)  {

								CRadians gamma = CRadians(advertisedDeposit.relLocation.GetX()) + packets[packetId].HorizontalBearing - CRadians::PI_OVER_TWO;
								CVector2 translatedPos = -CVector2(packets[packetId].Range/100 * cos(packets[packetId].HorizontalBearing.GetValue()) + advertisedDeposit.relLocation.GetY() * cos(gamma.GetValue()),
																				packets[packetId].Range/100 * sin(packets[packetId].HorizontalBearing.GetValue()) + advertisedDeposit.relLocation.GetY() * sin(gamma.GetValue()));
								////////////LOGERR << this->GetId() << " ----- done receiving from " << receivingFromId << std::endl;
								//-- got everything, stop saying that still receiving
								rangeAndBearingActuator->ClearData();
								rangeAndBearingActuator->SetData(0,SIGNAL_NONE);
								//-- create delay until it can beg again
								memory.timeTillBeggingLeft = 50;
								begToIdReceived = false;
								begRelEEReceived = false;
								begEEReceived = false;
								begAngleToTargetReceived = false;
								begDistanceToTargetReceived = false;
								memory.isReceivingBegginResponse = false;
								memory.isWaggleCommunicating = false;

								//-- if higher EE, set this as the curret deposit:
								if (advertisedDeposit.energyEfficiency - memory.targetDeposit.energyEfficiency > 0.3) { //
									//////////LOGERR << this->GetId() << " !!! changing to new depo " <<  memory.targetDeposit.energyEfficiency << " to " << advertisedDeposit.energyEfficiency << std::endl;
									memory.targetDeposit.energyEfficiency = advertisedDeposit.energyEfficiency;
									memory.targetDeposit.relEnergyEfficiency = advertisedDeposit.relEnergyEfficiency;
									memory.targetDeposit.relLocation = CVector2(translatedPos.GetX(), translatedPos.GetY());
									memory.targetDeposit.foundBySelf = false;
									memory.targetDeposit.infoOriginatorId = receivingFromId;

								}
							}
						}
					}
				}
			}
		}
		//=================== end of for loop through packets
		delete[] indeces;
	}

}

/**
 * Send parts of the waggle dance one by one. In each packet, its bits are:
 * 0 = signal type - acknowledgement this is a waggle dance message
 * 1 = id of the robot
 * 2-9 = encoded float value
 */
void CFootBotForaging::SendWaggleDance(CRadians angleToObserver_, int toId_) {
	if (memory.isWaggleCommunicating) {
		////LOGERR << this->GetId() << " RECRUITING " << toId_ << std::endl;

		//-- send data
		Real valueToSend;
		if (!waggleToIdSent) {
			valueToSend =  toId_;
			rangeAndBearingActuator->SetData(0, SIGNAL_WAGGLE_DANCE_TO_ID);
			waggleToIdSent = true;
			////LOGERR << this->GetId() << " sending to id" << std::endl;
		} else if (!waggleRelEESent) {
			valueToSend =  memory.targetDeposit.relEnergyEfficiency;
			rangeAndBearingActuator->SetData(0, SIGNAL_WAGGLE_DANCE_RELEE);
			waggleRelEESent = true;
		} else if (!waggleEESent) {
			valueToSend = memory.targetDeposit.energyEfficiency;
			rangeAndBearingActuator->SetData(0, SIGNAL_WAGGLE_DANCE_EE);
			waggleEESent = true;
		} else if (!waggleAngleToTargetSent) {
			//-- send angle to target, adjusted by angle to the receiver
			valueToSend = (angleToObserver_ - (CRadians::PI_OVER_TWO + memory.targetDeposit.relLocation.Angle())).UnsignedNormalize().GetValue();
			rangeAndBearingActuator->SetData(0, SIGNAL_WAGGLE_DANCE_ANGLE_TO_TARGET);
			waggleAngleToTargetSent = true;
		} else if (!waggleDistanceToTargetSent) {
			//-- send distance to the target
			valueToSend = memory.targetDeposit.relLocation.Length(); //memory.targetDeposit.relLocation.GetY();
			rangeAndBearingActuator->SetData(0, SIGNAL_WAGGLE_DANCE_DISTANCE_TO_TARGET);
			waggleDistanceToTargetSent = true;
		}

		//-- add the encoded float into the packet data
		UInt8* encodedFloat = (UInt8*)(&valueToSend);
		for (int i=0; i<8; i++) {
			rangeAndBearingActuator->SetData(i+2,encodedFloat[i]);
		}

		//-- check if all information has been sent, in which case memory of what was sent is reset
		if (waggleRelEESent && waggleRelEESent && waggleAngleToTargetSent && waggleDistanceToTargetSent && waggleToIdSent ) {
			////LOGERR << this->GetId() << "--- reset WD vars " << std::endl;
			SetState(memory.state, false);
		}


	}
}

/**
 * Do tremble dancing, moving to the dance floor and discouraging waggle dancers
 */
void CFootBotForaging::PerformTrembleDance() {
	//-- make sure it is on dance floor
	if (memory.location == Memory::LOCATION_RESTING_BAY) {
		NavigateFromLight();
	} else if (memory.location > Memory::LOCATION_UNLOADING_BAY) {
		NavigateToLight();
	} else {
		//-- find out if should continue tremble dancing (also got here if useModelB == false
		if (behaviour.trembleDanceTime <= 0) {
			//-- end of tremble dance, if there is a target, decide if should waggle dance. If there is
			//   no target, just be observer
			////////////////LOGERR << this->GetId() << "GAMMA " << memory.targetDeposit.relEnergyEfficiency << "  pC " << behaviour.reactivationProb << std::endl;
			if (behaviour.willWaggleDance) {
				////////////////LOGERR << this->GetId() << " remain foraging after p(C)=" <<  behaviour.reactivationProb << "(#" << noThrown << "," << memory.targetDeposit.isNull << ")" << std::endl;
				/*
				 if (behaviour.useModelC) {
					//========== MODEL C ========
					//-- go sampling encounters if it is going to waggle dance
					if (behaviour.waggleDanceTime > 0) {
						SetState(Memory::STATE_SAMPLING_ENCOUNTERS);
					} else {
						//-- this will last for t=0
						SetState(Memory::STATE_WAGGLE_DANCING);
					}
					//===========================
				} else {
					SetState(Memory::STATE_WAGGLE_DANCING);
				}
				*/
				SetState(Memory::STATE_WAGGLE_DANCING);
			} else {
				//////////LOGERR << "!!!" << this->GetId() << " become observer after p(C)=" <<  behaviour.reactivationProb << std::endl;
				SetState(Memory::STATE_OBSERVING);
			}
		} else {
			memory.didTrembleDance = true;

			//-- slow random walk across the dance floor
			DoRandomWalk(wheelParams.maxSpeed);
			behaviour.trembleDanceTime--;

			//-- send data, where index:
			//	0 = signal type - tremble dance
			//	1 = id of the robot
			rangeAndBearingActuator->ClearData();
			rangeAndBearingActuator->SetData(0,SIGNAL_TREMBLE_DANCE);
			rangeAndBearingActuator->SetData(1,id);

		}
	}
}

/**
 * Observe signals of others on the dance floor
 */
void CFootBotForaging::Observe() {
	memory.isWaggleCommunicating = false;
	bool sendingAcknowledgementSignalToWaggleDancer = false;

	//-- make sure it is on dance floor
	if (memory.location == Memory::LOCATION_RESTING_BAY) {
		NavigateFromLight();
	} else if (memory.location > Memory::LOCATION_DANCE_FLOOR) {
		NavigateToLight();
	} else {
		//-- slow random walk across the dance floor
		DoRandomWalk(wheelParams.maxSpeed);
	}


	//-- generate random order in which packets are read
	const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
	int* indeces = new int[packets.size()];
	for(size_t i = 0; i < packets.size(); ++i) {
		int randInt;
		bool unique = true;
		int counter = 0;
		do {
			randInt = randNumGen->Uniform(CRange<int>(0,packets.size()));
			for (int j=0; j<i; j++) {
				if (indeces[j] == randInt) {
					unique = false;
					break;
				} else {
					unique = true;
				}
			}
		} while (!unique);

		indeces[i] = randInt;
	}


	for(int p=0; p<packets.size(); p++) {
		int packetId = indeces[p];
		//============ MODEL B ======
		if (behaviour.useModelB) {
			//-- listen to tremble dance signals
			if (!memory.isOverExited) {
				if (packets[packetId].Data[0] == SIGNAL_TREMBLE_DANCE && packets[packetId].Range < parameters.trembleSignalDistance) {
					//-- stop sending anything and go to sleep
					rangeAndBearingActuator->ClearData();
					this->SetState(Memory::STATE_RESTING);
					//////LOGERR << this->GetId() << " going to sleep by TD" << std::endl;
					break;
				}
			}
		}
		//==========================

		//============ MODEL C ======
		if (behaviour.useModelC ) {
			//-- send signal that tells this robot is observing, having a certain id. Needs to use other data indeces than the SIGNAL_WAGGLE_DANCE_RECEIVING and no clearData() because robots are not synchronised
			rangeAndBearingActuator->SetData(4, SIGNAL_OBSERVING);
			rangeAndBearingActuator->SetData(5, id);
			////////LOGERR << " Observing " << this->GetId() << std::endl;
		}
		//==========================

		//-- find out if this is any part of the waggle dance signal and if signal is close enough
		////////LOGERR << this->GetId() << " packet " << packetId << std::endl;
		if (((packets[packetId].Data[0] >= SIGNAL_WAGGLE_DANCE_RELEE && packets[packetId].Data[0] <= SIGNAL_WAGGLE_DANCE_DISTANCE_TO_TARGET) || packets[packetId].Data[4] == SIGNAL_WAGGLE_DANCE_AVAILABLE) && packets[packetId].Range < parameters.signalDistance) {

			//-- if in the middle of receiving from another robot, ignore this
			if ((waggleRelEEReceived || waggleEEReceived || waggleAngleToTargetReceived || waggleDistanceToTargetReceived || waggleToIdReceived) && receivingFromId != packets[packetId].Data[1]) {
				//-- someone else's message, ignore
				////LOGERR << this->GetId() << "!!! Ignore WD from " << packets[packetId].Data[1] << std::endl;
			} else {
				//-- set who currently receiving from
				memory.isWaggleCommunicating = true;
				sendingAcknowledgementSignalToWaggleDancer = true;
				receivingFromId = packets[packetId].Data[1];
				////LOGERR << this->GetId() << " Receiving WD " << packets[packetId].Data[0] << " from " << receivingFromId << std::endl;
				CRadians angleToSender = packets[packetId].HorizontalBearing;


				//-- stop moving
				SetWheelSpeeds(0,0);

				//-- decode the float
				UInt8* encodedFloat;
				encodedFloat = new UInt8[8];
				for(size_t d = 2; d < 10; ++d) {
					encodedFloat[d-2] = packets[packetId].Data[d];
				}
				Real receivedFloat = *(Real*)encodedFloat;
				delete encodedFloat;
				//-- remember it
				if (packets[packetId].Data[0] == SIGNAL_WAGGLE_DANCE_TO_ID) {
					//-- reset everything if this message is to a different observer
					if (int(receivedFloat) != this->GetNumId()) {
						////LOGERR << this->GetId() << "!!! Ignore WD from " << packets[packetId].Data[1] << " to " << receivedFloat << std::endl;
						SetState(Memory::STATE_OBSERVING,false);
					} else {
						waggleToIdReceived = true;
					}
				} else if (packets[packetId].Data[0] == SIGNAL_WAGGLE_DANCE_RELEE) {
					advertisedDeposit.relEnergyEfficiency = receivedFloat;
					waggleRelEEReceived = true;
				} else if (packets[packetId].Data[0] == SIGNAL_WAGGLE_DANCE_EE) {
					advertisedDeposit.energyEfficiency = receivedFloat;
					waggleEEReceived = true;
				} else if (packets[packetId].Data[0] == SIGNAL_WAGGLE_DANCE_ANGLE_TO_TARGET) {
					//-- use the x axis of the vector to store angle for now
					advertisedDeposit.relLocation.SetX(-receivedFloat);
					waggleAngleToTargetReceived = true;
				} else if (packets[packetId].Data[0] == SIGNAL_WAGGLE_DANCE_DISTANCE_TO_TARGET) {
					//-- use y axis of the vector to store distance for now
					advertisedDeposit.relLocation.SetY(receivedFloat);
					waggleDistanceToTargetReceived = true;
				}

				SetWheelSpeeds(0,0);
				//-- check if received everything about the advertised deposit
				if (waggleRelEEReceived && waggleEEReceived && waggleAngleToTargetReceived && waggleDistanceToTargetReceived && waggleToIdReceived) {
					////////////////LOGERR << this->GetId() << " ---- GOT LOC " << advertisedDeposit.relLocation << std::endl;
					////////////////LOGERR << this->GetId() << " GOT EE " << advertisedDeposit.energyEfficiency << "  rel EE " << advertisedDeposit.relEnergyEfficiency << std::endl;

					CRadians gamma = CRadians(advertisedDeposit.relLocation.GetX()) + packets[packetId].HorizontalBearing - CRadians::PI_OVER_TWO;
					CVector2 translatedPos = -CVector2(packets[packetId].Range/100 * cos(packets[packetId].HorizontalBearing.GetValue()) + advertisedDeposit.relLocation.GetY() * cos(gamma.GetValue()),
												packets[packetId].Range/100 * sin(packets[packetId].HorizontalBearing.GetValue()) + advertisedDeposit.relLocation.GetY() * sin(gamma.GetValue()));

					//========== MODEL B
					if (behaviour.useModelB) {
						//-- do stopping signal, tagging who it is for
						if (randNumGen->Uniform(CRange<Real>(0,1)) < behaviour.stopSignalProb) {
							////////////////LOGERR << this->GetId() << "sending stopping signal" << std::endl;
							rangeAndBearingActuator->SetData(0, SIGNAL_STOP);
							rangeAndBearingActuator->SetData(1,receivingFromId);
						}
					}
					//===================
					//-- make the deposit current target and go foraging for it
					memory.targetDeposit.foundBySelf = false;
					memory.targetDeposit.infoOriginatorId = receivingFromId;
					memory.targetDeposit.isNull = false;
					memory.targetDeposit.energyEfficiency = advertisedDeposit.energyEfficiency;
					memory.targetDeposit.relEnergyEfficiency = advertisedDeposit.relEnergyEfficiency;
					memory.targetDeposit.relLocation = CVector2(translatedPos.GetX(), translatedPos.GetY());
					////LOGERR << this->GetId() << "----- recruited by " << receivingFromId << std::endl;
					sendingAcknowledgementSignalToWaggleDancer = false;
					SetState(Memory::STATE_FORAGING);


				}
			}
		}
	}
	delete[] indeces;

	if (sendingAcknowledgementSignalToWaggleDancer) {
		//-- send a signal to the robot specifying someone is receiving
		rangeAndBearingActuator->ClearData();
		rangeAndBearingActuator->SetData(0, SIGNAL_WAGGLE_DANCE_RECEIVING);
		rangeAndBearingActuator->SetData(1,receivingFromId);
		rangeAndBearingActuator->SetData(2,this->GetNumId());
	} else if (receivingFromId >= 0 && memory.state == Memory::STATE_OBSERVING) { //only reset when abandoning listening, we don't want to clear range and bearing data on every loop because observers send observing signals
		this->SetState(Memory::STATE_OBSERVING,false);
		rangeAndBearingActuator->SetData(0, SIGNAL_NONE);
	}

	memory.timeObserving ++;
	if (memory.timeObserving > parameters.minObserverTime) {
		//-- scout with scouting probability
		if (randNumGen->Uniform(CRange<Real>(0,1)) < parameters.scoutProb) {
			SetState(Memory::STATE_SCOUTING);
		}

		//-- go to sleep with sleeping probability
		Real num = randNumGen->Uniform(CRange<Real>(0,1));
		//LOG << num << std::endl;
		if (num < parameters.restingProb) {
			SetState(Memory::STATE_RESTING);
			//LOG << "!!!!!!!!!!!" << std::endl;
		}
	}


}

/**
 * MODEL C: Sampling of how many observers there are
 */
void CFootBotForaging::SampleEncounters() {
	//-- make sure it is on dance floor
	if (memory.location == Memory::LOCATION_RESTING_BAY) {
		NavigateFromLight();
	} else if (memory.location > Memory::LOCATION_DANCE_FLOOR) {
		NavigateToLight();
	} else {
		if (memory.timeSamplingEncountersLeft > 0) {
			//-- random walk across the dance floor
			DoRandomWalk(wheelParams.maxSpeed);

			//-- pick up observer signals and note down the unique ids
			const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
			for(size_t i = 0; i < packets.size(); ++i) {
				if (packets[i].Data[4] == SIGNAL_OBSERVING && packets[5].Range < parameters.encounterSignalDistance) {
					bool alreadyNoted = false;
					for(UInt32 j = 0; j < memory.encounteredIds.size(); j++) {
						if (memory.encounteredIds[j] == packets[i].Data[5]) {
							alreadyNoted = true;
							break;
						}
					}
					if (!alreadyNoted) {
						memory.encounteredIds.push_back(packets[i].Data[5]);
					}
				}
			}
			memory.timeSamplingEncountersLeft--;
		} else {
			uint numOfEncounters = memory.encounteredIds.size();
			//-- decide if needs to do waking signals and how long to waggle dance
			behaviour.CalculateWakingSignalTime(numOfEncounters,parameters.maxEncounters,memory.targetDeposit.relEnergyEfficiency,parameters.maxInteractionTime, parameters.wakeSignalTimeMultiplier);
			//////LOGERR << " !!! " << this->GetId() << "  encounters " << numOfEncounters << " T(WS) " << behaviour.wakingSignalTime << "  T(WD) " << behaviour.waggleDanceTime << std::endl;
			SetState(Memory::STATE_WAKING_OTHERS);
		}
	}
}

/**
 * Rest in the resting bay
 */
void CFootBotForaging::Rest() {
	//-- make sure it's in resting bay
	if (memory.location != Memory::LOCATION_RESTING_BAY ) {
		NavigateToLight();
		memory.timeSpentGettingToMiddle = 0;
	} else {
		// in resting bay already, take some time to get as much to the middle as possible
		if (memory.timeSpentGettingToMiddle > 200 || this->GetLightIntensity() > 0.5) {
			SetWheelSpeeds(0,0);
		} else {
			memory.timeSpentGettingToMiddle++;
			NavigateToLight();
		}
	}
	//-- awake with waking probability
	if (randNumGen->Uniform(CRange<Real>(0,1)) < parameters.wakingProb) {
		SetState(Memory::STATE_OBSERVING);
		//LOGERR << this->GetId() << "WOKE UP SPONTANEOUSLY" << std::endl;
	}

	//========== MODEL C ========
	if (behaviour.useModelC) {

		if (memory.location == Memory::LOCATION_RESTING_BAY || !behaviour.useModelBNonSocial) { // when using modeBnonsocial, robot can only be waken up if it reached the resting bay
			////LOGERR << this->GetId() << "   waiting" << std::endl;
			//-- wake up if received waking signal
			const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
			for(size_t i = 0; i < packets.size(); ++i) {
				if (packets[i].Range < parameters.signalDistance) {
					//LOGERR << this->GetId() << "   receiving" <<  packets[i].Data[0] << "    " << packets[i].Data[4] << " | ";
				}
				if (((packets[i].Data[0] >= SIGNAL_WAGGLE_DANCE_RELEE && packets[i].Data[0] <= SIGNAL_WAGGLE_DANCE_DISTANCE_TO_TARGET) || packets[i].Data[4] == SIGNAL_WAGGLE_DANCE_AVAILABLE) && packets[i].Range < parameters.signalDistance) {
					SetState(Memory::STATE_OBSERVING);
					//LOGERR << this->GetId() << "   WOKEN UP" << std::endl;
					break;
				}
				if (packets[i].Range < parameters.signalDistance) {
					//LOGERR << std::endl;
				}
			}
		}
	}
	//===========================

}


/* ================================ FORAGERS =============================== */

/**
 * Scout for Ns nesources
 */
void CFootBotForaging::Scout() {

	if (memory.location <= Memory::LOCATION_UNLOADING_BAY) {
		//-- do levy walk outside of the base
		NavigateFromLight();
	} else {
		DoLevyWalk();
		//-- test if at a deposit
		if (memory.location == Memory::LOCATION_DEPOSIT && memory.timeTillNextSample == 0) {
			////////////////LOGERR << this->GetId() << "Navigate tow depo" << std::endl;
			if (NavigateTowardsDeposit()) {
				//-- sample the deposit and set delay on when can sample next
				scoutingStatus = 1;
				memory.targetDeposit.infoOriginatorId = id;
				memory.targetDeposit.foundBySelf = true;
				SetState(Memory::STATE_LOADING_SAMPLE);
				memory.timeTillNextSample = 200;
			}
		}
		if (memory.timeTillNextSample > 0) {
			memory.timeTillNextSample --;
		}

		//-- test if sampled enough or for too long
		if (memory.timeScouting >= parameters.maxTimeToScout) {
			scoutingStatus = 0;
			SetState(Memory::STATE_RETURN_TO_BASE_UNSUCCESSFUL);
		}
	}

	memory.timeScouting++;

	//============ MODEL B ======
	if (behaviour.useModelB) {
		const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
		for(size_t i = 0; i < packets.size(); ++i) {
			//-- listen to tremble dance signals
			if (!memory.isOverExited) {
				if (packets[i].Data[0] == SIGNAL_TREMBLE_DANCE && packets[i].Range < parameters.trembleSignalDistance) {
					//-- stop sending anything and go to sleep
					rangeAndBearingActuator->ClearData();
					this->SetState(Memory::STATE_RESTING);
					//////LOGERR << this->GetId() << " going to sleep by TD" << std::endl;
					break;
				}
			}
		}
	//==========================
	//============ MODEL B Non Social ======
	} else if (behaviour.useModelBNonSocial) {
		//-- check if still in the base after TU expired since it entered unloading bay
		if (memory.location == Memory::LOCATION_UNLOADING_BAY) {
			if (memory.timeSinceEnteredUnloadingBay > parameters.maxUnloadTime) {
				SetState(Memory::STATE_RESTING);
				//LOGERR << this->GetId() << " GOING TO SLEEP, can't get out" << std::endl;
			}
			memory.timeSinceEnteredUnloadingBay++;
		}
	}
	//==========================

}

/**
 * Go towards a known target
 */
void CFootBotForaging::Forage() {


	if (memory.location == Memory::LOCATION_DEPOSIT) {
		//-- test if it is on the same location based on deposit gradient radius or if searching, in which case take any deposit found
		if (Abs(memory.targetDeposit.relLocation.Length()) <= 3.0 || memory.depositSearchTimeLeft > 0) {
			////////////////LOGERR << this->GetId() << "A" << std::endl;
			//-- on deposit gradient, test if at the deposit
			if (NavigateTowardsDeposit()) {
				////////////////LOGERR << this->GetId() << " start loading" << std::endl;
				SetState(Memory::STATE_LOADING);
				neighbourhoodSearchStatus = 1;
			}
		} else {
			//-- outside of base, navigate towards target deposit
			////////////////LOGERR << this->GetId() << "B" << std::endl;
			NavigateTowardsVector(-wheelParams.maxSpeed*memory.targetDeposit.relLocation);
		}
	} else {
		//-- test if searching
		if (memory.depositSearchTimeLeft > 0) {
			//-- search in neighbourhood
			DoRandomWalk(wheelParams.maxSpeed);
			if (Abs(memory.targetDeposit.relLocation.Length()) >= parameters.searchRadius) {
				NavigateTowardsVector(-wheelParams.maxSpeed*memory.targetDeposit.relLocation);
			}
			memory.depositSearchTimeLeft--;
			reportedNeighbourhoodSearchTime++;
			////////////////LOGERR << this->GetId() << "searching for" << memory.targetDeposit.relEnergyEfficiency << std::endl;
			//-- when searched finished, abandon deposit and return to base
			if (memory.depositSearchTimeLeft <= 0) {
				neighbourhoodSearchStatus = 0;
				memory.targetDeposit.isNull = true;
				SetState(Memory::STATE_RETURN_TO_BASE_UNSUCCESSFUL);
			}
		} else {
			//-- test if should have reached the deposit already based on believed location
			if (Abs(memory.targetDeposit.relLocation.Length()) <= 0.2) {
				//-- start searching
				////////////////LOGERR << this->GetId() << "C" << std::endl;
				memory.depositSearchTimeLeft = parameters.searchTime;
			} else {
				////////////////LOGERR << this->GetId() << "D" << std::endl;
				//-- outside of base and not searching, navigate towards target deposit
				NavigateTowardsVector(-wheelParams.maxSpeed*memory.targetDeposit.relLocation);
			}
		}

	}

	memory.timeForaging++;
	if (memory.timeForaging > 18000) { //quarter of the simulation time
		//-- got lost, return to base
		memory.targetDeposit.isNull = true;
		SetState(Memory::STATE_RETURN_TO_BASE_UNSUCCESSFUL);
	}

	//============ MODEL B ======
	if (behaviour.useModelB) {
		const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
		for(size_t i = 0; i < packets.size(); ++i) {
			//-- listen to tremble dance signals
			if (!memory.isOverExited) {
				if (packets[i].Data[0] == SIGNAL_TREMBLE_DANCE && packets[i].Range < parameters.trembleSignalDistance) {
					//-- stop sending anything and go to sleep
					rangeAndBearingActuator->ClearData();
					this->SetState(Memory::STATE_RESTING);
					//////LOGERR << this->GetId() << " going to sleep by TD" << std::endl;
					break;
				}
			}
		}
	//==========================
	//============ MODEL B Non Social ======
	} else if (behaviour.useModelBNonSocial) {
		//-- check if still in the base after TU expired since it entered unloading bay
		if (memory.location == Memory::LOCATION_UNLOADING_BAY) {
			if (memory.timeSinceEnteredUnloadingBay > parameters.maxUnloadTime) {
				SetState(Memory::STATE_RESTING);
				//LOGERR << this->GetId() << " GOING TO SLEEP, can't get out" << std::endl;
			}
			memory.timeSinceEnteredUnloadingBay++;
		}
	}
	//==========================


}

/**
 * Return to nest with resource in cart
 */
void CFootBotForaging::ReturnToBase() {
	//-- test if should unload or keep going
	if (memory.location == Memory::LOCATION_UNLOADING_BAY) {
		//- in unloading bay, unload and proceed
		SetState(Memory::STATE_UNLOADING);
	} else {
		//-- still going to the base
		NavigateToLight();

		//-- test if found a deposit - only when foraging was unsuccessful
		if (memory.location == Memory::LOCATION_DEPOSIT && memory.state == Memory::STATE_RETURN_TO_BASE_UNSUCCESSFUL) {
			////////////////LOGERR << this->GetId() << "Found depo by returning home" << std::endl;
			if (NavigateTowardsDeposit()) {
				//-- sample the deposit and set delay on when can sample next
				memory.targetDeposit.infoOriginatorId = id;
				memory.targetDeposit.foundBySelf = true;
				SetState(Memory::STATE_LOADING_SAMPLE);
			}
		}
	}
}


/**
 * Unload a certain amount of resource to unloading bay
 */
float CFootBotForaging::PerformUnload() {
	////////////////LOGERR << GetId() << " unloading " << cart.amount << std::endl;
	Real toUnload = 0;
	bool unloadFinished = true; //set to true in case nothing at all in cart

	if (cart.amount >= 0.00001) {
		unloadFinished = false; //something in cart - unload potentially not finished
		//-- can only unload when there is nothing too close
		if (!IsSomethingNearby() && memory.location == Memory::LOCATION_UNLOADING_BAY) {
			if (memory.timeTillNextUnload <= 0) {
				SetWheelSpeedsFromVector(CVector2(0.0f,0.0f));
				toUnload = parameters.unloadPerPellet;
				if (cart.amount < toUnload) {
					toUnload = cart.amount;
				}
				cart.amount -= toUnload;

				if (toUnload == 0) {
					unloadFinished = true;
				}
				memory.timeTillNextUnload = 20; //2 second
			} else {
				//-- could theoretically unload but not enough time has passed since the last one
				memory.timeTillNextUnload--;
			}
		} else {
			//-- can't unload, move
			DoRandomWalk(wheelParams.maxSpeed/4);
		}
	}

	memory.timeUnloading++;

	if (unloadFinished) {

		//-- navigate towards the middle of dance floor where the next state will be determined
		if (memory.location > Memory::LOCATION_DANCE_FLOOR) {
			NavigateToLight();
		} else {
			if (!didHandleUnloadFinishedEvent) {
				//-- unload finished event - the following only happens once
				didHandleUnloadFinishedEvent = true;

				//-- decide on what will do after the unload
				//======== MODEL B Non Social =========
				if (behaviour.useModelBNonSocial) {
					//-- go to sleep under the same condition that would trigger tremble dance in modelB
					behaviour.CalculateTrembleDaceTime(memory.timeUnloading,parameters.maxUnloadTime,parameters.unloadTimeThreshold,parameters.maxInteractionTime);
					//LOGERR << this->GetId() << " tU=" << memory.timeUnloading <<  "   tTD: " << behaviour.trembleDanceTime << "  tWD:" << behaviour.waggleDanceTime << std::endl;
					if (behaviour.trembleDanceTime > 0) {
						//LOGERR << "      " << this->GetId() << " GOING TO SLEEP " << std::endl;
						SetState(Memory::STATE_RESTING);
						return -1;
					}
				}
				//==========================

				//======== MODEL B =========
				if (behaviour.useModelB) {
					behaviour.CalculateTrembleDaceTime(memory.timeUnloading,parameters.maxUnloadTime,parameters.unloadTimeThreshold,parameters.maxInteractionTime);
					behaviour.UpdateWaggleDanceTimeByUnloadTime(memory.timeUnloading, parameters.maxUnloadTime,parameters.unloadTimeThreshold);
					////LOGERR << this->GetId() << " tU=" << memory.timeUnloading <<  "   tTD: " << behaviour.trembleDanceTime << "  tWD:" << behaviour.waggleDanceTime << std::endl;
					if (behaviour.trembleDanceTime > 0) {
						behaviour.willTrembleDance = true;
					}
				}
				//==========================

				Real noThrown = randNumGen->Uniform(CRange<Real>(0,1));
				if (!memory.targetDeposit.isNull && noThrown < behaviour.reactivationProb) {
					behaviour.willWaggleDance = true;
				}
				////////////////LOGERR << this->GetId() << " waggle? " << behaviour.willWaggleDance << " tremble? " << behaviour.willTrembleDance << std::endl;
				return -1; //signal that unload finished so that event can be written down
			}

			if (behaviour.willTrembleDance || behaviour.willWaggleDance) {
				//-- if it will do any sort of communication, needs to get to the middle of the base first
				if (memory.location > Memory::LOCATION_BASE_DANCE_FLOOR_MIDDLE) {
					NavigateToLight();
				} else {
					SetState(Memory::STATE_TREMBLE_DANCING);
				}
			} else {
				//////////LOGERR << "!!!!" << this->GetId() << " become observer " << std::endl;
				SetState(Memory::STATE_OBSERVING);
			}
		}




	} else {
		//-- make sure it stays within unloading bay
		if (memory.location < Memory::LOCATION_UNLOADING_BAY) {
			NavigateFromLight();
		} else if (memory.location > Memory::LOCATION_UNLOADING_BAY) {
			NavigateToLight();
		}
		didHandleUnloadFinishedEvent = false;
	}

	return toUnload*cart.resourceQuality;
}


/**
 * Load a certain amount of resource to cart. Remember the deposit info
 * and update reactivation and abandonment probabilities.
 * Return true if loading just started
 */
bool CFootBotForaging::PerformLoad(Real amount_, Real resourceQuality_, Real netResourceLeftInDeposit_) {
	////////////////LOGERR << this->GetId() << " loading Q=" << resourceQuality_ << "net: " << netResourceLeftInDeposit_ << std::endl;
	bool loadingDone = false;
	//-- loading during foraging
	if (amount_ == 0) {
		//-- nothing to load anymore
		loadingDone = true;
	} else {
		SetWheelSpeeds(0.0f,0.0f);
		cart.amount += amount_;
		cart.resourceQuality = resourceQuality_;
		////////////////LOGERR << this->GetId() << " loaded " << cart.amount << std::endl;
		if (parameters.cartCapacity - cart.amount < 0.001) {
			cart.amount = parameters.cartCapacity;
		//	//////////////LOGERR << this->GetId() << " GOING HOME WITH " << cart.amount << std::endl;
			loadingDone = true;
		}
	}

	if (loadingDone) {
		//-- set state based on whether scouting
		if (memory.state == Memory::STATE_LOADING) {
			SetState(Memory::STATE_RETURN_TO_BASE_AFTER_FORAGING);
		} else {
			SetState(Memory::STATE_RETURN_TO_BASE_AFTER_SCOUTING);
		}

		//---- update deposit info

		//-- adjust energy efficiency in memory
		////////////////LOGERR << this->GetId() << " OLD ee" << memory.targetDeposit.energyEfficiency << " relEe " << memory.targetDeposit.relEnergyEfficiency << std::endl;
		Real energyEfficiencyNow = EstimateDepositEnergyEfficiency(netResourceLeftInDeposit_, memory.relBaseLocation);
		////////////////LOGERR << memory.relBaseLocation.Length() << std::endl;
		memory.targetDeposit.relEnergyEfficiency = EstimateDepositRelativeEnergyEfficiency(energyEfficiencyNow);

		//-- remember other deposit details
		memory.targetDeposit.relLocation = CVector2(0,0);
		memory.targetDeposit.energyEfficiency = energyEfficiencyNow;
		memory.targetDeposit.isNull = false;
		////////////////LOGERR << this->GetId() << " NEW ee" << memory.targetDeposit.energyEfficiency << " relEe " << memory.targetDeposit.relEnergyEfficiency << std::endl;

		//-- memory about target updated, now update the probabilities and times
		behaviour.CalculateReactivationProb(memory.targetDeposit.relEnergyEfficiency, parameters.minReactivationProb, parameters.maxReactivationProb);
		behaviour.CalculateAbandonmentProb(memory.targetDeposit.relEnergyEfficiency);
		behaviour.CalculateWaggleDaceTime(memory.targetDeposit.relEnergyEfficiency, parameters.maxInteractionTime);

		////////////////LOGERR << this->GetId() << " loading Q=" << resourceQuality_ << " gamma " << memory.targetDeposit.relEnergyEfficiency << " p(C)=" << behaviour.reactivationProb << " tWD" << behaviour.waggleDanceTime << std::endl;

		//-- abandon with certain probability. If just scouting, this probability is 0.
		if (randNumGen->Uniform(CRange<Real>(0,1)) < behaviour.abandonmentProb) {
			memory.targetDeposit.isNull = true;
			////////////////LOGERR << this->GetId() << "!! ABANDONED" << std::endl;
		} else {
			//-- not abandoned, continue with the target in memory
			if (memory.state == Memory::STATE_RETURN_TO_BASE_AFTER_SCOUTING || memory.state == Memory::STATE_RETURN_TO_BASE_UNSUCCESSFUL) {
				memory.targetDeposit.foundBySelf = true;
				memory.targetDeposit.infoOriginatorId = id;
			}
		}

	}

	if (isNewLoad) {
		isNewLoad = false;
		return true;
	} else {
		return false;
	}
}

/**
 * Estimate deposit energy efficiency based on Seeley's equation
 * 		energyEfficiency = (gainedIncome-energySpent)/energySpent;
 * Assume that robot could measure how much resource there is left of what quality.
 * Assume that 0.1 unit of energy is spent per 1 unit of distance from base.
 */
Real CFootBotForaging::EstimateDepositEnergyEfficiency(Real netResource_, CVector2 location_) {
	//-- measure distance to base
	Real energySpent = location_.Length();// * 0.1;
	//Real energyEfficiency = (netResource_-energySpent) / energySpent;
	Real energyEfficiency = (netResource_ / energySpent);
	////////////////LOGERR << "NET RES: " << netResource_ << " E:" << energySpent << " EE: " << energyEfficiency << std::endl;
	return energyEfficiency;
}

/**
 * Estimate relative energy efficiency based on previous experience of EE.
 * If EE before was 0, return 1.
 */
Real CFootBotForaging::EstimateDepositRelativeEnergyEfficiency(Real energyEfficiencyNow_) {
	if (energyEfficiencyNow_ <= 0) {
		return 0.0;
	} else	if (memory.targetDeposit.energyEfficiency == 0) { //|| memory.targetDeposit.isNull
		return 1.0;
	} else {
		if (parameters.doesRelEnergyEfficiency) {
			return Min(1.0,(energyEfficiencyNow_/memory.targetDeposit.energyEfficiency)*memory.targetDeposit.relEnergyEfficiency);
		} else {
			return 1.0;
		}
	}

}

/**
 * Returns true if two locations are maximum parameters.searchRadius away
 */
bool CFootBotForaging::IsTheSameLocation(CVector2& location1_, CVector2& location2_) {
	////////////////LOGERR << "Testing " << location1_ << ":" << location2_ << std::endl;
	return ((location1_ - location2_).Length() <= parameters.searchRadius);
}

/* ================================ MOVEMENT =============================== */
/**
 * Navigate down the gradient towards deposit (where deposit is black).
 * Returns true of at the deposit, otherwise false.
 */
bool CFootBotForaging::NavigateTowardsDeposit() {

	//-- find out if at the deposit already
	const CCI_FootBotProximitySensor::TReadings& proximitySensorData = proximitySensor->GetReadings();
	const CCI_FootBotMotorGroundSensor::TReadings& groudSensorData = groundSensor->GetReadings();
	////////////////LOGERR << "Tow depo " << groudSensorData[1].Value << "   " << groudSensorData[0].Value << std::endl;
	if (groudSensorData[bodyParams.groundSensorFRId].Value < 0.45 || groudSensorData[bodyParams.groundSensorFLId].Value < 0.45) {
		////////////////LOGERR << "--------- THERE" << std::endl;
		for(size_t i = 0; i < proximitySensorData.size(); ++i) {
			////////////////LOGERR << "  " << i << ":" << proximitySensorData[i].Value;
			if (proximitySensorData[i].Value > 0 && proximitySensorData[i].Value <= GetLoadingProximity()) {
				////////////////LOGERR << "!!!!!" << std::endl;
				return true;
			}
			////////////////LOGERR << std::endl;
		}
	}

	//-- go down the gradient towards deposit
	if (groudSensorData[bodyParams.groundSensorFRId].Value > groudSensorData[bodyParams.groundSensorFLId].Value) {
		//-- left side of robot senses lower value, turn towards it
		////////////////LOGERR << "Turn left" << std::endl;
		SetWheelSpeeds(wheelParams.maxSpeed/2.0, wheelParams.maxSpeed);
	} else {
		//-- right side of robot senses higher value, turn towards it
		////////////////LOGERR << "Turn right" << std::endl;
		SetWheelSpeeds(wheelParams.maxSpeed, wheelParams.maxSpeed/2.0);

	}
	return false;
}
/**
 * Navigate towards the light source
 */
void CFootBotForaging::NavigateToLight() {
	//bool bCollision;
	//SetWheelSpeedsFromVector(wheelParams.maxSpeed*0.5 *DiffusionVector(bCollision)  + wheelParams.maxSpeed * 0.75f * CalculateVectorToLight());
	NavigateTowardsVector(wheelParams.maxSpeed * 0.75f * CalculateNormalisedVectorToLight());
	////////////////LOGERR << "COL " << diffustionVec << std::endl;
	//-- test if back is facing the light, which means that the robot has to turn by max angle
	const CCI_FootBotLightSensor::TReadings& tLightReads = lightSensor->GetReadings();
	if (tLightReads[11].Value > 0 && tLightReads[12].Value > 0) {
		if (!memory.lightNavigationAngleDecided) {
			memory.lightNavigationAngleDecided = true;
			/*if (memory.location <= Memory::LOCATION_UNLOADING_BAY) {
				//////////////LOGERR << this->GetId() << "  choosing where to turn IN" << std::endl;
			}*/
			if (randNumGen->Uniform(CRange<Real>(0,1)) < 0.5) {
				memory.randWalkTurnLeft = true;
			} else {
				memory.randWalkTurnLeft = false;
			}
		}
		if (memory.randWalkTurnLeft) {
			SetWheelSpeeds(wheelParams.maxSpeed, 0);
		} else {
			SetWheelSpeeds(0,wheelParams.maxSpeed);
		}
	} else {
		memory.lightNavigationAngleDecided = false;
	}

}

/**
 * Navigate away from light
 */
void CFootBotForaging::NavigateFromLight() {
	//bool bCollision;
	//SetWheelSpeedsFromVector(wheelParams.maxSpeed * 0.5 * DiffusionVector(bCollision) - wheelParams.maxSpeed * 0.75f * CalculateVectorToLight());
	NavigateTowardsVector(-wheelParams.maxSpeed * 0.75f * CalculateNormalisedVectorToLight());
	//-- test if front is facing the light , which means that the robot has to turn by max angle
	const CCI_FootBotLightSensor::TReadings& tLightReads = lightSensor->GetReadings();
	if (tLightReads[0].Value > 0 && tLightReads[23].Value > 0) {
		if (!memory.lightNavigationAngleDecided) {
			memory.lightNavigationAngleDecided = true;
			/*if (memory.location <= Memory::LOCATION_UNLOADING_BAY) {
				//////////////LOGERR << this->GetId() << "  choosing where to turn OUT" << std::endl;
			}*/
			if (randNumGen->Uniform(CRange<Real>(0,1)) < 0.5) {
				memory.randWalkTurnLeft = true;
			} else {
				memory.randWalkTurnLeft = false;
			}
		}
		if (memory.randWalkTurnLeft) {
			SetWheelSpeeds(wheelParams.maxSpeed, 0);
		} else {
			SetWheelSpeeds(0,wheelParams.maxSpeed);
		}
	} else {
		memory.lightNavigationAngleDecided = false;
	}
}

/**
 * Perform random walk of certain speed
 */
void CFootBotForaging::DoRandomWalk(const float& baseSpeed_) {

	CRadians angle = CRadians(0);
	//-- turn only each N ticks
	if (memory.randWalkTurnCounter == 0) {
		//-- turn with small probability
		if (randNumGen->Uniform(CRange<Real>(0,1)) < 0.1) {
			memory.randWalkTurnCounter = 10;
			if (randNumGen->Uniform(CRange<Real>(0,1)) < 0.5) {
				memory.randWalkTurnLeft = true;
			} else {
				memory.randWalkTurnLeft = false;
			}
		}
	}
	if (memory.randWalkTurnCounter > 0) {
		//-- turn
		if (memory.randWalkTurnLeft) {
			angle = CRadians(randNumGen->Uniform(CRange<Real>(0.5,1)));
		} else {
			angle = CRadians(-randNumGen->Uniform(CRange<Real>(0.5,1)));
		}
		memory.randWalkTurnCounter--;
	}
	bool bCollision;
	SetWheelSpeedsFromVector(wheelParams.maxSpeed * DiffusionVector(bCollision) + wheelParams.maxSpeed* CVector2(1.0f,angle));
}

/**
 * Do Levy-random walk
 */
void CFootBotForaging::DoLevyWalk() {
	CRadians angle = CRadians(gsl_ran_levy(gslRandNumGen, 3.0, 1.0));
	if (angle.GetAbsoluteValue() < 0.25) {
		angle = CRadians(0);
	}
	NavigateTowardsVector(wheelParams.maxSpeed * 0.75f * CVector2(1.0f,angle));
}

/**
 * Find out whether in one of the sections of base or outside
 */
void CFootBotForaging::ReadCurrentLocation() {

	memory.location = Memory::LOCATION_OUT;

	//-- Read values from the ground sensor. Test value from sensors at the back (2,3), whre 0 = black, 1=white  */
	const CCI_FootBotMotorGroundSensor::TReadings& groudSensorData = groundSensor->GetReadings();
	////////////////LOGERR << "LOC? L " << groudSensorData[bodyParams.groundSensorFLId].Value << "   " << groudSensorData[bodyParams.groundSensorBLId].Value << "R " << groudSensorData[bodyParams.groundSensorFRId].Value << "  " << groudSensorData[bodyParams.groundSensorBRId].Value << std::endl;
	if ((groudSensorData[bodyParams.groundSensorFRId].Value > 0.37f && groudSensorData[bodyParams.groundSensorFRId].Value < 0.9f) ||
		(groudSensorData[bodyParams.groundSensorFLId].Value > 0.37f && groudSensorData[bodyParams.groundSensorFLId].Value < 0.9f)) {
		memory.location = Memory::LOCATION_DEPOSIT;
	} else if (groudSensorData[bodyParams.groundSensorBRId].Value > 0.27f && groudSensorData[bodyParams.groundSensorBRId].Value < 0.33f &&
			groudSensorData[bodyParams.groundSensorBLId].Value > 0.27f && groudSensorData[bodyParams.groundSensorBLId].Value < 0.33f) {
		memory.location = Memory::LOCATION_DANCE_FLOOR;
	} else if (groudSensorData[bodyParams.groundSensorBRId].Value > 0.17f && groudSensorData[bodyParams.groundSensorBRId].Value < 0.23f &&
			groudSensorData[bodyParams.groundSensorBLId].Value > 0.17f && groudSensorData[bodyParams.groundSensorBLId].Value < 0.23f) {
		memory.location = Memory::LOCATION_BASE_DANCE_FLOOR_MIDDLE;
	} else if (groudSensorData[bodyParams.groundSensorBRId].Value > 0.07f && groudSensorData[bodyParams.groundSensorBRId].Value < 0.13f &&
			groudSensorData[bodyParams.groundSensorBLId].Value > 0.07f && groudSensorData[bodyParams.groundSensorBLId].Value < 0.13f) {
		memory.location = Memory::LOCATION_UNLOADING_BAY;
		memory.relBaseLocation = CVector2(0,0);
	} else if (groudSensorData[bodyParams.groundSensorBRId].Value < 0.05f &&
			groudSensorData[bodyParams.groundSensorBLId].Value < 0.05f) {
		memory.location = Memory::LOCATION_RESTING_BAY;
	}

	////////////////LOGERR << this->GetId() << "  location " << memory.location << " sens L " << groudSensorData[bodyParams.groundSensorFLId].Value << " sens R " << groudSensorData[bodyParams.groundSensorFRId].Value << std::endl;
}

/**
 * Update relative location of target in memory
 */
void CFootBotForaging::UpdateOdometryToLocation(CVector2& location_) {
	//-- update relative location of the deposit
	Real distR, distL, interWheel;

	const CCI_DifferentialSteeringSensor::SReading& steeringData = steeringSensor->GetReading();
	interWheel = steeringData.WheelAxisLength * 0.01;
	distR = steeringData.CoveredDistanceRightWheel * 0.01;
	distL = steeringData.CoveredDistanceLeftWheel * 0.01;

	Real stepDist = ((distR + distL)/2);
	CRadians stepAngle = CRadians((distR-distL)/(interWheel));

	location_ += CVector2(stepDist, stepAngle);
	location_.Rotate(-stepAngle);
}

/* ================================ ACTION PROBABILITIES =================== */

void CFootBotForaging::Behaviour::CalculateReactivationProb(const Real& gamma_, const Real& minReactivationProb_,const Real& maxReactivationProb_) {
	reactivationProb = Min(maxReactivationProb_, Max(gamma_,minReactivationProb_));
	//reactivationProb = Min(1.0, gamma_);
	////////////////LOGERR << gamma_ << " Reactivation " << reactivation << std::endl;
}

void CFootBotForaging::Behaviour::CalculateAbandonmentProb(const Real& gamma_) {
	if (gamma_ == 0) {
		abandonmentProb = 1;
	} else {
		abandonmentProb = 0;
	}
}

void CFootBotForaging::Behaviour::CalculateWaggleDaceTime(const Real& gamma_, const Real& maxInteractionTime_) {
	waggleDanceTime = gamma_*maxInteractionTime_;
	////////////////LOGERR << gamma_ << " T waggle dance " << waggleDanceTime << std::endl;
}


/* ====== MODEL B ===== */
void CFootBotForaging::Behaviour::CalculateTrembleDaceTime(const uint& unloadTime_, const Real& maxUnloadTime_, const Real& unloadTimeThreshold, const Real& maxInteractionTime_) {
	/*if (unloadTime_ > maxUnloadTime_ - unloadTimeThreshold*maxUnloadTime_) {
		trembleDanceTime = Min(unloadTime_/maxUnloadTime_ * maxInteractionTime_, maxInteractionTime_);
	} else {
		trembleDanceTime = 0.0;
	}
	*/
	if (unloadTime_ > maxUnloadTime_) {
		trembleDanceTime = maxInteractionTime_;
	} else {
		trembleDanceTime = 0.0;
	}
}

void CFootBotForaging::Behaviour::UpdateWaggleDanceTimeByUnloadTime(const uint& unloadTime_, const Real& maxUnloadTime_, const Real& unloadTimeThreshold) {
	////////////////LOGERR << " unload T=" << unloadTime_ << std::endl;
	/*if (unloadTime_ <= unloadTimeThreshold*maxUnloadTime_) {
		//-- no change, leave the waggle dance time as calculated based on source profitability
	} else {
		waggleDanceTime = 0.0;
	}*/
	if (unloadTime_ > maxUnloadTime_) {
		waggleDanceTime = 0.0;
	} else {
		//-- no change, leave the waggle dance time as calculated based on source profitability
	}
}

void CFootBotForaging::Behaviour::CalculateStopSignalProb(bool trembleDanced_, const Real& stopSignalProbTrembleDancer_, const Real& stopSignalProbNonTrembleDancer_) {
	if (trembleDanced_) {
		stopSignalProb = stopSignalProbTrembleDancer_;
	} else {
		stopSignalProb = stopSignalProbNonTrembleDancer_;
	}
}

/* ====== MODEL C ===== */
void CFootBotForaging::Behaviour::CalculateWakingSignalTime(const uint& encounters_, const Real& maxEncounters_, const Real& gamma_, const Real& maxInteractionTime_, const Real& multiplier_) {
	if (encounters_ <= maxEncounters_) {
		////////////////LOGERR << " WS after encoutering " << encounters_ << std::endl;
		wakingSignalTime = multiplier_*gamma_*maxInteractionTime_;
	} else {
		wakingSignalTime = 0;
	}
}


/* ================================ VECTOR CALCULATIONS AND NAVIATION ==================== */

/**
 * Calculate vector to light that has normalised length of 1
 */
CVector2 CFootBotForaging::CalculateNormalisedVectorToLight() {
   /* Get readings from light sensor */
   const CCI_FootBotLightSensor::TReadings& tLightReads = lightSensor->GetReadings();
   /* Sum them together */
   CVector2 cAccumulator;
   for(size_t i = 0; i < tLightReads.size(); ++i) {
	   cAccumulator += CVector2(tLightReads[i].Value, tLightReads[i].Angle);
   }
   //-- if the light was perceived, return the vector, otherwise return 0
   ////////////////LOGERR << "ANGLE " << cAccumulator.Angle() << std::endl;
   if(cAccumulator.Length() > 0.0f) {
	   return CVector2(1.0f, cAccumulator.Angle());
   } else {
	   return CVector2();
   }
}

/**
 * Calculate average intensity of light perceived
 */
Real CFootBotForaging::GetLightIntensity() {
   const CCI_FootBotLightSensor::TReadings& tLightReads = lightSensor->GetReadings();
   Real maxIntensity = 0;
   uint counter = 0;
   for(size_t i = 0; i < tLightReads.size(); ++i) {
	   if (fabs(tLightReads[i].Value) > 0.1) {
		   counter ++;
		   maxIntensity += fabs(tLightReads[i].Value);
	   }
   }
   maxIntensity /= counter;
   ////////LOGERR << this->GetId() << "  " << maxIntensity << std::endl;
   return maxIntensity;
}



/*
 * Calculates the diffusion vector. If there is a close obstacle,
 * it points away from it; it there is none, it points forwards.
 * The b_collision parameter is used to return true or false whether
 * a collision avoidance just happened or not. It is necessary for the
 * collision rule.
 */
CVector2 CFootBotForaging::DiffusionVector(bool& b_collision) {
   /* Get readings from proximity sensor */
   const CCI_FootBotProximitySensor::TReadings& tProxReads = proximitySensor->GetReadings();
   /* Sum them together */
   CVector2 cDiffusionVector;
   for(size_t i = 0; i < tProxReads.size(); ++i) {
      cDiffusionVector += CVector2(tProxReads[i].Value, tProxReads[i].Angle);
   }
   ////////////////LOGERR << cDiffusionVector.Length() << cDiffusionVector.Angle() << std::endl;
   /* If the angle of the vector is small enough and the closest obstacle
      is far enough, ignore the vector and go straight, otherwise return
      it */
   if(diffustionParams.GoStraightAngleRange.WithinMinBoundIncludedMaxBoundIncluded(cDiffusionVector.Angle()) &&
      cDiffusionVector.Length() < diffustionParams.Delta ) {
      b_collision = false;
      return CVector2::X;
   }
   else {
      b_collision = true;
      cDiffusionVector.Normalize();
      return -cDiffusionVector;
   }

}

/**
 * Returns true if an object was detected by proximity sensors
 */
bool CFootBotForaging::IsSomethingNearby() {
	const CCI_FootBotProximitySensor::TReadings& tProxReads = proximitySensor->GetReadings();
	for(size_t i = 0; i < tProxReads.size(); ++i) {
		////////////////LOGERR << tProxReads[i].Value << " :" << tProxReads[i].Angle << std::endl;
		if (tProxReads[i].Value > 0) {
			return true;
		}
	}
	return false;
}



/**
 * Navigates towards a vector, avoiding objects at the same time.
 * Returns true if collision detected.
 */
bool CFootBotForaging::NavigateTowardsVector(const CVector2& vector_) {
	bool collision;
	CVector2 diffusionVec = DiffusionVector(collision);

	if (!collision) {
		//-- no collision, head towards the target
		SetWheelSpeedsFromVector(wheelParams.maxSpeed*diffusionVec + vector_);
		collisionAvoidingAngle = CRadians(0);
		/*if (id == 2) {
		//////////////LOGERR << this->GetId() << "Following to vector" << vector_ << std::endl;
		}*/
	} else {
		//-- has direction of this collision's turn been specified?
		if (collisionAvoidingAngle.GetValue() == 0) {
			//-- no, specify a new one
			if (randNumGen->Uniform(CRange<Real>(0,1)) < 2) {
				collisionAvoidingAngle = CRadians(1);
			} else {
				collisionAvoidingAngle = -CRadians(1);
			}
		}
		//-- turn away
		SetWheelSpeedsFromVector(wheelParams.maxSpeed*diffusionVec);
	}
	return collision;
}
/*
 * Gets a direction vector as input and transforms it into wheel
 * actuation.
 */
void CFootBotForaging::SetWheelSpeedsFromVector(const CVector2& c_heading) {
   /* Get the heading angle */
   CRadians cHeadingAngle = c_heading.Angle().SignedNormalize();
   ////////////////LOGERR << cHeadingAngle << std::endl;
   /* Get the length of the heading vector */
   Real fHeadingLength = c_heading.Length();
   /* Clamp the speed so that it's not greater than MaxSpeed */
   Real fBaseAngularWheelSpeed = Min<Real>(fHeadingLength, wheelParams.maxSpeed);

   ////////////////LOGERR << Abs(cHeadingAngle) << std::endl;
   /* Turning state switching conditions */
   if(Abs(cHeadingAngle) <= wheelParams.noTurnAngleThreshold) {
      /* No Turn, heading angle very small */
      wheelParams.turningMechanism = WheelTurningParams::NO_TURN;
     // //////////////LOGERR << " > too low " << Abs(cHeadingAngle) << std::endl;
   }
   else if(Abs(cHeadingAngle) > wheelParams.hardTurnOnAngleThreshold) {
      /* Hard Turn, heading angle very large */
      wheelParams.turningMechanism = WheelTurningParams::HARD_TURN;
     // //////////////LOGERR << " > hard turn " << Abs(cHeadingAngle) << std::endl;
   }
   else  { //if(wheelTurningParams.TurningMechanism == WheelTurningParams::NO_TURN && Abs(cHeadingAngle) > wheelTurningParams.SoftTurnOnAngleThreshold
      /* Soft Turn, heading angle in between the two cases */
      wheelParams.turningMechanism = WheelTurningParams::SOFT_TURN;
     // //////////////LOGERR << " > soft turn " << Abs(cHeadingAngle) << std::endl;
   }

   /* Wheel speeds based on current turning state */
   Real fSpeed1, fSpeed2;
   switch(wheelParams.turningMechanism) {
      case WheelTurningParams::NO_TURN: {
         /* Just go straight */
         fSpeed1 = fBaseAngularWheelSpeed;
         fSpeed2 = fBaseAngularWheelSpeed;
         break;
      }

      case WheelTurningParams::SOFT_TURN: {
         /* Both wheels go straight, but one is faster than the other */
         Real fSpeedFactor = (wheelParams.hardTurnOnAngleThreshold - Abs(cHeadingAngle)) / wheelParams.hardTurnOnAngleThreshold;
         fSpeed1 = fBaseAngularWheelSpeed - fBaseAngularWheelSpeed * (1.0 - fSpeedFactor);
         fSpeed2 = fBaseAngularWheelSpeed + fBaseAngularWheelSpeed * (1.0 - fSpeedFactor);
         break;
      }

      case WheelTurningParams::HARD_TURN: {
         /* Opposite wheel speeds */
         fSpeed1 = -wheelParams.maxSpeed;
         fSpeed2 =  wheelParams.maxSpeed;
         break;
      }
   }

   /* Apply the calculated speeds to the appropriate wheels */
   Real fLeftWheelSpeed, fRightWheelSpeed;
   if(cHeadingAngle > CRadians::ZERO) {
      /* Turn Left */
	  // //////////////LOGERR << "Turn left" << "|" << fSpeed1 << "|" << fSpeed2 << std::endl;
      fLeftWheelSpeed  = fSpeed1;
      fRightWheelSpeed = fSpeed2;
   }
   else {
      /* Turn Right */
	   ////////////////LOGERR << "Turn right" << "|" << fSpeed2 << "|" << fSpeed1 << std::endl;
      fLeftWheelSpeed  = fSpeed2;
      fRightWheelSpeed = fSpeed1;
   }
   /* Finally, set the wheel speeds */

   SetWheelSpeeds(fLeftWheelSpeed, fRightWheelSpeed);
}

/**
 * Sets wheel speed and remembers the decision
 */
void CFootBotForaging::SetWheelSpeeds(Real left_, Real right_) {
	wheels->SetLinearVelocity(left_, right_);
}




/****************************************/
/****************************************/

/*
 * This statement notifies ARGoS of the existence of the controller.
 * It binds the class passed as first argument to the string passed as
 * second argument.
 * The string is then usable in the XML configuration file to refer to
 * this controller.
 * When ARGoS reads that string in the XML file, it knows which controller
 * class to instantiate.
 * See also the XML configuration files for an example of how this is used.
 */
REGISTER_CONTROLLER(CFootBotForaging, "footbot_foraging_controller")
