/*
 * AUTHOR: Carlo Pinciroli <cpinciro@ulb.ac.be>
 *
 * An example foraging controller for the foot-bot.
 *
 * This controller is meant to be used with the XML file:
 *    experiments/foraging.argos
 */

#ifndef FOOTBOT_FORAGING_H
#define FOOTBOT_FORAGING_H

/*
 * Include some necessary headers.
 */
#include <argos3/core/control_interface/ci_controller.h>
#include <argos3/plugins/robots/generic/control_interface/ci_differential_steering_actuator.h>
#include <argos3/plugins/robots/generic/control_interface/ci_leds_actuator.h>
#include <argos3/plugins/robots/generic/control_interface/ci_range_and_bearing_actuator.h>
#include <argos3/plugins/robots/generic/control_interface/ci_range_and_bearing_sensor.h>
#include <argos3/plugins/robots/foot-bot/control_interface/ci_footbot_proximity_sensor.h>
#include <argos3/plugins/robots/foot-bot/control_interface/ci_footbot_light_sensor.h>
#include <argos3/plugins/robots/foot-bot/control_interface/ci_footbot_motor_ground_sensor.h>
#include <argos3/plugins/robots/generic/control_interface/ci_differential_steering_sensor.h>
#include <argos3/core/utility/math/rng.h>

#include <gsl/gsl_randist.h>


/*
 * All the ARGoS stuff in the 'argos' namespace.
 * With this statement, you save typing argos:: every time.
 */
using namespace argos;

/*
 * A controller is simply an implementation of the CCI_Controller class.
 */
class CFootBotForaging : public CCI_Controller {

public:


   struct DiffusionParams {
      /*
       * Maximum tolerance for the proximity reading between
       * the robot and the closest obstacle.
       * The proximity reading is 0 when nothing is detected
       * and grows exponentially to 1 when the obstacle is
       * touching the robot.
       */
      Real Delta;
      /* Angle tolerance range to go straight. */
      CRange<CRadians> GoStraightAngleRange;
      DiffusionParams();
      void Init(TConfigurationNode& t_tree);
   };

   struct WheelTurningParams {
      enum ETurningMechanism {
         NO_TURN = 0, // go straight
         SOFT_TURN,   // both wheels are turning forwards, but at different speeds
         HARD_TURN    // wheels are turning with opposite speeds
      } turningMechanism;

      CRadians hardTurnOnAngleThreshold;
      CRadians softTurnOnAngleThreshold;
      CRadians noTurnAngleThreshold;
      Real maxSpeed;

      void Init(TConfigurationNode& t_tree);
   };

   struct BodyParams {
	   bool isLinuxBased; //true if the simulation is called from linux, e.g. iridis
	   uint groundSensorFLId; //id of front left ground sensor
	   uint groundSensorBLId;
	   uint groundSensorFRId;
	   uint groundSensorBRId;
	   void Init(TConfigurationNode& t_tree);
   };

   struct Deposit {
	  CVector2 relLocation;
	  float energyEfficiency;
	  float relEnergyEfficiency;
	  bool isNull;
	  bool foundBySelf;
	  uint infoOriginatorId; //who told the robot about the deposit - will be set to robot id if found by self

	  Deposit(CVector2 location_, float energyEfficiency_, float relEnergyEfficiency_) { //
		  relLocation = CVector2(location_);
		  energyEfficiency = energyEfficiency_;
		  relEnergyEfficiency = relEnergyEfficiency_;
		  isNull = false;
		  foundBySelf = false;
		  infoOriginatorId = 0;
	  }

	  Deposit() {
		  relLocation = CVector2();
		  energyEfficiency = 0;
		  relEnergyEfficiency = 0;
		  isNull = true;
		  foundBySelf = false;
	  }
   };


   struct Memory {
      /* The three possible states in which the controller can be */
      enum STATE {
         STATE_RESTING = 0,
         STATE_SCOUTING,
         STATE_FORAGING,
         STATE_OBSERVING,
         STATE_RETURN_TO_BASE_AFTER_FORAGING,
         STATE_RETURN_TO_BASE_AFTER_SCOUTING,
         STATE_UNLOADING,
         STATE_LOADING,
         STATE_LOADING_SAMPLE,
         STATE_WAGGLE_DANCING,
         STATE_TREMBLE_DANCING,
         STATE_SAMPLING_ENCOUNTERS,
         STATE_WAKING_OTHERS,
         STATE_RETURN_TO_BASE_UNSUCCESSFUL
      } state;

      enum LOCATION {
    	  LOCATION_RESTING_BAY = 0,
    	  LOCATION_BASE_DANCE_FLOOR_MIDDLE,
    	  LOCATION_DANCE_FLOOR,
    	  LOCATION_UNLOADING_BAY,
    	  LOCATION_DEPOSIT,
    	  LOCATION_OUT
      } location;


      //-- movement
      uint randWalkTurnCounter;
      bool randWalkTurnLeft;
      bool lightNavigationAngleDecided;

      //-- foraging
      Deposit targetDeposit;
      //Deposit visitedDeposit;
      uint numDepositsScouted;
      uint timeTillNextSample;
      uint timeScouting;
      uint timeUnloading;
      uint depositSearchTimeLeft;
      uint timeObserving;

      //-- other
      CVector2 relBaseLocation;
      uint timeWaggleDancing;
      uint timeTillNextUnload;
      uint timeForaging;
      UInt64 timeCounter;
      bool isWaggleCommunicating;

      //-- model B
      bool didTrembleDance;
      bool wasInterruptedDuringWD;
      uint timeSpentGettingToMiddle;

      //-- model B non social
      uint timeSinceEnteredUnloadingBay;

      //-- model C
      uint timeSamplingEncountersLeft;
      std::vector<uint>encounteredIds;
      uint wakingSignalTimeLeft;
      bool isOverExited;
      uint overExcitementTimeLeft;

      //-- model D
      //uint timeSamplingDepositsLeft;
      //Real maxSampledDepositEE;
      bool isReceivingBegginResponse;
      int sendingBeggingResponseToId;
      int timeTillBeggingLeft;


      Memory();
      void Init(TConfigurationNode& t_node);
      void Reset();
   };

   struct Parameters {
	   bool doesRelEnergyEfficiency; // if true, rel EE will be calcualted for deposits, meaning that waggle dance time etc. will depend on the EE found.

	   Real cartCapacity;
	   Real unloadPerPellet;

	   uint maxInteractionTime;
	   uint minObserverTime; // min time a robot stays in the observer state before going to sleep or scouting

	   Real scoutProb;
	   uint maxTimeToScout;

	   Real searchRadius;
	   uint searchTime;
	   Real restingProb;
	   Real wakingProb;

	   uint signalDistance; // distance in cm of how far signals can be picked up


	   Real minReactivationProb;
	   Real maxReactivationProb;

	   //--- model B
	   Real maxUnloadTime;
	   Real stopSignalProbTrembleDancer;
	   Real stopSignalProbNonTrebleDancer;
	   Real unloadTimeThreshold;
	   Real trembleSignalDistance;

	   //--- model C
	   uint maxEncounters;
	   Real encounterSignalDistance;
	   uint samplingEncountersTime;
	   Real wakeSignalTimeMultiplier;

	   //--- model D
	   Real beggingSignalDistance;
	   Real beggingProb;




	   Parameters() {};
	   void Init(TConfigurationNode& t_node);
   };

   struct Cart {
	   Real amount;
	   Real resourceQuality;
	   Cart() {};
	   void Reset();
   };

   struct Behaviour {
	   Real reactivationProb;
	   Real abandonmentProb;
	   Real waggleDanceTime;
	   Real trembleDanceTime;
	   Real stopSignalProb;
	   Real wakingSignalTime;

	   bool useModelB; // tremble dancing
	   bool useModelC; // waking resters in resting bay
	   bool useModelD; // begging signals, recruit for others resource if it is better
	   bool useModelBNonSocial; // go to rest under same conditions as tremble dancing, but don't affect anyone else

	   bool willBeg;
	   bool willWaggleDance;
	   bool willTrembleDance;

	   Behaviour() {
		   reactivationProb = 0;
		   abandonmentProb = 0;
		   waggleDanceTime = 0;
		   trembleDanceTime = 0;
		   stopSignalProb = 0;
		   wakingSignalTime = 0;
		   useModelB = false;
		   useModelBNonSocial = false;
		   useModelC = false;
		   useModelD = false;
		   willBeg = false;
		   willWaggleDance = false;
		   willTrembleDance = false;

	   }
	   void Init(TConfigurationNode& t_node);
	   void CalculateReactivationProb(const Real& gamma_, const Real& minReactivationProb_, const Real& maxReactivationProb_);
	   void CalculateAbandonmentProb(const Real& gamma_);
	   //void IncreaseAbandonmentProb(const uint& maxTimeToAbandon_);
	   void CalculateWaggleDaceTime(const Real& gamma_, const Real& maxInteractionTime_);

	   //--- model B
	   void CalculateTrembleDaceTime(const uint& unloadTime_, const Real& maxUnloadTime_, const Real& unloadTimeThreshold, const Real& maxInteractionTime_);
	   void UpdateWaggleDanceTimeByUnloadTime(const uint& unloadTime_, const Real& maxUnloadTime_, const Real& unloadTimeThreshold);
	   void CalculateStopSignalProb(bool trembleDanced_, const Real& stopSignalProbTrembleDancer_, const Real& stopSignalProbNonTrembleDancer_);

	   //--- model C
	   void CalculateWakingSignalTime(const uint& encounters_, const Real& maxEncounters_, const Real& gamma_, const Real& maxInteractionTime_,const Real& multiplier_);

	   //-- model D
   };

   enum SIGNAL_TYPE {
	   SIGNAL_NONE = 0,
	   SIGNAL_WAGGLE_DANCE_RELEE,
	   SIGNAL_WAGGLE_DANCE_EE,
	   SIGNAL_WAGGLE_DANCE_TO_ID,
	   SIGNAL_WAGGLE_DANCE_ANGLE_TO_TARGET,
	   SIGNAL_WAGGLE_DANCE_DISTANCE_TO_TARGET,
	   SIGNAL_WAGGLE_DANCE_RECEIVING,
	   SIGNAL_WAGGLE_DANCE_AVAILABLE,
	   SIGNAL_TREMBLE_DANCE,
	   SIGNAL_STOP,
	   SIGNAL_WAKE,
	   SIGNAL_OBSERVING,

	   SIGNAL_BEG,
	   SIGNAL_BEG_RESPONSE_EE,
	   SIGNAL_BEG_RESPONSE_RELEE,
	   SIGNAL_BEG_RESPONSE_TO_ID,
	   SIGNAL_BEG_RESPONSE_ANGLE_TO_TARGET,
	   SIGNAL_BEG_RESPONSE_DISTANCE_TO_TARGET,
	   SIGNAL_BEG_RESPONSE_RECEIVING,
	   SIGNAL_BEG_INFO_AVAILABLE,

   };

public:

   static float GetLoadingProximity() { return 0.2; } //proximity to deposit at which loading can begin

   CFootBotForaging();
   virtual ~CFootBotForaging() {}

   virtual void Init(TConfigurationNode& t_node);
   virtual void ControlStep();
   virtual void Reset();
   virtual void Destroy() {}


   float PerformUnload();
   bool PerformLoad(Real amount_, Real resourceQuality_, Real netResourceLeftInDeposit_);

   bool GetIsUnloading() { return memory.state == Memory::STATE_UNLOADING; }
   bool GetIsLoading() { return memory.state == Memory::STATE_LOADING || memory.state == Memory::STATE_LOADING_SAMPLE; }
   Deposit GetTargetDeposit() { return memory.targetDeposit; }
   CVector2 GetRelativeBaseLocation() { return memory.relBaseLocation; }
   uint GetNumId() { return id; }
   void SetId(const uint& id_) { id = id_; }
   Memory::STATE GetState() { return memory.state; }
   Real GetCartAmount() { return cart.amount; }
   Real EstimateDepositEnergyEfficiency(Real netResource_, CVector2 location_);
   Real EstimateDepositRelativeEnergyEfficiency(Real energyEfficiencyNow_);
   uint GetDepositSearchTimeLeft() { return memory.depositSearchTimeLeft; }
   Real GetTargetRelativeEE() { return memory.targetDeposit.relEnergyEfficiency; }
   Real GetTargetEE() { return memory.targetDeposit.energyEfficiency; }
   bool GetTargetFoundBySelf() { return memory.targetDeposit.foundBySelf; }
   uint GetTargetInfoOriginatorId() { return memory.targetDeposit.infoOriginatorId; }

   uint GetScoutingStatus() { uint retVal=scoutingStatus; scoutingStatus=2; return retVal; } // scouting status: 0=unsuccessful, 1=success, 2=nothing to report
   uint GetScoutingTime() { return memory.timeScouting; }
   uint GetNumDepositsScouted() { return memory.numDepositsScouted; }

   int GetReportedWaggleDanceTime() { int retVal=reportedWaggleDanceTime; reportedWaggleDanceTime=-2; return retVal;} //waggle dance time: >= 0: actual time, -1: abandoned instead, -2: nothing to report
   int GetReportedUnloadTime() { return memory.timeUnloading; }
   Real GetReportedUnloadAmount() { return unloadAmount; }
   Real GetReportedUnloadNetAmount() { return unloadNetAmount; }

   uint GetNeighbourhoodSearchStatus() { uint retVal=neighbourhoodSearchStatus; neighbourhoodSearchStatus=2; return retVal; } // search status: 0=unsuccessful, 1=success, 2=nothing to report
   uint GetNeighbourhoodSearchTime() { return reportedNeighbourhoodSearchTime; }

   Real GetCartCapacity() { return parameters.cartCapacity; }

private:

   void Rest();
   void Observe();
   void Forage();
   void Scout();
   void ReturnToBase();
   void PerformWaggleDance();
   void PerformTrembleDance();
   void SampleEncounters();

   bool NavigateTowardsDeposit();
   void NavigateToLight();
   void NavigateToLight(const float& baseSpeed_);
   void NavigateFromLight();
   void DoRandomWalk(const float& baseSpeed_);
   void DoLevyWalk();

   void BroadcastTargetInfo();
   void SendWaggleDance(CRadians angleToObserver_, int toId_);

   void UpdateOdometryToLocation(CVector2& location_);

   bool IsTheSameLocation(CVector2& location1_, CVector2& location2_);

   void ReadCurrentLocation();
   CVector2 CalculateNormalisedVectorToLight();
   Real GetLightIntensity();
   CVector2 DiffusionVector(bool& b_collision);
   bool IsSomethingNearby();
   bool NavigateTowardsVector(const CVector2& vector_);
   void SetWheelSpeedsFromVector(const CVector2& c_heading);
   void SetWheelSpeeds(Real left_, Real right_);
   void SetState(Memory::STATE state_, const bool& isNewState_ = true);

private:
   //-- sensors
   CCI_RangeAndBearingSensor* rangeAndBearingSensor;
   CCI_FootBotProximitySensor* proximitySensor;
   CCI_FootBotLightSensor* lightSensor;
   CCI_FootBotMotorGroundSensor* groundSensor;
   CCI_DifferentialSteeringSensor* steeringSensor;

   //-- actuators
   CCI_DifferentialSteeringActuator* wheels;
   CCI_LEDsActuator* leds;
   CCI_RangeAndBearingActuator*  rangeAndBearingActuator;

   CRandom::CRNG* randNumGen;
   gsl_rng * gslRandNumGen;

   uint id;
   Memory memory;
   Cart cart;
   Parameters parameters;
   Behaviour behaviour;
   WheelTurningParams wheelParams;
   DiffusionParams diffustionParams;
   BodyParams bodyParams;

   CRadians collisionAvoidingAngle;

   //-- sending signals
   bool waggleToIdSent;
   bool waggleRelEESent;
   bool waggleEESent;
   bool waggleAngleToTargetSent;
   bool waggleDistanceToTargetSent;
   bool begToIdSent;
   bool begRelEESent;
   bool begEESent;
   bool begAngleToTargetSent;
   bool begDistanceToTargetSent;

   //-- receiving signals
   bool waggleToIdReceived;
   bool waggleRelEEReceived;
   bool waggleEEReceived;
   bool waggleAngleToTargetReceived;
   bool waggleDistanceToTargetReceived;
   int receivingFromId;
   Deposit advertisedDeposit;
   bool begToIdReceived;
   bool begRelEEReceived;
   bool begEEReceived;
   bool begAngleToTargetReceived;
   bool begDistanceToTargetReceived;

   //-- reporting
   uint scoutingStatus;
   int reportedWaggleDanceTime;
   Real unloadAmount;
   Real unloadNetAmount;
   bool didHandleUnloadFinishedEvent;
   uint neighbourhoodSearchStatus;
   uint reportedNeighbourhoodSearchTime;
   bool isNewLoad;


};

#endif
