#include "foraging_loop_functions.h"
#include <argos3/core/simulator/simulator.h>
#include <argos3/core/utility/configuration/argos_configuration.h>
#include <argos3/plugins/robots/foot-bot/simulator/footbot_entity.h>




//#include <argos3/core/utility/plugins/factory.h>

/*================================== INITIALISATION =================== */

CForagingLoopFunctions::CForagingLoopFunctions() :
   m_cForagingArenaSideX(-10.0, 10.0),
   m_cForagingArenaSideY(-10.0, 10.0),
   timeTillNextPelletDissapears(0),
   floor(NULL),
   randNumGenerator(NULL)
   {

}


void CForagingLoopFunctions::Init(TConfigurationNode& t_node) {
   try {
      TConfigurationNode& tForaging = GetNode(t_node, "foraging");
      //-- get a pointer to the floor entity
      floor = &GetSpace().GetFloorEntity();

      //-- deposit parameters
      GetNodeAttribute(tForaging, "depositGradientRadius", depositGradientRadius);
      GetNodeAttribute(tForaging, "arenaSize", arenaSize);
      GetNodeAttribute(tForaging, "depositMinDistanceFromBase", depositMinDistanceFromBase);
      GetNodeAttribute(tForaging, "pelletDisappearRate", pelletDisappearRate);
      GetNodeAttribute(tForaging, "numOfGroups", numOfGroups);
      GetNodeAttribute(tForaging, "group1Quality", group1Quality);
      GetNodeAttribute(tForaging, "group2Quality", group2Quality);
      GetNodeAttribute(tForaging, "group3Quality", group3Quality);
      GetNodeAttribute(tForaging, "group4Quality", group4Quality);
      GetNodeAttribute(tForaging, "depositMaxDistanceFromBase", depositMaxDistanceFromBase);
      GetNodeAttribute(tForaging, "depositDisappearRate", depositDisappearRate);
      GetNodeAttribute(tForaging, "depositRenewabilityThreshold", depositRenewabilityThreshold);
      GetNodeAttribute(tForaging, "evenlySpacedDeposits", evenlySpacedDeposits);
      GetNodeAttribute(tForaging, "depositAppearRate", depositAppearRate);
      GetNodeAttribute(tForaging, "qualityChangeRate", qualityChangeRate);
      GetNodeAttribute(tForaging, "restingBayRadius", restingBayRadius);
      GetNodeAttribute(tForaging, "danceFloorMiddleRadius", danceFloorMiddleRadius);
      GetNodeAttribute(tForaging, "infiniteDeposits", infiniteDeposits);

      //-- create a new RNG
      randNumGenerator = CRandom::CreateRNG("argos");


      GetNodeAttribute(tForaging, "outputFileNamePrefix", outputFileNamePrefix);

      OnNewRun();


   }
   catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error parsing loop functions!", ex);
   }
}


void CForagingLoopFunctions::Reset() {
	outputFileGlobal.close();
	outputFileRobots.close();
	outputFileLoadEvents.close();
	outputFileScoutingEvents.close();
	outputFileWaggleDanceEvents.close();
	outputFileNeighbSearchEvents.close();
	outputFileUnloadEvents.close();
	outputFileDepositRenewEvents.close();
	OnNewRun();

}

void CForagingLoopFunctions::OnNewRun() {

	LOGERR << "=============== " << outputFileNamePrefix << std::endl;

	resourceCollected = 0;
	timeTillNextDepositDisappears = depositDisappearRate;
	timeTillNextDepositAppears = 0;
	timeTillQualityReassigned = qualityChangeRate;

	//-- reset the output file
	std::ostringstream fileNameGlobal;
	fileNameGlobal << outputFileNamePrefix << "_global.txt";
	outputFileGlobal.open(fileNameGlobal.str(), std::ios_base::trunc | std::ios_base::out);
	outputFileGlobal << "time\tcollected\tpellets" << std::endl;

	std::ostringstream fileNameRobots;
	fileNameRobots << outputFileNamePrefix << "_robots.txt";
	outputFileRobots.open(fileNameRobots.str(), std::ios_base::trunc | std::ios_base::out);
	outputFileRobots << "time\tscouts\tforgrs\tWDers\tobsrvrs\tresters\tunloaders\tTDers\tencSampl\twakers\tdepSamp\tinsprs\tloaders\tladenFor\tscoutLoaders\tladenScouts\tunsucFor" << std::endl;

	std::ostringstream fileNameLoadEvents;
	fileNameLoadEvents << outputFileNamePrefix << "_loadEvents.txt";
	outputFileLoadEvents.open(fileNameLoadEvents.str(), std::ios_base::trunc | std::ios_base::out);
	outputFileLoadEvents << "time\tquality\tEE\tdFromBase\tmemEE\tmemRelEE\tfoundBySearch\tfoundBySelf\trobotId\tdepositX\tdepositY\tgroupId\tinfoFromId" << std::endl;

	std::ostringstream fileNameScoutingEvents;
	fileNameScoutingEvents << outputFileNamePrefix << "_scoutingEvents.txt";
	outputFileScoutingEvents.open(fileNameScoutingEvents.str(), std::ios_base::trunc | std::ios_base::out);
	outputFileScoutingEvents << "time\trobotId\tsuccess\tduration\tnumScouted" << std::endl;

	std::ostringstream fileNameWaggleDanceEvents;
	fileNameWaggleDanceEvents << outputFileNamePrefix << "_wdEvents.txt";
	outputFileWaggleDanceEvents.open(fileNameWaggleDanceEvents.str(), std::ios_base::trunc | std::ios_base::out);
	outputFileWaggleDanceEvents << "time\trobotId\tduration" << std::endl;

	std::ostringstream fileNameUnloadEvents;
	fileNameUnloadEvents << outputFileNamePrefix << "_unloadEvents.txt";
	outputFileUnloadEvents.open(fileNameUnloadEvents.str(), std::ios_base::trunc | std::ios_base::out);
	outputFileUnloadEvents << "time\trobotId\tduration\tamount\tnetAmount" << std::endl;

	std::ostringstream fileNameNeighbSearchEvents;
	fileNameNeighbSearchEvents << outputFileNamePrefix << "_neighbSearchEvents.txt";
	outputFileNeighbSearchEvents.open(fileNameNeighbSearchEvents.str(), std::ios_base::trunc | std::ios_base::out);
	outputFileNeighbSearchEvents << "time\trobotId\tsuccess\tduration" << std::endl;

	std::ostringstream fileNameDepositRenewEvents;
	fileNameDepositRenewEvents << outputFileNamePrefix << "_depositRenewEvents.txt";
	outputFileDepositRenewEvents.open(fileNameDepositRenewEvents.str(), std::ios_base::trunc | std::ios_base::out);
	outputFileDepositRenewEvents << "time\tgroupId\tdepoMass" << std::endl;



	timeTillNextPelletDissapears = 0;

	//-- give robots ids
	CSpace::TMapPerType& robots = GetSpace().GetEntitiesByType("foot-bot");
	for (UInt32 i=0; i<robots.size(); i++) {
		std::ostringstream oss;
		oss << "robot" << i;
		CFootBotEntity& footBotEntity = *any_cast<CFootBotEntity*>(robots[oss.str()]);
		CFootBotForaging& robot = dynamic_cast<CFootBotForaging&>(footBotEntity.GetControllableEntity().GetController());
		robot.SetId(i);
	}
	numOfRobots = robots.size();
	LOG << "[INFO] Number of robots: " << numOfRobots << std::endl;



	//---- create info about items in environment
	depositData.clear();
	CSpace::TMapPerType& deposits = GetSpace().GetEntitiesByType("cylinder");

	CVector2 groupMiddlePosition = CVector2(0,0);
	Real distanceFromMiddle;
	uint groupId = 0;
	uint groupNo = 0;
	numOfDeposits = deposits.size();

	//-- decide if using evenly spaced predefined locations
	numOfPredefinedLocations = 0;
	//-- if evenly spaced deposits, generate exactly the amount of pre-defined locations
	if (evenlySpacedDeposits) {
		if (numOfGroups == 0) {
			numOfPredefinedLocations = numOfDeposits;
		} else {
			numOfPredefinedLocations = numOfGroups;
		}

		//-- if renewable deposits, generate pre-specified locations for them, double the amount as deposits
		if (depositRenewabilityThreshold >= 0 || depositDisappearRate > 0) {
			if (numOfGroups == 0) {
				numOfPredefinedLocations = numOfDeposits*PREDEFINED_DEPOSIT_POSITIONS_MULTIPLIER;
			} else {
				numOfPredefinedLocations = numOfGroups*PREDEFINED_DEPOSIT_POSITIONS_MULTIPLIER;
			}
		}

		/*//-- if spontaneously appearing deposits, the amount of location is num of deposits/groups to start with + how many will appear given the total simulation time and appearance rate
		if (depositAppearRate > 0) {
			if (numOfGroups == 0) {
				numOfPredefinedLocations = numOfDeposits + totalSimulationTime/depositAppearRate;
			} else {
				numOfPredefinedLocations = numOfGroups + totalSimulationTime/depositAppearRate;

			}
			LOGERR << " max time " << totalSimulationTime << " ar " << depositAppearRate << " locations " << numOfPredefinedLocations << std::endl;
		}*/
	}


	//-- generate predefined deposit / group locatins if needed
	//LOGERR << "locaations " << numOfPredefinedLocations << std::endl;
	if (numOfPredefinedLocations > 0) {
		preSpecifiedDepositPositions.clear();
		//LOGERR << "generating " << numOfPredefinedLocations << " predefined locations.. " << std::endl;
		Real startingAngle = randNumGenerator->Uniform(CRange<Real>(0,360));
		Real angleStep = 360.0/(numOfPredefinedLocations);
		for (UInt32 i=0; i<numOfPredefinedLocations; i++) {
			CRadians angle = CRadians(0);
			angle.FromValueInDegrees(startingAngle+angleStep*i);
			//LOGERR << i << " angle " << (startingAngle+angleStep*i) << std::endl;
			preSpecifiedDepositPositions.push_back(CVector2(randNumGenerator->Uniform(CRange<Real>(depositMinDistanceFromBase,depositMaxDistanceFromBase)), angle));
		}
	}

	//-- distribute the deposits
	for (UInt32 i=0; i<numOfDeposits; i++) {
		std::ostringstream oss;
		oss << "deposit" << i;
		CCylinderEntity& deposit = *any_cast<CCylinderEntity*>(deposits[oss.str()]);
		CVector2 depositPos = CVector2(deposit.GetEmbodiedEntity().GetPosition().GetX(), deposit.GetEmbodiedEntity().GetPosition().GetY());

		if (numOfGroups == 0) {
			groupId = i;
			//-- uniform random distribution, make sure deposits are at least minDistance and no more than maxDistance away
			depositPos = generateDepositPosition(deposit,groupId);

		} else {
			//-- deposits grouped by GROUP_SIZE. Make sure they stay within specified range
			//-- adjust quality based on group

			groupId = groupNo;


			//-- checked if group middle has been picked up yet
			distanceFromMiddle = groupMiddlePosition.Length();
			if (distanceFromMiddle == 0) {
				//LOG << " NEW CENTRE " << std::endl;
				//-- picking new centre of group
				depositPos = generateDepositPosition(deposit,groupId);
				groupMiddlePosition = CVector2(depositPos);
				distanceFromMiddle = groupMiddlePosition.Length();

			} else {
				//-- centre of group already picked, place randomly around it
				depositPos = generateDepositPositionAroundGroupMiddle(deposit,groupMiddlePosition);

				if ((i+1)%GROUP_SIZE == 0) {
					//-- new group should start for the next deposit, reset the middle position
					groupMiddlePosition = CVector2(0,0);
					groupNo++;
				}
			}
		}

		//-- adjust quality
		deposit.SetMass(generateGroupMass(groupId));

		Real netResource = (deposit.GetMass()-100000);
		//LOG << "group " << groupId << "  res " << netResource << std::endl;

		//-- measure height
		if (i==0) {
			originalDepositHeight = deposit.GetHeight();
			depositRadius = deposit.GetRadius();
		}

		depositData.push_back(DepositInfo(CVector2(depositPos),groupId));

		//-- if deposits should appear gradually, move all of them initially to a high z position
		if (depositAppearRate > 0) {
			deposit.GetEmbodiedEntity().MoveTo(CVector3(15, 15, 0.5), CQuaternion());
			deposit.GetEmbodiedEntity().MoveTo(CVector3(depositPos.GetX(), depositPos.GetY(), 2), CQuaternion());
			depositData[i].isActive = false; //tell the floor not to render the gradient around
		}
	}

	//LOGERR << "Deposits done" << std::endl;
	//-- create pellets outside of the area
	for (UInt32 i=0; i<300; i++) {
		std::ostringstream pelletId;
		pelletId << "pellet" << i;
	    CBoxEntity* pellet = new CBoxEntity(pelletId.str(),CVector3(50,i,0),CQuaternion(),true,CVector3(0.1,0.1,0.1),0.1);
	    AddEntity(*pellet);
	    pellets.push_back(pellet);
	}
	//LOGERR << "Pellets done" << std::endl;


}


void CForagingLoopFunctions::Destroy() {
	outputFileGlobal.close();
	outputFileRobots.close();
	outputFileLoadEvents.close();
	outputFileScoutingEvents.close();
	outputFileUnloadEvents.close();
	outputFileWaggleDanceEvents.close();
	outputFileNeighbSearchEvents.close();
	outputFileDepositRenewEvents.close();

}

/* ============================================== */

CColor CForagingLoopFunctions::GetFloorColor(const CVector2& position_) {
	//-- mark base sections. X and Y coordinates are swapped in the simulator for some reason
	float x = position_.GetX();
	float y = position_.GetY();
	float distanceFromMiddle = sqrt(x*x + y*y);
	if (distanceFromMiddle < restingBayRadius) {
		//-- resting bay
		return CColor::BLACK;
	} else if (distanceFromMiddle < danceFloorMiddleRadius) {
		//-- dance floor middle
		return CColor::GRAY20;
	} else if (distanceFromMiddle < 2) {
		//-- dance floor
		return CColor::GRAY30;
	} else if (distanceFromMiddle < 3) {
		//-- unloading bay
		return CColor::GRAY10;
	}

	//-- draw gradient around food, goes from 0.5 dark -> black
	for(UInt32 i = 0; i < depositData.size(); ++i) {
		if (depositData[i].isActive) {
			float distanceFromFoodMiddle = (position_ - depositData[i].position).Length() - depositRadius;
			float gradientLength = depositGradientRadius - depositRadius;
			if (distanceFromFoodMiddle <= gradientLength){
				UInt8 grayVal = 102 + 128*distanceFromFoodMiddle/gradientLength; //0.4 - 0.9
				return CColor(grayVal,grayVal,grayVal);
			}
		}
	}
	return CColor::WHITE;
}

/****************************************/
/****************************************/

void CForagingLoopFunctions::PreStep() {

	//-- go through all robots
	numOfScouts = 0;
	numOfForagers = 0;
	numOfWaggleDancers = 0;
	numOfObservers = 0;
	numOfResters = 0;
	numOfUnloaders = 0;
	numOfTrembleDancers = 0;
	numOfEncounterSamplers = 0;
	numOfShakers = 0;
	numOfDepositSamplers = 0;
	numOfLoaders = 0;
	numOfLadenForagers = 0;
	numOfScoutLoaders = 0;
	numOfLadenScouts = 0;
	numOfUnsuccessfulForagers = 0;

	CSpace::TMapPerType& m_cFootbots = GetSpace().GetEntitiesByType("foot-bot");
	for(CSpace::TMapPerType::iterator it = m_cFootbots.begin(); it != m_cFootbots.end(); ++it) {
      CFootBotEntity& footBotEntity = *any_cast<CFootBotEntity*>(it->second);
      CFootBotForaging& footBot = dynamic_cast<CFootBotForaging&>(footBotEntity.GetControllableEntity().GetController());
      CVector3 robotPosition = footBotEntity.GetEmbodiedEntity().GetPosition();

      //-- note down scouting events
      uint scoutingStatus = footBot.GetScoutingStatus();
      if (scoutingStatus < 2) {
    	  //LOG << "!!!! scouting event " << scoutingStatus << footBot.GetId() << std::endl;
    	  outputFileScoutingEvents << GetSpace().GetSimulationClock()/10 << "\t"
    			  << footBot.GetNumId() << "\t"
    			  << scoutingStatus << "\t"
    			  << footBot.GetScoutingTime() << "\t"
    			  << footBot.GetNumDepositsScouted() << std::endl;
      }

      //-- not down neighb search events
      uint neighbSearchStatus = footBot.GetNeighbourhoodSearchStatus();
      if (neighbSearchStatus < 2) {
    	  outputFileNeighbSearchEvents << GetSpace().GetSimulationClock()/10 << "\t"
				  << footBot.GetNumId() << "\t"
				  << neighbSearchStatus << "\t"
				  << footBot.GetNeighbourhoodSearchTime() << std::endl;
      }

      //-- note down waggle dance end events
      int waggleDanceTime = footBot.GetReportedWaggleDanceTime();
      if (waggleDanceTime >= -1) {
    	  outputFileWaggleDanceEvents << GetSpace().GetSimulationClock()/10 << "\t"
    			  << footBot.GetNumId() << "\t"
    			  << waggleDanceTime << std::endl;
      }

      if (footBot.GetIsUnloading()) {
    	  //-- collect resource from the robot
    	  Real robotOutput = footBot.PerformUnload();
    	  //-- test for NAN value
    	  if (robotOutput != robotOutput) {
    		  robotOutput = 0;
    	  }
    	  if (robotOutput >= 0) { //the robot outputs -1 when it just finished unloading so that event can be written down
			  resourceCollected += robotOutput;
			  //LOG << "BaseResource " << resourceCollected << " just got " << robotOutput << std::endl;
			  if (robotOutput > 0) {
				  //-- find a free pellet and put it behind the robot
				  UInt8 id = 0;
				  bool freePelletFound = false;
				  while(!freePelletFound) {
					  if (pellets[id]->GetEmbodiedEntity().GetPosition().GetX() >= 50) {
						  CVector3 pelletPosition;
						  Real distance = 0.16;
						  Real distanceY = 0;
						  bool canPlace = false;
						  while (!canPlace) {
							  pelletPosition = CVector3(-distance,distanceY,0);
							  pelletPosition.Rotate(footBotEntity.GetEmbodiedEntity().GetOrientation());
							  canPlace = pellets[id]->GetEmbodiedEntity().MoveTo(CVector3(robotPosition + pelletPosition),CQuaternion());
							  distance += 0.01;
							  distanceY += 0.03;
							  if (distance > 0.5) {
								  canPlace = true;
								  LOGERR << "!! could not place pellet" << std::endl;
							  }
						  }

						  pelletIdsInUnloadingBay.push_back(id);
						  freePelletFound = true;
					  } else if (id == pellets.size()-1) {
						  LOGERR << "!!! Not enough pellets " << std::endl;
						  freePelletFound = true;
					  }
					  id++;
				  }
			  }
    	  } else {
    		  //-- note down unload event
    		  //LOG << " UNLOAD EVENT " << std::endl;
    		  outputFileUnloadEvents << GetSpace().GetSimulationClock()/10 << "\t"
    		      			  << footBot.GetNumId() << "\t"
    		      			  << footBot.GetReportedUnloadTime() << "\t"
    		      			  << footBot.GetReportedUnloadAmount() << "\t"
    		      			  << footBot.GetReportedUnloadNetAmount() << std::endl;
    	  }
      } else if (footBot.GetIsLoading()) {

    	  //-- check if the deposit it is gathering from has resource left
    	  int depositId = GetDepositIdByPosition(robotPosition);
    	  //LOG << footBot.GetId() << " loading from " << depositId << std::endl;
    	  if (depositId >= 0) {
    		  CCylinderEntity* deposit = GetDepositById(depositId);
    		  if (deposit != NULL) {
    			  //LOG << footBot.GetId() << " deposit ok" << std::endl;
				  Real amountToLoad = deposit->GetHeight();
				  Real loadStep = footBot.GetCartCapacity() / 10.0; //takes 1s to load
				  if (amountToLoad > loadStep) {
					  amountToLoad = loadStep;
				  }

				  if (!infiniteDeposits) {
					  deposit->SetHeight(deposit->GetHeight() - amountToLoad);
				  }
				  //LOGERR << "DEPO H " << deposit->GetHeight() << " Q " << (deposit->GetMass()-100000) << std::endl;

				  //-- pass the amount and net resource left (quantity*quality) to the robot. Quality offset by 100000
				  Real netResource = deposit->GetHeight()*(deposit->GetMass()-100000);
				  if (netResource < loadStep/10.0) {
					  netResource = 0;
				  }
				  bool isNewEvent = footBot.PerformLoad(amountToLoad, deposit->GetMass()-100000, netResource);
				  //LOGERR << footBot.GetId() << " to load " << amountToLoad << "  cart am " << footBot.GetCartAmount() << " new ev? " << isNewEvent << std::endl;
				  if (isNewEvent) {
					  //- this was a first time loading, note down as event. The if statement automatically ignores scouting
					  //LOGERR << "-- " << footBot.GetId() << "  Load event from " << depositId << " q " << deposit->GetMass()-100000  << " info from " << footBot.GetTargetInfoOriginatorId() << std::endl;
					  outputFileLoadEvents << GetSpace().GetSimulationClock()/10 << "\t"
										 << deposit->GetMass()-100000  << "\t"
										 << footBot.EstimateDepositEnergyEfficiency(netResource,CVector2(robotPosition.Length()-3, robotPosition.GetZAngle())) << "\t"
										 << robotPosition.Length()-3 << "\t"
										 << footBot.GetTargetEE() << "\t"
										 << footBot.GetTargetRelativeEE() << "\t";
					  if (footBot.GetDepositSearchTimeLeft() == 0) {
						  //-- found directly
						  outputFileLoadEvents << "0\t";
					  } else {
						  //-- found by search
						  outputFileLoadEvents << "1\t";
					  }
					  if (footBot.GetTargetFoundBySelf() || footBot.GetState() == CFootBotForaging::Memory::STATE_LOADING_SAMPLE) {
						  //LOG << " YES" << std::endl;
						  outputFileLoadEvents << "1\t";
					  } else {
						  //LOG << " NO" << std::endl;
						  outputFileLoadEvents << "0\t";
					  }
					  outputFileLoadEvents << footBot.GetNumId() << "\t"
							  	  	  	  << depositData[depositId].position.GetX() << "\t"
							  	  	  	  << depositData[depositId].position.GetY() << "\t"
							  	  	  	  << depositData[depositId].groupId << "\t"
					  	  	  	  	  	  << footBot.GetTargetInfoOriginatorId() << std::endl;


				  }

				  //-- check if deposit should be renewed or depleted
				  if (depositRenewabilityThreshold >= 0) {
					  //---- deposits when depleted dissappear and appear somewhere else
					  if (numOfGroups == 0) {
						  //-- no groups, only check this deposit
						  if (deposit->GetHeight() - originalDepositHeight*depositRenewabilityThreshold <= 0.00001) {
							   //LOG << "MOVING " << depositId << std::endl;
							   deposit->SetHeight(originalDepositHeight);
							   depositData[depositId].position = CVector2(generateDepositPosition(*deposit,depositId));
							   floor->SetChanged();

							   //-- note down the event
							   outputFileDepositRenewEvents << GetSpace().GetSimulationClock()/10 << "\t"
									   	   	   	   	  << depositId << std::endl;
						  }
					   } else {
						   //-- find out if the whole group is depleted
						   uint groupId = std::floor(depositId/GROUP_SIZE);
						   uint groupMiddleId = groupId*GROUP_SIZE;
						   Real currentGroupVolume = 0;
						   for (uint i = groupMiddleId; i<groupMiddleId+GROUP_SIZE; i++) {
							   CCylinderEntity* deposit = GetDepositById(i);
							   currentGroupVolume += deposit->GetHeight();
						   }
						   if (currentGroupVolume - originalDepositHeight*GROUP_SIZE*depositRenewabilityThreshold <= 0.00001) {
							   //LOG << "MOVING GROUP" << groupId << std::endl;
							   //-- move the middle
							   CCylinderEntity* deposit = GetDepositById(groupMiddleId);
							   deposit->SetHeight(originalDepositHeight);
							   CVector2 groupMiddlePosition = generateDepositPosition(*deposit,groupId);
							   depositData[groupMiddleId].position = CVector2(groupMiddlePosition);
							   //-- move the others in the group
							   for (uint idToMove = groupMiddleId+1; idToMove < groupMiddleId+GROUP_SIZE; idToMove++) {
								   CCylinderEntity* deposit = GetDepositById(idToMove);
								   deposit->SetHeight(originalDepositHeight);
								   depositData[idToMove].position = CVector2(generateDepositPositionAroundGroupMiddle(*deposit,groupMiddlePosition));
							   }
							   floor->SetChanged();

							   //-- note down the event
							   outputFileDepositRenewEvents << GetSpace().GetSimulationClock()/10 << "\t"
													  << groupId << std::endl;
						   } else {
							   //-- not moving the whole group, but still check if this particular deposit was depleted
							   if (deposit->GetHeight() <= 0.00001) {
								  //-- depleted deposit - move it away and update floor color
								  deposit->GetEmbodiedEntity().MoveTo(CVector3(30 + depositId,30,0),CQuaternion());
								  depositData[depositId].position = CVector2(30,30);
								  floor->SetChanged();
							  }
						   }

					   }
				  } else if (deposit->GetHeight() <= 0.00001 && qualityChangeRate > 0) {
					  //---- deposits that will be renewed at the same location but need to disappear for now - just make them inactive, this will assure that robots will still avoid them
					  depositData[depositId].isActive = false;
					  floor->SetChanged();
				  } else {
					  //---- deposits when depleted dissapear forever
					  if (deposit->GetHeight() <= 0.00001) {
						  //-- depleted deposit - move it away and update floor color
						  deposit->GetEmbodiedEntity().MoveTo(CVector3(30 + depositId,30,0),CQuaternion());
						  depositData[depositId].position = CVector2(30,30);
						  floor->SetChanged();
					  }
				  }
    		  } else {
    			  //-- tell robot there is nothing to load
    			  footBot.PerformLoad(0,0,0);
    		  }

    	  } else {
    		  //-- tell robot there is nothing to load
    		  footBot.PerformLoad(0,0,0);
    	  }
      }

      //-- collect data about robot states
      CFootBotForaging::Memory::STATE state = footBot.GetState();
      switch (state) {
      case CFootBotForaging::Memory::STATE_FORAGING:
    	  numOfForagers++;
    	  break;
      case CFootBotForaging::Memory::STATE_LOADING:
    	  numOfLoaders++;
    	  break;
      case CFootBotForaging::Memory::STATE_RETURN_TO_BASE_AFTER_FORAGING:
    	  numOfLadenForagers++;
    	  break;
      case CFootBotForaging::Memory::STATE_OBSERVING:
    	  numOfObservers++;
    	  break;
      case CFootBotForaging::Memory::STATE_RESTING:
          numOfResters++;
          break;
      case CFootBotForaging::Memory::STATE_SCOUTING:
    	  numOfScouts++;
    	  break;
      case CFootBotForaging::Memory::STATE_LOADING_SAMPLE:
    	  numOfScoutLoaders++;
    	  break;
      case CFootBotForaging::Memory::STATE_RETURN_TO_BASE_AFTER_SCOUTING:
          numOfLadenScouts++;
          break;
      case CFootBotForaging::Memory::STATE_WAGGLE_DANCING:
          numOfWaggleDancers++;
          break;
      case CFootBotForaging::Memory::STATE_UNLOADING:
    	  numOfUnloaders++;
    	  break;
      case CFootBotForaging::Memory::STATE_TREMBLE_DANCING:
		  numOfTrembleDancers++;
		  break;
      case CFootBotForaging::Memory::STATE_SAMPLING_ENCOUNTERS:
    	  numOfEncounterSamplers++;
    	  break;
      case CFootBotForaging::Memory::STATE_WAKING_OTHERS:
    	  numOfShakers++;
    	  break;
      case CFootBotForaging::Memory::STATE_RETURN_TO_BASE_UNSUCCESSFUL:
    	  numOfUnsuccessfulForagers++;
    	  break;
	  default:
		  LOGERR << "!!!! Memory state not noted down" << std::endl;
      }
   }

   //-- disappear pellets at a certain rate
   if (timeTillNextPelletDissapears <= 0) {
	   timeTillNextPelletDissapears = pelletDisappearRate;
	   if (pelletIdsInUnloadingBay.size() > 0) {
		   uint id = uint(randNumGenerator->Uniform(CRange<Real>(0,pelletIdsInUnloadingBay.size()-1)));
		   //-- move the pellet away
		  // LOG << "REMOVE id "<<pelletIdsInUnloadingBay[id] << "(" << id << ")" << std::endl;
		   pellets[pelletIdsInUnloadingBay[id]]->GetEmbodiedEntity().MoveTo(CVector3(50,pelletIdsInUnloadingBay[id],0),CQuaternion());
		   //-- remove from the array of ids
		   pelletIdsInUnloadingBay.erase(pelletIdsInUnloadingBay.begin()+id);
	   }
   }
   if (timeTillNextPelletDissapears > 0) {
	   timeTillNextPelletDissapears--;
   }

   //-- disappear deposits at a certain rate
   if (depositDisappearRate >= 1) {
	   if (timeTillNextDepositDisappears == 0) {
		   timeTillNextDepositDisappears = depositDisappearRate;
		   if (numOfGroups == 0) {
			   //-- pick a random deposit and move it somwhere else
			   uint idToMove = uint(randNumGenerator->Uniform(CRange<Real>(0,numOfDeposits)));
			   CCylinderEntity* deposit = GetDepositById(idToMove);
			   deposit->SetHeight(originalDepositHeight);
			   //LOG << "MOVING " << idToMove << std::endl;
			   depositData[idToMove].position = CVector2(generateDepositPosition(*deposit,idToMove));
			   floor->SetChanged();

			   //-- note down the event
			   outputFileDepositRenewEvents << GetSpace().GetSimulationClock()/10 << "\t"
									  << idToMove << std::endl;
		   } else {
			   //- pick a random group and move all deposits in it somwhere else
			   uint groupIdToMove = uint(randNumGenerator->Uniform(CRange<Real>(0,numOfGroups)));
			   //LOG << "MOVING GROUP" << groupIdToMove << std::endl;
			   //-- move the middle
			   uint groupMiddleId = groupIdToMove*GROUP_SIZE;
			   CCylinderEntity* deposit = GetDepositById(groupMiddleId);
			   deposit->SetHeight(originalDepositHeight);
			   CVector2 groupMiddlePosition = generateDepositPosition(*deposit,groupIdToMove);
			   depositData[groupMiddleId].position = CVector2(groupMiddlePosition);
			   //-- move the others in the group
			   for (uint idToMove = groupMiddleId+1; idToMove < groupMiddleId+GROUP_SIZE; idToMove++) {
				   CCylinderEntity* deposit = GetDepositById(idToMove);
				   deposit->SetHeight(originalDepositHeight);
				  // LOG << "MOVING " << idToMove << std::endl;
				   depositData[idToMove].position = CVector2(generateDepositPositionAroundGroupMiddle(*deposit,groupMiddlePosition));
			   }
			   floor->SetChanged();
			   //-- note down the event
			   outputFileDepositRenewEvents << GetSpace().GetSimulationClock()/10 << "\t"
									  << groupIdToMove << std::endl;
		   }
	   } else {
		   timeTillNextDepositDisappears--;
	   }
   }

   //-- appear deposits at a certain rate
   if (depositAppearRate > 0) {
	   if (timeTillNextDepositAppears == 0) {
		   timeTillNextDepositAppears = depositAppearRate;

		   if (numOfGroups == 0) {
			   //- pick the next deposit and make it available by moving it down and redrawing the floor
			   uint idToMove = std::floor(GetSpace().GetSimulationClock()/depositAppearRate);
			   CCylinderEntity* deposit = GetDepositById(idToMove);
			   deposit->GetEmbodiedEntity().MoveTo(CVector3(15, 15, 0.5), CQuaternion());
			   deposit->GetEmbodiedEntity().MoveTo(CVector3(depositData[idToMove].position.GetX(), depositData[idToMove].position.GetY(), 0), CQuaternion());
			   depositData[idToMove].isActive = true;
			   floor->SetChanged();
			   //-- check that nest group id will still be valid (is not higher than total amount of groups), if not this will never happen again.
			   if (idToMove == numOfDeposits-1) {
				   depositAppearRate = 0;
			   }

		   } else {
			   //- pick the next group and make it available by moving it down and redrawing the floor
			   uint groupIdToMove = std::floor(GetSpace().GetSimulationClock()/depositAppearRate);
			   //-- move the middle
			   uint groupMiddleId = groupIdToMove*GROUP_SIZE;
			   CCylinderEntity* deposit = GetDepositById(groupMiddleId);

			   deposit->GetEmbodiedEntity().MoveTo(CVector3(15, 15, 0.5), CQuaternion());
			   deposit->GetEmbodiedEntity().MoveTo(CVector3(depositData[groupMiddleId].position.GetX(), depositData[groupMiddleId].position.GetY(), 0), CQuaternion());
			   depositData[groupMiddleId].isActive = true;

			   //-- move the others in the group
			   for (uint idToMove = groupMiddleId+1; idToMove < groupMiddleId+GROUP_SIZE; idToMove++) {
				   CCylinderEntity* deposit = GetDepositById(idToMove);
				   deposit->GetEmbodiedEntity().MoveTo(CVector3(15, 15, 0.5), CQuaternion());
				   deposit->GetEmbodiedEntity().MoveTo(CVector3(depositData[idToMove].position.GetX(), depositData[idToMove].position.GetY(), 0), CQuaternion());
				   depositData[idToMove].isActive = true;
			   }
			   floor->SetChanged();

			   //-- check that nest group id will still be valid (is not higher than total amount of groups), if not this will never happen again.
			   if (groupIdToMove == numOfGroups-1) {
				   depositAppearRate = 0;
			   }
		   }


	   } else {
		   timeTillNextDepositAppears--;
	   }
   }

   //-- change quality at a certain rate
   if (qualityChangeRate > 0) {

	   if (timeTillQualityReassigned == 0) {
		   timeTillQualityReassigned = qualityChangeRate;
		   //-- create the array that records which group qualities have been assigned
		   bool qualityGroupAssignments[4];
		   uint i;
		   for (i=0; i< 4; i++) { qualityGroupAssignments[i] = false; }
		   //-- go through deposits, renew them and assign them new unique quality
		   if (numOfGroups == 0) {
			   for (uint d = 0; d< numOfDeposits; d++) {
				   CCylinderEntity* deposit = GetDepositById(d);
				   deposit->SetHeight(originalDepositHeight);
				   //-- deposit could have been depleted, make it active again
				   depositData[d].isActive = true;
				   floor->SetChanged();

				   //-- search for a new unassigned group quality that is different than previous quality of this group
				   int qualityGroupId = randNumGenerator->Uniform(CRange<uint>(0,numOfDeposits));
				   bool qualityDifferentForGroup = false;
				   int counter = 0;
				   while (!qualityDifferentForGroup || qualityGroupAssignments[qualityGroupId] == true) {
					   qualityGroupId = randNumGenerator->Uniform(CRange<uint>(0,numOfDeposits));
					   counter++;
					   Real oldMass = deposit->GetMass()-100000;
					   Real newMass = generateGroupMass(qualityGroupId)-100000;

					   if (oldMass != newMass || counter > 100) {
						   qualityDifferentForGroup = true;
					   }
				   }
				   //-- mark this group quality as used and assign it
				   qualityGroupAssignments[qualityGroupId] = true;
				   deposit->SetMass(generateGroupMass(qualityGroupId));
				   //LOG << " Depo " << d << " group quality " << qualityGroupId << "  q " << deposit->GetMass()-100000 << "  vol " << deposit->GetHeight() << std::endl;

				   //-- note down the event
				   outputFileDepositRenewEvents << GetSpace().GetSimulationClock()/10 << "\t"
												<< d << "\t" << deposit->GetMass()-100000 << std::endl;
			   }
		   } else {


		   }
	   } else {
		   timeTillQualityReassigned--;
	   }
   }

   //-- output stuff to files:
   if (GetSpace().GetSimulationClock() % 10 == 0) {
	   outputFileGlobal << GetSpace().GetSimulationClock()/10 << "\t"
				 << resourceCollected << "\t"
				 << pelletIdsInUnloadingBay.size() << "\t"
				 << std::endl;
	   outputFileRobots << GetSpace().GetSimulationClock()/10 << "\t"
	   				 << numOfScouts/numOfRobots  << "\t"
	   				 << numOfForagers/numOfRobots << "\t"
	   				 << numOfWaggleDancers/numOfRobots << "\t"
	   				 << numOfObservers/numOfRobots << "\t"
	   				 << numOfResters/numOfRobots << "\t"
	   				 << numOfUnloaders/numOfRobots << "\t"
	   				 << numOfTrembleDancers/numOfRobots << "\t"
	   				 << numOfEncounterSamplers/numOfRobots << "\t"
	   				 << numOfShakers/numOfRobots << "\t"
	   				 << numOfDepositSamplers/numOfRobots << "\t"
	   				 << 0 << "\t"
	   				 << numOfLoaders/numOfRobots << "\t"
	   				 << numOfLadenForagers/numOfRobots << "\t"
	   				 << numOfScoutLoaders/numOfRobots << "\t"
	   				 << numOfLadenScouts/numOfRobots << "\t"
	   				 << numOfUnsuccessfulForagers/numOfRobots
	   				 << std::endl;

   }
}

Real CForagingLoopFunctions::generateGroupMass(uint groupId_) {
	if (groupId_ == 0) {
		return (100000 + group1Quality);
	} else if (groupId_ == 1) {
		return(100000 + group2Quality);
	} else if (groupId_ == 2) {
		return(100000 + group3Quality);
	} else if (groupId_ == 3) {
		return(100000 + group4Quality);
	}
	return (100001 - (group1Quality-1));

}
/* ================================================== SCENARIO GENERATION */

/**
 * Generate a position between min and max distance from base for a deposit and move it there
 */
CVector2 CForagingLoopFunctions::generateDepositPosition(CCylinderEntity& deposit_, uint groupId_) {
	CVector2 depositPos;
	bool can = false;
	while (!can) {
		if (numOfPredefinedLocations > 0) {
			/*if (numOfPredefinedLocations == numOfGroups) {
				//-- just assign position belonging to that group, not much choice here
				depositPos = CVector2(preSpecifiedDepositPositions[groupId_]);
				LOGERR << "!! Using predefined loc for " << groupId_ << std::endl;
			} else {*/
				//-- if there are pre-defined locations, pick randomly from them
				bool prespecifiedLocationFree = false;
				while (!prespecifiedLocationFree) {
					//-- find a free location, this check is needed for groups where group middle could already be depleted so a simple check against it would not work
					depositPos = CVector2(preSpecifiedDepositPositions[uint(randNumGenerator->Uniform(CRange<Real>(0,numOfPredefinedLocations)))]);
					if (GetDepositIdByPosition(CVector3(depositPos.GetX(),depositPos.GetY(), 0), 2.0) >= 0) {
						prespecifiedLocationFree = false;
					} else {
						prespecifiedLocationFree = true;
					}
				}
			//}
		} else {
			depositPos = CVector2(randNumGenerator->Uniform(CRange<Real>(depositMinDistanceFromBase,depositMaxDistanceFromBase)),randNumGenerator->Uniform(CRange<CRadians>(CRadians(0),CRadians(6.5))));
		}
		can = deposit_.GetEmbodiedEntity().MoveTo(CVector3(depositPos.GetX(), depositPos.GetY(), 0), CQuaternion());
		if (!can) {
			LOGERR << "!! Can't use location "<<  depositPos.GetX() << " " << depositPos.GetY() << " for group " << groupId_ << std::endl;

		}
	}
	return depositPos;
}

/**
 * Generate a position that is close to a group middle and move the deposit there
 */
CVector2 CForagingLoopFunctions::generateDepositPositionAroundGroupMiddle(CCylinderEntity& deposit_, CVector2 groupMiddlePosition_) {
	CVector2 depositPos;
	bool can = false;
	while (!can) {
		depositPos = CVector2(groupMiddlePosition_ + CVector2(randNumGenerator->Uniform(CRange<Real>(0.5,1.5)),randNumGenerator->Uniform(CRange<CRadians>(CRadians(0),CRadians(6.5)))));
		can = deposit_.GetEmbodiedEntity().MoveTo(CVector3(depositPos.GetX(), depositPos.GetY(), 0), CQuaternion());
	}
	return depositPos;
}


/* ================================================== OTHER */

/**
 * Returns a deposit id that is very near a position
 */
int CForagingLoopFunctions::GetDepositIdByPosition(CVector3 position_) {
	return GetDepositIdByPosition(position_, CFootBotForaging::GetLoadingProximity() + 0.2);
}

/**
 * Returns the first deposit id that is a certain distance from a position
 */
int CForagingLoopFunctions::GetDepositIdByPosition(CVector3 position_, Real maxDistance_) {
	CVector2 position = CVector2(position_.GetX(),position_.GetY());
	for(UInt32 i = 0; i < depositData.size(); ++i) {
		float distanceFromFoodMiddle = (position - depositData[i].position).Length() - depositRadius;
		if (distanceFromFoodMiddle <= maxDistance_) {
			return i;
		}
	}
	return -1;
}



/**
 * Returns a deposit that is very near a position
 */
CCylinderEntity* CForagingLoopFunctions::GetDepositByPosition(CVector3 position_) {
	CVector2 position = CVector2(position_.GetX(),position_.GetY());
	for(UInt32 i = 0; i < depositData.size(); ++i) {
		float distanceFromFoodMiddle = (position - depositData[i].position).Length() - depositRadius;
		if (distanceFromFoodMiddle <= CFootBotForaging::GetLoadingProximity() + 0.2) {
			//-- found it, retreieve it from the cylinder entities
			std::ostringstream oss;
			oss << "deposit" << i;
			CSpace::TMapPerType& deposits = GetSpace().GetEntitiesByType("cylinder");
			CCylinderEntity* deposit = any_cast<CCylinderEntity*>(deposits[oss.str()]);
			return deposit;
		}
	}
	return NULL;
}

/**
 * Returns a deposit that has a certain id
 */
CCylinderEntity* CForagingLoopFunctions::GetDepositById(const int& id_) {
	CSpace::TMapPerType& deposits = GetSpace().GetEntitiesByType("cylinder");
	if (id_ >= 0 && id_ < deposits.size()) {
		std::ostringstream oss;
		oss << "deposit" << id_;
		CCylinderEntity* deposit = any_cast<CCylinderEntity*>(deposits[oss.str()]);
		return deposit;
	}
	return NULL;
}

/**
 * Returns a robot that has a certain id
 */
CFootBotForaging& CForagingLoopFunctions::GetRobotById(const int& id_) {
	CSpace::TMapPerType& robots = GetSpace().GetEntitiesByType("foot-bot");
	std::ostringstream oss;
	oss << "robot" << id_;
	CFootBotEntity& footBotEntity = *any_cast<CFootBotEntity*>(robots[oss.str()]);
	CFootBotForaging& robot = dynamic_cast<CFootBotForaging&>(footBotEntity.GetControllableEntity().GetController());
	return robot;

}

/****************************************/
/****************************************/

REGISTER_LOOP_FUNCTIONS(CForagingLoopFunctions, "foraging_loop_functions")
