% BCJR algorithm for a terminated unity-rate recursive convolutional code
% having 3 memory elements, a generator polynomial of [1,1,0,1] and a feedback
% polynomial of [1,0,1,1]. This is as used in the UMTS turbo code, as specified
% in ETSI TS 125 212 (search for it on Google if you like). For more
% information, see Section 2.2.2 of Liang Li's nine-month report
% (http://users.ecs.soton.ac.uk/rm/wp-content/liang_li_nine_month_report.pdf)
% or the BCJR paper (http://ieeexplore.ieee.org/xpls/abs_all.jsp?arnumber=1055186).
% Copyright (C) 2010  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your
% option) any later version.

% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.



% c_tilde_a is a matrix of a priori encoded LLRs
% b_tilde_a is a vector of a priori uncoded LLRs
% c_tilde_e is a matrix of extrinsic encoded LLRs
% b_tilde_p is a vector of a posteriori uncoded LLRs
function [c_tilde_e, b_tilde_p] = CC_decoder_4state_bcjr(c_tilde_a, b_tilde_a)

bit_count = length(b_tilde_a);

if(size(c_tilde_a,2) ~= bit_count)
    error('LLR sequences must have the same length');
end


% All calculations are performed in the logarithmic domain in order to
% avoid numerical issues. These occur in the normal domain, because some of
% the confidences can get smaller than the smallest number the computer can
% store. See Section 2.2.2 of Liang Li's nine month report for more information
% on this.
%
% A multiplication of two confidences is achieved using the addition of the
% corresponding log-confidences. If A = log(a) and B = log(b), then
% log(a*b) = A+B (Equation 2.12 in Liang Li's nine month report).
%
% An addition of two confidences is achieved using the Jacobian logarithm
% of the corresponding log-confidences. The Jacobian logarithm is defined
% in the jac.m file. If A = log(a) and B = log(b), then
% log(a+b) = max(A,B) + log(1+exp(-abs(A-B))) (Equation 2.12 in Liang Li's
% nine month report).

% Matrix to describe the trellis
% Each row describes one transition in the trellis
% Each state is allocated an index 1,2,3,... Note that this list starts
% from 1 rather than 0.
%               FromState,  ToState,    UncodedBit, EncodedBit1,EncodedBit2
transitions =  [1,          1,          0,          0,          0;
                1,          3,          1,          1,          1;
                2,          1,          0,          0,          1;
                2,          3,          1,          1,          0;
                3,          4,          0,          1,          0;
                3,          2,          1,          0,          1;
                4,          4,          0,          1,          1;
                4,          2,          1,          0,          0];

% Find the largest state index in the transitions matrix
% In this example, we have eight states since the code has three memory elements
state_count = max(max(transitions(:,1)),max(transitions(:,2)));
transition_count = size(transitions,1);
codeword_length = size(c_tilde_a,1);

gammas = zeros(transition_count, bit_count, codeword_length+1);
for transition_index = 1:transition_count
    if transitions(transition_index, 3)
        gammas(transition_index, b_tilde_a ~= inf,:) = repmat(b_tilde_a(b_tilde_a ~= inf),[1,1,codeword_length+1]);
    else
        gammas(transition_index, b_tilde_a == inf,:) = -inf;
    end
end
for codebit_index = 1:codeword_length
    for transition_index = 1:transition_count
        if transitions(transition_index, 3+codebit_index)
            gammas(transition_index, c_tilde_a(codebit_index,:) ~= inf,[1:codebit_index, codebit_index+2:codeword_length+1]) = add(gammas(transition_index, c_tilde_a(codebit_index,:) ~= inf,[1:codebit_index, codebit_index+2:codeword_length+1]),repmat(c_tilde_a(codebit_index, c_tilde_a(codebit_index,:) ~= inf),[1,1,codeword_length])); % 8 additions
        else
            gammas(transition_index, c_tilde_a(codebit_index,:) == inf,[1:codebit_index, codebit_index+2:codeword_length+1]) = -inf;
        end
    end
end

%gammas

% Forward recursion to calculate state log-confidences. This is similar to
% Equation 1.13 in Rob's thesis or Equations 5 and 6 in the BCJR paper.
alphas=-inf*ones(state_count,bit_count);
alphas(1,1)=0; % We know that this is the first state
for bit_index = 2:bit_count
    set = zeros(1,state_count);
    for transition_index = 1:transition_count
        if(set(transitions(transition_index,2)))
            alphas(transitions(transition_index,2),bit_index) = maxstar(alphas(transitions(transition_index,2),bit_index), add(alphas(transitions(transition_index,1),bit_index-1), gammas(transition_index, bit_index-1,1)));  % 4 maxstars + 4 addition
        else
            alphas(transitions(transition_index,2),bit_index) = add(alphas(transitions(transition_index,1),bit_index-1), gammas(transition_index, bit_index-1,1));  % 4 addition
            set(transitions(transition_index,2)) = 1;
        end
    end
end

%alphas

% Backwards recursion to calculate state log-confidences. This is similar
% to Equation 1.14 in Rob's thesis or Equations 7 and 8 in the BCJR paper.
betas=-inf*ones(state_count,bit_count);
betas(:,end)=0; % We know that this is the final state
for bit_index = bit_count-1:-1:1
    set = zeros(1,state_count);
    for transition_index = 1:transition_count
        if(set(transitions(transition_index,1)))
            betas(transitions(transition_index,1),bit_index) = maxstar(betas(transitions(transition_index,1),bit_index), add(betas(transitions(transition_index,2),bit_index+1), gammas(transition_index, bit_index+1,1))); % 7 maxstars
        else
            betas(transitions(transition_index,1),bit_index) = add(betas(transitions(transition_index,2),bit_index+1), gammas(transition_index, bit_index+1,1));  % 1 addition
            set(transitions(transition_index,1)) = 1;
        end
    end
end

%betas

% Calculate a posteriori transition log-confidences. This is similar to
% Equation 1.15 in Rob's thesis or Equation 4 in the BCJR paper.
deltas=zeros(transition_count,bit_count,codeword_length+1);
for transition_index = 1:transition_count
    deltas(transition_index, :,:) = repmat(add(alphas(transitions(transition_index,1),:), betas(transitions(transition_index,2),:)),[1,1,codeword_length+1]); % 8 additions
end
deltas = add(deltas,gammas); % 8 addition

%deltas

% Calculate the extrinsic LLRs. This is similar to Equation 1.16 in
% Rob's thesis.

bc_tilde_e = zeros(codeword_length+1,bit_count);
for codebit_index = 1:codeword_length+1
    prob1=-inf*ones(1,bit_count);
    prob0=-inf*ones(1,bit_count);
    set1 = 0;
    set0 = 0;
    for transition_index = 1:transition_count
        if transitions(transition_index,2+codebit_index)
            if set1
                prob1 = maxstar(prob1, deltas(transition_index,:,codebit_index));    % 11 maxstar
            else
                prob1 = deltas(transition_index,:,codebit_index);
                set1 = 1;
            end
        else
            if set0
                prob0 = maxstar(prob0, deltas(transition_index,:,codebit_index));    % 11 maxstar
            else
                prob0 = deltas(transition_index,:,codebit_index);
                set0 = 1;
            end
        end

    end
    bc_tilde_e(codebit_index,:) = add(prob1,-prob0);         % 3 additions
end

b_tilde_p = bc_tilde_e(1,:);
c_tilde_e = bc_tilde_e(2:end,:);
