% BCJR algorithm for a terminated unity-rate recursive convolutional code
% having 3 memory elements, a generator polynomial of [1,1,0,1] and a feedback
% polynomial of [1,0,1,1]. This is as used in the UMTS turbo code, as specified
% in ETSI TS 125 212 (search for it on Google if you like). For more
% information, see Section 2.2.2 of Liang Li's nine-month report
% (http://users.ecs.soton.ac.uk/rm/wp-content/liang_li_nine_month_report.pdf)
% or the BCJR paper (http://ieeexplore.ieee.org/xpls/abs_all.jsp?arnumber=1055186).
% Copyright (C) 2010  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your
% option) any later version.

% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.



% c_tilde_a is a matrix of a priori encoded LLRs
% b_tilde_a is a vector of a priori uncoded LLRs
% c_tilde_e is a matrix of extrinsic encoded LLRs
% b_tilde_p is a vector of a posteriori uncoded LLRs
function b_hat = CC_decoder_4state_viterbi(c_tilde_a, b_tilde_a)

bit_count = length(b_tilde_a);

if(size(c_tilde_a,2) ~= bit_count)
    error('LLR sequences must have the same length');
end


% All calculations are performed in the logarithmic domain in order to
% avoid numerical issues. These occur in the normal domain, because some of
% the confidences can get smaller than the smallest number the computer can
% store. See Section 2.2.2 of Liang Li's nine month report for more information
% on this.
%
% A multiplication of two confidences is achieved using the addition of the
% corresponding log-confidences. If A = log(a) and B = log(b), then
% log(a*b) = A+B (Equation 2.12 in Liang Li's nine month report).
%
% An addition of two confidences is achieved using the Jacobian logarithm
% of the corresponding log-confidences. The Jacobian logarithm is defined
% in the jac.m file. If A = log(a) and B = log(b), then
% log(a+b) = max(A,B) + log(1+exp(-abs(A-B))) (Equation 2.12 in Liang Li's
% nine month report).

% Matrix to describe the trellis
% Each row describes one transition in the trellis
% Each state is allocated an index 1,2,3,... Note that this list starts
% from 1 rather than 0.
%               FromState,  ToState,    UncodedBit, EncodedBit1,EncodedBit2
transitions =  [1,          1,          0,          0,          0;
                1,          3,          1,          1,          1;
                2,          1,          0,          0,          1;
                2,          3,          1,          1,          0;
                3,          4,          0,          1,          0;
                3,          2,          1,          0,          1;
                4,          4,          0,          1,          1;
                4,          2,          1,          0,          0];

% Find the largest state index in the transitions matrix
% In this example, we have eight states since the code has three memory elements
state_count = max(max(transitions(:,1)),max(transitions(:,2)));
transition_count = size(transitions,1);
codeword_length = size(c_tilde_a,1);

gammas = zeros(transition_count, bit_count, codeword_length+1);
for transition_index = 1:transition_count
    if transitions(transition_index, 3)
        gammas(transition_index, b_tilde_a ~= inf,:) = repmat(b_tilde_a(b_tilde_a ~= inf),[1,1,codeword_length+1]);
    else
        gammas(transition_index, b_tilde_a == inf,:) = -inf;
    end
end
for codebit_index = 1:codeword_length
    for transition_index = 1:transition_count
        if transitions(transition_index, 3+codebit_index)
            gammas(transition_index, c_tilde_a(codebit_index,:) ~= inf,[1:codebit_index, codebit_index+2:codeword_length+1]) = add(gammas(transition_index, c_tilde_a(codebit_index,:) ~= inf,[1:codebit_index, codebit_index+2:codeword_length+1]),repmat(c_tilde_a(codebit_index, c_tilde_a(codebit_index,:) ~= inf),[1,1,codeword_length]));
        else
            gammas(transition_index, c_tilde_a(codebit_index,:) == inf,[1:codebit_index, codebit_index+2:codeword_length+1]) = -inf;
        end
    end
end

%gammas

% Forward recursion to calculate state log-confidences. This is similar to
% Equation 1.13 in Rob's thesis or Equations 5 and 6 in the BCJR paper.
alphas=-inf*ones(state_count,bit_count+1);
alphas(1,1)=0; % We know that this is the first state
for bit_index = 2:bit_count+1
    set = zeros(1,state_count);
    for transition_index = 1:transition_count
        if(set(transitions(transition_index,2)))
            alphas(transitions(transition_index,2),bit_index) = maxstar(alphas(transitions(transition_index,2),bit_index), add(alphas(transitions(transition_index,1),bit_index-1), gammas(transition_index, bit_index-1,1)));
        else
            alphas(transitions(transition_index,2),bit_index) = add(alphas(transitions(transition_index,1),bit_index-1), gammas(transition_index, bit_index-1,1));
            set(transitions(transition_index,2)) = 1;
        end
    end
end

% Viterbi
[large_state, state] = max(alphas(:,end));
%best_state = 1;
for bit_index = bit_count:-1:1
    best_alpha = -inf;    
    for transition_index = 1:size(transitions,1)
        if transitions(transition_index, 2) == state && alphas(transitions(transition_index,1),bit_index) > best_alpha
            best_alpha = alphas(transitions(transition_index,1), bit_index);
            b_hat(bit_index) = transitions(transition_index,3);
            best_state = transitions(transition_index,1);
        end       
    end
    state = best_state;
end

% [LLR, States] = max(alphas);
% bit_hat = zeros(1, bit_count);
% for bit_index = bit_count+1:-1:1
%     for state_index = 1:state_count
%         if (States(bit_index) == transitions(state_index,2)) && (States(bit_index-1) == transitions(state_index,1))
%             bit_hat(bit_index-1) = transitions(state_index,3);
%         end        
%     end
% end
