% BCJR algorithm for a half-rate systematic recursive convolutional code
% having 1 memory element, a generator polynomial of [1,0] and a feedback
% polynomial of [1,1]. For more information, see Section 1.3.2.2 of Rob's
% thesis (http://eprints.ecs.soton.ac.uk/14980) or the BCJR paper
% (http://ieeexplore.ieee.org/xpls/abs_all.jsp?arnumber=1055186).
% Copyright (C) 2008  Robert G. Maunder

% This program is free software: you can redistribute it and/or modify it
% under the terms of the GNU General Public License as published by the
% Free Software Foundation, either version 3 of the License, or (at your
% option) any later version.

% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
% Public License for more details.

% The GNU General Public License can be seen at http://www.gnu.org/licenses/.
% This version of trellis_decoder is used for the decoding of the 1st frame in 6th_adap_arithmatic
%

function [c_tilde_e, b_tilde_e, acs_c, acs_b] = trellis_decoder_bcjr_acs_commencing(c_tilde_a, codewords, p, l, m)

% All calculations are performed in the logarithmic domain in order to
% avoid numerical issues. These occur in the normal domain, because some of
% the confidences can get smaller than the smallest number the computer can
% store. See Section 1.3.2.4 of Rob's thesis for more information on this.
%
% A multiplication of two confidences is achieved using the addition of the
% corresponding log-confidences. If A = log(a) and B = log(b), then
% log(a*b) = A+B (Equation 1.17 in Rob's thesis).
%
% An addition of two confidences is achieved using the Jacobian logarithm
% of the corresponding log-confidences. The Jacobian logarithm is defined
% in the jac.m file. If A = log(a) and B = log(b), then
% log(a+b) = max(A,B) + log(1+exp(-abs(A-B))) (Equation 1.19 in Rob's
% thesis).

global acs;
acs = 0;

%%% p is the probobility vector that needed to be known of the first r/2-1 symbols
% if length(p) ~= size(codewords,1)-1
%     error('length(p) should equal size(codewords,1)-1');
% end

%%% c_tilde_a is the received bits
if size(c_tilde_a,1) ~= size(codewords,2)
    error('size(c_tilde_a,1) should equal size(codewords,2)');
end

codeword_count = size(codewords,1);
%%% r/2, number of codewords, e.g. codewords = [0 1; 1 1]

state_count = 2*size(codewords,1);
%%% 2* r/2 = r

transition_count = 4*size(codewords,1);
%%%  4* r/2 = 2*r, particularly this decoder.m based on Fig.3(a)

codeword_length = size(codewords,2);
%%% n, length of codewords, e.g. codewords = [0 1; 1 1]

bit_count = size(c_tilde_a,2);
%%% number of bit known in advance

% Build the trellis and initialise the gammas
transitions = zeros(transition_count, codeword_length+3);
%%% transition = [from-state1 to-state1 uncoded1 encoded1;from-state2 to-state2 uncoded2 encoded2;.......;]

gammas = zeros(transition_count, bit_count);
%%% (3) in BCJR

for even = 0:1   %%% the upper r/2 states and the lower r/2 states
    for uncoded_bit = 0:1   %%% the input bit, 0 or 1
        for codeword_index = 1:codeword_count   %%% codewords should be wrriten in the proper order according to the trellis
            
            % Set the from state
            transitions(even + 2*uncoded_bit + 4*codeword_index - 3, 1) = codeword_index + even*codeword_count;
            
            % Set the to state
            if uncoded_bit == 0  %%% input bit = 0, diverge
                if codeword_index < codeword_count
                    transitions(even + 2*uncoded_bit + 4*codeword_index - 3, 2) = codeword_index + even*codeword_count + 1; %%% state < r/2, to-state=fromstate+1
                else
                    transitions(even + 2*uncoded_bit + 4*codeword_index - 3, 2) = codeword_index + even*codeword_count;     %%% state = r/2, to-state=fromstate
                end
            else                 %%% input bit = 1, converge
                if even == 0  %%% upper r/2 states converge to the 1st state in the lower r/2 state
                    transitions(even + 2*uncoded_bit + 4*codeword_index - 3, 2) = codeword_count + 1;  % upper part converges to the first state in the lower part codeword_count + 1
                else
                    transitions(even + 2*uncoded_bit + 4*codeword_index - 3, 2) = 1;                   % lower part converges to the first state in the upper part 1
                end
            end
            
            % Set the uncoded bit
            transitions(even + 2*uncoded_bit + 4*codeword_index - 3, 3) = uncoded_bit;
            
            % Set the encoded bits
            transitions(even + 2*uncoded_bit + 4*codeword_index - 3, 4:end) = xor(xor(even,~uncoded_bit),codewords(codeword_index,:));
            
            %%%  even    uncoded_bit      situation
            %%%    0          1        upper,converge
            %%%    0          0        upper,diverge
            %%%    1          1        lower,converge
            %%%    1          1        lower,diverge
            
            % Initialise the gammas
            %%% Equation(10) & (14) in Rob's paper
            
            %             if uncoded_bit == 0
            %                 if codeword_index < codeword_count
            %                     %%% (3)
            %                     gammas(even + 2*uncoded_bit + 4*codeword_index - 3, :) = log((1-sum(p(1:codeword_index)))/(p(codeword_index)+1-sum(p(1:codeword_index))));
            %                 else
            %                     %%% (1)
            %                     gammas(even + 2*uncoded_bit + 4*codeword_index - 3, :) = log((l-codeword_count+sum((codeword_count-1:-1:1).*p))/(1-sum(p)+l-codeword_count+sum((codeword_count-1:-1:1).*p)));
            %                 end
            %                 %%% l is the expectation of bit length of UEC encoder'output, Equation(6)
            %             else
            %                 if codeword_index < codeword_count
            %                     %%% (4)
            %                     gammas(even + 2*uncoded_bit + 4*codeword_index - 3, :) = log(p(codeword_index)/(p(codeword_index)+1-sum(p(1:codeword_index))));
            %                 else
            %                     %%% (2)
            %                     gammas(even + 2*uncoded_bit + 4*codeword_index - 3, :) = log((1-sum(p))/(1-sum(p)+l-codeword_count+sum((codeword_count-1:-1:1).*p)));
            %                 end
            %             end
        end
    end
end

%%%  Equation(9) in BCJR paper, R(Yt,X)
for codebit_index = 1:codeword_length
    for transition_index = 1:transition_count
        if transitions(transition_index, 3+codebit_index)
            gammas(transition_index, c_tilde_a(codebit_index,:) ~= inf) = add(gammas(transition_index, c_tilde_a(codebit_index,:) ~= inf),c_tilde_a(codebit_index, c_tilde_a(codebit_index,:) ~= inf));
        else
            gammas(transition_index, c_tilde_a(codebit_index,:) == inf) = -inf;
        end
    end
end


% Forward recursion to calculate state log-confidences. This is similar to
% Equation 1.13 in Rob's thesis or Equations 5 and 6 in the BCJR paper.
alphas=-inf*ones(state_count,bit_count);
alphas(1,1)=0; % We know that this is the first state
for bit_index = 2:bit_count
    set = zeros(1,state_count);
    for transition_index = 1:transition_count
        if(set(transitions(transition_index,2))) %%% determine if this is the first addition, in order to avoid the maxstar calculation
            alphas(transitions(transition_index,2),bit_index) = maxstar(alphas(transitions(transition_index,2),bit_index), add(alphas(transitions(transition_index,1),bit_index-1), gammas(transition_index, bit_index-1)));
        else
            alphas(transitions(transition_index,2),bit_index) = add(alphas(transitions(transition_index,1),bit_index-1), gammas(transition_index, bit_index-1));
            set(transitions(transition_index,2)) = 1;
        end
    end
end



% Backwards recursion to calculate state log-confidences. This is similar
% to Equation 1.14 in Rob's thesis or Equations 7 and 8 in the BCJR paper.
betas=-inf*ones(state_count,bit_count);
if mod(m,2) == 0
    betas(1,end)=0; % We know that this is the final state
else
    betas(codeword_count+1,end)=0; % We know that this is the final state
end
for bit_index = bit_count-1:-1:1
    set = zeros(1,state_count);
    for transition_index = 1:transition_count
        if(set(transitions(transition_index,1)))
            betas(transitions(transition_index,1),bit_index) = maxstar(betas(transitions(transition_index,1),bit_index), add(betas(transitions(transition_index,2),bit_index+1), gammas(transition_index, bit_index+1)));
        else
            betas(transitions(transition_index,1),bit_index) = add(betas(transitions(transition_index,2),bit_index+1), gammas(transition_index, bit_index+1));
            set(transitions(transition_index,1)) = 1;
        end
    end
end



% Calculate a posteriori transition log-confidences. This is similar to
% Equation 1.15 in Rob's thesis or Equation 4 in the BCJR paper.
deltas=zeros(transition_count,bit_count);
for transition_index = 1:transition_count
    deltas(transition_index, :) = add(alphas(transitions(transition_index,1),:), betas(transitions(transition_index,2),:), gammas(transition_index, :));
end

acs_temp = acs;
acs = 0;

bc_tilde_p = zeros(codeword_length+1,bit_count);
for codebit_index = 1:1
    prob1=-inf*ones(1,bit_count);  %%%  Pr(x=1)
    prob0=-inf*ones(1,bit_count);  %%%  Pr(x=0)
    set1 = 0;
    set0 = 0;
    for transition_index = 1:transition_count
        if transitions(transition_index,2+codebit_index)
            if set1
                prob1 = maxstar(prob1, deltas(transition_index,:));
            else
                prob1 = deltas(transition_index,:);   %%% avoid the maxstar operation for the first time
                set1 = 1;
            end
        else
            if set0
                prob0 = maxstar(prob0, deltas(transition_index,:));
            else
                prob0 = deltas(transition_index,:);
                set0 = 1;
            end
        end
        
    end
    bc_tilde_p(codebit_index,:) = add(prob1,-prob0);
end

b_tilde_e = bc_tilde_p(1,:);

acs_b = acs_temp + acs;

acs = 0;

% Calculate the extrinsic LLRs. This is similar to Equation 1.16 in
% Rob's thesis.

bc_tilde_p = zeros(codeword_length+1,bit_count);
for codebit_index = 2:codeword_length+1
    prob1=-inf*ones(1,bit_count);  %%%  Pr(x=1)
    prob0=-inf*ones(1,bit_count);  %%%  Pr(x=0)
    set1 = 0;
    set0 = 0;
    for transition_index = 1:transition_count
        if transitions(transition_index,2+codebit_index)
            if set1
                prob1 = maxstar(prob1, deltas(transition_index,:));
            else
                prob1 = deltas(transition_index,:);   %%% avoid the maxstar operation for the first time
                set1 = 1;
            end
        else
            if set0
                prob0 = maxstar(prob0, deltas(transition_index,:));
            else
                prob0 = deltas(transition_index,:);
                set0 = 1;
            end
        end
        
    end
    bc_tilde_p(codebit_index,:) = add(prob1,-prob0);
end

c_tilde_e = add(bc_tilde_p(2:end,:), -c_tilde_a);    %%% According to Fig.2(a) in Rob's paper
c_tilde_e(isinf(c_tilde_a)) = c_tilde_a(isinf(c_tilde_a));

acs_c = acs_temp + acs;

end
