#!/usr/bin/env python
from __future__ import unicode_literals, print_function, division
import csv
import os

TEMPLATE_NAMES_FILENAME = "raw/template-names.txt"
HEADERS = ["template", "name", "size", "binding", "size", "expansion", "size"]

template_names = dict()

if os.path.isfile(TEMPLATE_NAMES_FILENAME):
    with open(TEMPLATE_NAMES_FILENAME) as f:
        reader = csv.reader(f)
        for template_id, name in reader:
            template_names[template_id] = name


def filename_without_ext(filepath):
    name, _ = os.path.splitext(os.path.basename(filepath))  # removing the path and ext
    return name


# Reading file sizes into memory
def read_filesizes_from_file(filename):
    filesizes = dict()
    with open(filename) as f:
        reader = csv.reader(f, skipinitialspace=True)
        for row in reader:
            filepath, size = row
            size = int(size)  # converting filesize to int
            name = filename_without_ext(filepath)
            filesizes[name] = size
    return filesizes


def read_mappings_file(filename):
    rows = []
    with open(filename) as f:
        for line in f:
            row = list(map(filename_without_ext, line.split()))
            rows.append(row)
    return rows


template_sizes = read_filesizes_from_file("outputs/templates_output.txt")
bindings_v1_sizes = read_filesizes_from_file("outputs/bindings_output.txt")
bindings_v2_sizes = read_filesizes_from_file("outputs/bindings2_output.txt")
expansion_sizes = read_filesizes_from_file("outputs/expansions_output.txt")
mappings = read_mappings_file("raw/index.txt")


def write_stats_table(filename, bindings_sizes):
    with open(filename, "w") as f:
        writer = csv.writer(f)
        writer.writerow(HEADERS)

        for template_id, bindings_id, expansion_id in mappings:
            # Skipping missing data points
            if template_id not in template_sizes:
                continue
            if bindings_id not in bindings_sizes:
                continue
            if expansion_id not in expansion_sizes:
                continue

            writer.writerow((
                template_names.get(template_id, template_id), template_sizes[template_id],
                bindings_id, bindings_sizes[bindings_id],
                expansion_id, expansion_sizes[expansion_id]
            ))


write_stats_table("outputs/normalised_stats.csv", bindings_v1_sizes)
write_stats_table("outputs/normalised2_stats.csv", bindings_v2_sizes)
