% ======================== Description ==================================
% This script computes the cochlear input impedance at given frequencies

% Detailed description can be found in:

% Modelling the effect of round window stiffness on residual hearing 
%                  after cochlear implantation
%            Elliott et al. (2016) Hearing Research


% This script is based on Figure 5 and Equation (4)
% Parameter values used here are listed in Table 1 


% INPUT: 
%       freq:  frequencies of interest, can be either a scalr of a vector [Hz]
% OUTPUT: 
%       Z_C: cochlear input impedance [Pa s m^-3]
 

% Written by          Guangjian NI       06/08/2016
%                     ISVR University of Southampton
%                     niguangjian@gmail.com

% MATLAB release: MATLAB R2015b
% 

function Z_C=CochleaInputImpedance(freq)
%% Parameters

if nargin ==0
    fprintf('Please input simulating frequency/frequencies in Hz\n')
    fprintf('AS:    f_CochleaInputImpedance(freq)\n')
    fprintf('freq can be either a scalar or a vector \n')
        return
elseif nargin > 1;
        fprintf('Only ONE input is needed !!!!\n')    
    return
end

w=2*pi*freq;% Frequency [rad/s]
s=1i*w;     % 

% Inertance of the fluid in the two chambers and the helicotrema (Eq. 2)
L_FL=7e7; 

% Wave impedance (Aibara et al., 2001)
R_WA=2.0e10; 

% Round window properties (fitted based on nakajima et al., 2009)
L_RW=1.0e6;     % Round window inertance [N s^2 m^-5]
R_RW=2.5e9;     % Round window Resistance [N s m^-5]
C_RW0=1.0e-13;   % Round window norminal compliance [N^-1 m^5]
C_RW=C_RW0;
% You may change this C_RW to see the effect of Round Window
% stiffening after cochlear implantation

rho=1000; % cochlear fluid density [kg/m^3]
eta=7e-4; % coefficient of viscosity for cochelar fluid [kg m^-1 s^-1]

% Cochlear aqueduct and Vestibular aqueduct detailes can be found in 
% Appendix B: Acoustic Impedance of the Cochlear Aqueduct and
% Vestibular Aqueduct

% Cochlear aqueduct inertance and resistance Eqs. (B4,B5) 
l_CA=10e-3; % Cochlear aqueduct duct length [m] (Stenfelt, 2015) 
a_CA=0.15e-3/2; % Cochlear aqueduct duct radius [m] (Stenfelt, 2015)
L_CA=rho*l_CA/(pi*a_CA^2); % Cochlear aqueduct inertance
R_CA=l_CA/(pi*a_CA^3)*sqrt(eta*w*rho/2); % Cochlear aqueduct resistance Eq. (B5)

% Vestibular aqueduct inertance and resistance Eqs. (B4,B5) 
l1_VA=1.5e-3;
a1_VA=0.3e-3/2;
L1_VA=rho*l1_VA/(pi*a1_VA^2);
R1_VA=l1_VA/(pi*a1_VA^3)*sqrt(eta*w*rho/2);

l2_VA=8.5e-3; 
a2_VA=0.6e-3/2;
L2_VA=rho*l2_VA/(pi*a2_VA^2);
R2_VA=l2_VA/(pi*a2_VA^3)*sqrt(eta*w*rho/2);

L_VA=L1_VA+L2_VA; % Vestibular aqueduct inertance
R_VA=R1_VA+R2_VA; % Vestibular aqueduct resistance
%% Cochlear input impedance
Z_VA=s*L_VA+R_VA; % Vestibular aqueduct impedance
Z_CA=s*L_CA+R_CA; % Cochlear aqueduct impedance
Z_RW=s*L_RW+R_RW+1./(s*C_RW);  % Round window impedance 
Z_BM=1./(1./R_WA+1./(s*L_FL)); % Basilar membrane impedance
% cochlear input impedance Eq(4)
Z_C=Z_VA.*(Z_BM.*Z_RW+Z_BM.*Z_CA+Z_CA.*Z_RW)./((Z_VA+Z_BM).*(Z_RW+Z_CA)+Z_CA.*Z_RW); 