% ======================== Description ==================================
% This script computes the acoustic impedance at the ear drum, as Eq. (8)

% Detailed description of this 2-port Middle ear model can be found in:

% Kringlebotn, M. 1988. Network model for the human middle ear.
% Scandinavian audiology 17, 75-85.

% This script is adapted from the model that was initially written by
% Emery Ku, in
% Modelling the human cochlea,2008,University of Southampton, Southampton.
% http://eprints.soton.ac.uk/64535/based on Figure 5 and Equation (4)
% and updated by Ben Lineton, in
% "Middle Ear Mechanics" for"Cochlear Mechanics workshop", ISVR, Southampton,
% 2009
% Lecture note is avaiable at
% Eprints link (will be added when available)

% This script needs to call another two fucntions
% 1. Z_parallel.m      For calculating a parallel impedance, e.g. [R1R2/(R1+R2)]
% 2. CochleaInputImpedance.m  For calculating cochlear input impedance

% INPUT:
%       Fre: frequencies of interest, can be either a scalr of a vector [Hz]
% OUTPUT:
%       Ped_by_Qed: Acoustic impedance next to the tympanic membrane at the
%                   ear drum  [Pa s m^-3]


% Written by          Guangjian NI       06/08/2016
%                     ISVR University of Southampton
%                     niguangjian@gmail.com

% MATLAB release: MATLAB R2015b
%
function Ped_by_Qed = EarDrumImpedance(freq)

if nargin ==0
    fprintf('Please input simulating frequency/frequencies in Hz\n')
    fprintf('AS:    EarDrumImpedance(freq)\n')
    fprintf('freq can be either a scalar or a vector \n')
        return
elseif nargin > 1;
        fprintf('Only ONE input is needed !!!!\n')    
    return
end
%% Define Outer and Middle Ear Terms

% % %  Middle Ear components (Kringlebotn,1988)  % % %
cgs_La_ME = 1e-3;                   % g*cm^-4
cgs_Ld_ME = 7.5e-3;                 % g*cm^-4
cgs_Ls_ME = 66e-3;                  % g*cm^-4
% cgs_Lr_ME = 0;                      % MISSING IN TABLE
% cgs_Lm_ME = 0;                      % MISSING IN TABLE
cgs_Lo_ME = 22e-3;                  % g*cm^-4
cgs_Lc_ME = 46e-3;                  % g*cm^-4
cgs_Ca_ME = 3.9e-6;                 % cm^4*s^2*g^-1
cgs_Ct_ME = 0.4e-6;                 % cm^4*s^2*g^-1
cgs_Cs_ME = 0.3e-6;                 % cm^4*s^2*g^-1
cgs_Cr_ME = 1.3e-6;                 % cm^4*s^2*g^-1
cgs_Cm_ME = 0.38e-6;                % cm^4*s^2*g^-1
% cgs_Co_ME = inf;                    % MISSING IN TABLE
cgs_Ci_ME = 0.3e-6;                 % cm^4*s^2*g^-1 WRONG IN BENOIT
cgs_Cc_ME = 0.56e-6;
cgs_Ra_ME = 60;                     % g*cm^-4*s^-2
cgs_Rs_ME = 20;                     % g*cm^-4*s^-2
cgs_Rr_ME = 120;                    % g*cm^-4*s^-2
cgs_Rm_ME = 120;                    % g*cm^-4*s^-2
cgs_Ro_ME = 20*10;                     % g*cm^-4*s^-2
cgs_Ri_ME = 6000;                   % g*cm^-4*s^-2

% Conversion Factors for Changing Parameters from CGS to SI Units
%  1 cm = 0.01 m
cm_to_m = 0.01;
%  1 g = 0.001 kg
g_to_kg = 0.001;
% % %  Middle Ear % % %
La_ME = cgs_La_ME * g_to_kg*cm_to_m^-4; % g*cm^-4 => kg*m^-4
Ld_ME = cgs_Ld_ME * g_to_kg*cm_to_m^-4; % g*cm^-4 => kg*m^-4
Ls_ME = cgs_Ls_ME * g_to_kg*cm_to_m^-4; % g*cm^-4 => kg*m^-4
% Lr_ME = cgs_Lr_ME * g_to_kg*cm_to_m^-4; % g*cm^-4 => kg*m^-4
% Lm_ME = cgs_Lr_ME * g_to_kg*cm_to_m^-4; % g*cm^-4 => kg*m^-4
Lo_ME = cgs_Lo_ME * g_to_kg*cm_to_m^-4; % g*cm^-4 => kg*m^-4
Lc_ME = cgs_Lc_ME * g_to_kg*cm_to_m^-4; % g*cm^-4 => kg*m^-4
Ca_ME = cgs_Ca_ME * cm_to_m^4*g_to_kg^-1; % cm^4*s^2*g^-1 => m^4*s^2*g^-1
Ct_ME = cgs_Ct_ME * cm_to_m^4*g_to_kg^-1; % cm^4*s^2*g^-1 => m^4*s^2*g^-1
Cs_ME = cgs_Cs_ME * cm_to_m^4*g_to_kg^-1; % cm^4*s^2*g^-1 => m^4*s^2*g^-1
Cr_ME = cgs_Cr_ME * cm_to_m^4*g_to_kg^-1; % cm^4*s^2*g^-1 => m^4*s^2*g^-1
Cm_ME = cgs_Cm_ME * cm_to_m^4*g_to_kg^-1; % cm^4*s^2*g^-1 => m^4*s^2*g^-1
% Co_ME = cgs_Co_ME * cm_to_m^4*g_to_kg^-1; % cm^4*s^2*g^-1 => m^4*s^2*g^-1
Ci_ME = cgs_Ci_ME * cm_to_m^4*g_to_kg^-1; % cm^4*s^2*g^-1 => m^4*s^2*g^-1
Cc_ME = cgs_Cc_ME * cm_to_m^4*g_to_kg^-1; % cm^4*s^2*g^-1 => m^4*s^2*g^-1
Ra_ME = cgs_Ra_ME * g_to_kg*cm_to_m^-4; % g*cm^-4*s^-2 => kg*m^-4*s^-2
Rs_ME = cgs_Rs_ME * g_to_kg*cm_to_m^-4; % g*cm^-4*s^-2 => kg*m^-4*s^-2
Rr_ME = cgs_Rr_ME * g_to_kg*cm_to_m^-4; % g*cm^-4*s^-2 => kg*m^-4*s^-2
Rm_ME = cgs_Rm_ME * g_to_kg*cm_to_m^-4; % g*cm^-4*s^-2 => kg*m^-4*s^-2
Ro_ME = cgs_Ro_ME * g_to_kg*cm_to_m^-4; % g*cm^-4*s^-2 => kg*m^-4*s^-2
Ri_ME = cgs_Ri_ME * g_to_kg*cm_to_m^-4; % g*cm^-4*s^-2 => kg*m^-4*s^-2
%%  Begin frequency-loop
%   initialise
Ped_by_Qed = zeros(length(freq),1);

for i = 1:length(freq)
    f = freq(i);
    w = 2*pi*f;
    %% Individual Impedances
    s = 1j*w;
    Za_ME = Ra_ME + s.*La_ME + 1./(s.*Ca_ME);
    Zt_ME =                    1./(s.*Ct_ME);
    Zd_ME =         s.*Ld_ME;
    Zs_ME = Rs_ME + s.*Ls_ME + 1./(s.*Cs_ME);
    Zr_ME = Rr_ME            + 1./(s.*Cr_ME);
    Zm_ME = Rm_ME            + 1./(s.*Cm_ME);
    Zo_ME = Ro_ME + s.*Lo_ME;% + 1./(s.*Co_ME);
    Zi_ME = Ri_ME            + 1./(s.*Ci_ME);
    Zstapes_ME =    s.*Lc_ME + 1./(s.*Cc_ME);  % Impedance of the stapes without cochlear load
    %% Gain terms
    S_ed=0.6;                % [cm^2]
    k1 = S_ed*cm_to_m^2;     % Area of eardrum = 0.6 [cm^2]
    k2 = 1.3;                % ratio of v_m to v_i [unitless]
    k3 = 31.25*cm_to_m^-2;   % Inverse Area of stapes footplate [cm^-2]
    k1k2k3 = k1*k2*k3;       % total transformer ratio for middle ear
    %% Numbered Impedances
    Z1_ME = Za_ME.*Zt_ME./(Za_ME+Zt_ME) + Zd_ME + Zs_ME.*Zr_ME./(Zs_ME+Zr_ME);
    Z2_ME = Zm_ME;
    Z3_ME = Zo_ME;
    Z4_ME = Zi_ME;
    Z5_ME = Zstapes_ME;     % Impedance of the stapes
    % Z5_ME = 2e20; % in the case of Otosclerotic condition
    
    %%%%%   Transfer Matrix  (Ku,2008 Appendix A) %%%%%
    T_edst11 = (Z1_ME+Z2_ME) .* (Z2_ME+Z3_ME+Z4_ME)./(Z2_ME.*Z4_ME) - (Z2_ME./Z4_ME);
    A_ME = T_edst11/k1k2k3;
    
    % Z4 in parallel with Z5
    Z4pZ5 = Z_parallel(Z4_ME,Z5_ME);
    
    % Z2 in parallel with [Z3 + (Z4 in parallel with Z5)]
    Z2p_Z3plusZ4pZ5 = Z_parallel(Z2_ME,Z3_ME + Z4pZ5);
    
    % First voltage divider ratio (V1*V_div1 = Voltage across Z2)
    V_div1 = Z2p_Z3plusZ4pZ5./(Z1_ME + Z2p_Z3plusZ4pZ5);
    
    % Second voltage divider ratio (Voltage across Z2 * V_div2 = Vout)
    V_div2 = Z4pZ5./(Z3_ME + Z4pZ5);
    V_divs = V_div1.*V_div2;
    
    T_edst12 = Z5_ME./V_divs;
    B_ME = T_edst12*k1k2k3;
    
    T_edst21 = (Z2_ME+Z3_ME+Z4_ME)./(Z2_ME.*Z4_ME);
    C_ME = T_edst21/k1k2k3;
    
    %---%   D transmission element
    T_edst22 = ((Z2_ME+Z3_ME).*(Z4_ME+Z5_ME) + Z4_ME.*Z5_ME)./(Z2_ME.*Z4_ME);
    D_ME = T_edst22*k1k2k3;
    
    Z_C=CochleaInputImpedance(f);% obtain cochlear input impedance
    
    Ped_by_Qed(i)=(A_ME*Z_C+B_ME)/(C_ME*Z_C+D_ME);
end
