#include "mainLoopFunctions.h"
#include <argos3/core/simulator/simulator.h>
#include <argos3/core/utility/configuration/argos_configuration.h>
#include <argos3/plugins/robots/foot-bot/simulator/footbot_entity.h>




//#include <argos3/core/utility/plugins/factory.h>

/*================================== INITIALISATION =================== */

MainLoopFunctions::MainLoopFunctions() : m_cForagingArenaSideX(-10.0, 10.0), m_cForagingArenaSideY(-10.0, 10.0), floor(NULL), randNumGenerator(NULL) {

}


void MainLoopFunctions::Init(TConfigurationNode& t_node) {
   try {
      TConfigurationNode& tForaging = GetNode(t_node, "foraging");
      //-- get a pointer to the floor entity
      floor = &GetSpace().GetFloorEntity();

      //-- task parameters
      GetNodeAttribute(tForaging, "taskGradientRadius", taskGradientRadius);
      GetNodeAttribute(tForaging, "taskMinDistanceFromBase", taskMinDistanceFromBase);
      GetNodeAttribute(tForaging, "taskMaxDistanceFromBase", taskMaxDistanceFromBase);
      GetNodeAttribute(tForaging, "taskDisappearRate", taskDisappearRate);
      GetNodeAttribute(tForaging, "taskRenewabilityThreshold", taskRenewabilityThreshold);
      GetNodeAttribute(tForaging, "evenlySpacedTasks", evenlySpacedTasks);
      GetNodeAttribute(tForaging, "taskAppearRate", taskAppearRate);
      GetNodeAttribute(tForaging, "taskAppearTime", taskAppearTime);
      GetNodeAttribute(tForaging, "taskAppearStartingPercentage", taskAppearStartingPercentage);
      GetNodeAttribute(tForaging, "taskValueChangeRate", taskValueChangeRate);
      GetNodeAttribute(tForaging, "minRobotsForTask", minRobotsForTask);
      GetNodeAttribute(tForaging, "maxRobotsForTask", maxRobotsForTask);

      //-- convert times in seconds to time in ticks
      taskAppearRate *= 10;
      taskAppearTime *= 10;
      taskDisappearRate *= 10;

      std::string taskType;
      std::string collectStr = "COLLECT";
      GetNodeAttribute(tForaging, "defaultTaskType", taskType);
      if (taskType.compare(collectStr) == 0) {
    	  defaultTaskType = BaseRobot::TASK_COLLECT;
      } else {
    	  defaultTaskType = BaseRobot::TASK_PERFORM;
      }

      //-- create a new RNG
      randNumGenerator = CRandom::CreateRNG("argos");


      GetNodeAttribute(tForaging, "outputFileNamePrefix", outputFileNamePrefix);

      OnNewRun();


   }
   catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error parsing loop functions!", ex);
   }
}


void MainLoopFunctions::Reset() {
	outputFileGlobal.close();
	outputFileInfoEvents.close();
	OnNewRun();

}

void MainLoopFunctions::OnNewRun() {

	////LOGERR << "=============== " << outputFileNamePrefix << std::endl;

	workPerformed = 0;
	timeTillNextTaskDisappears = taskDisappearRate;
	timeTillNextTaskAppears = 0;
	timeTillQualityReassigned = taskValueChangeRate;

	//-- reset the output file
	std::ostringstream fileNameGlobal;
	fileNameGlobal << outputFileNamePrefix << "_global.txt";
	outputFileGlobal.open(fileNameGlobal.str(), std::ios_base::trunc | std::ios_base::out);
	outputFileGlobal << "time\tworkDone\trandSeed=" << std::endl;

	std::ostringstream fileNameRobots;
	fileNameRobots << outputFileNamePrefix << "_infoEvents.txt";
	outputFileInfoEvents.open(fileNameRobots.str(), std::ios_base::trunc | std::ios_base::out);
	outputFileInfoEvents << "time\tevtType\t\trobotId\ttaskId\ttaskVal\ttaskVol\ttaskU\trecruiterId\tD" << std::endl;

	//-- give robots ids
	CSpace::TMapPerType& robots = GetSpace().GetEntitiesByType("foot-bot");
	for (UInt32 i=0; i<robots.size(); i++) {
		std::ostringstream oss;
		oss << "robot" << i;
		CFootBotEntity& footBotEntity = *any_cast<CFootBotEntity*>(robots[oss.str()]);
		BaseRobot& robot = dynamic_cast<BaseRobot&>(footBotEntity.GetControllableEntity().GetController());
		robot.SetId(i);
	}
	numOfRobots = robots.size();
	////LOG << "[INFO] Number of robots: " << numOfRobots << std::endl;

	//---- create info about items in environment
	taskData.clear();
	CSpace::TMapPerType& tasks = GetSpace().GetEntitiesByType("cylinder");

	Real distanceFromMiddle;
	uint taskId = 0;
	numOfTasks = tasks.size();

	//-- resolve task appear time
	if (taskAppearTime > 0) {
		taskAppearRate = taskAppearTime / (numOfTasks - ceil(numOfTasks * taskAppearStartingPercentage));
		LOG << "task appear time " << taskAppearTime << "  task appear rate = " << taskAppearRate << std::endl;
	}


	//-- decide if using evenly spaced predefined locations
	numOfPredefinedLocations = 0;
	//-- if evenly spaced tasks, generate exactly the amount of pre-defined locations
	if (evenlySpacedTasks) {
		numOfPredefinedLocations = numOfTasks;
		//-- if renewable tasks, generate pre-specified locations for them, double the amount as tasks
		if (taskRenewabilityThreshold >= 0 || taskDisappearRate > 0) {
			numOfPredefinedLocations = numOfTasks*PREDEFINED_TASK_POSITIONS_MULTIPLIER;
		}
	}


	//-- generate predefined task / group locatins if needed
	////LOGERR << "locaations " << numOfPredefinedLocations << std::endl;
	if (numOfPredefinedLocations > 0) {
		preSpecifiedTaskPositions.clear();
		////LOGERR << "generating " << numOfPredefinedLocations << " predefined locations.. " << std::endl;
		Real startingAngle = randNumGenerator->Uniform(CRange<Real>(0,360));
		Real angleStep = 360.0/(numOfPredefinedLocations);
		for (UInt32 i=0; i<numOfPredefinedLocations; i++) {
			CRadians angle = CRadians(0);
			angle.FromValueInDegrees(startingAngle+angleStep*i);
			////LOGERR << i << " angle " << (startingAngle+angleStep*i) << std::endl;
			preSpecifiedTaskPositions.push_back(CVector2(randNumGenerator->Uniform(CRange<Real>(taskMinDistanceFromBase,taskMaxDistanceFromBase)), angle));
		}
	}

	//-- distribute the tasks
	numOfActiveTasks = numOfTasks;
	if (taskAppearRate > 0){
		numOfActiveTasks = ceil(numOfTasks * taskAppearStartingPercentage);
	}
	maxTaskId = 0;
	for (int taskId=0; taskId<numOfTasks; taskId++) {
		std::ostringstream oss;
		oss << "task" << taskId;
		CCylinderEntity& task = *any_cast<CCylinderEntity*>(tasks[oss.str()]);
		CVector2 taskPos = CVector2(task.GetEmbodiedEntity().GetPosition().GetX(), task.GetEmbodiedEntity().GetPosition().GetY());

		//-- uniform random distribution, make sure tasks are at least minDistance and no more than maxDistance away
		taskPos = GenerateTaskPosition(task);


		//-- adjust task value (resource quality)
		task.SetMass(GenerateTaskMass(taskId));
		Real netResource = (task.GetMass()-100000);

		//-- measure height
		if (taskId==0) {
			originalTaskHeight = task.GetHeight();
			taskRadius = task.GetRadius();
		}

		taskData.push_back(TaskInfo(CVector2(taskPos), defaultTaskType, randNumGenerator->Uniform(CRange<uint>(minRobotsForTask,maxRobotsForTask+1)), taskId ));
		////LOGERR << " task " << taskId << " robots req " << taskData[taskId].numOfRobotsRequired << std::endl;

		//-- keep the task on the ground if it should appear initially. Otherwise, move it to a high z position and set it as inactive (e..g when appearRate > 0)
		if (taskId < numOfActiveTasks) {
			//-- output
			Real taskDistanceFromBase = taskData[taskId].position.Length() - 3;
			outputFileInfoEvents << GetSpace().GetSimulationClock()/10 << "\t" << GetEnvironmentEventTypeDescription(E_EVENT_TASK_ADDED) << "\t" << -1 << "\t"
								<< taskId << "\t" << task.GetMass()-100000 << "\t" << task.GetHeight() << "\t" << 0 << "\t" << -1 << "\t" << taskDistanceFromBase << std::endl;

		} else {
			task.GetEmbodiedEntity().MoveTo(CVector3(15, 15, 0.5), CQuaternion());
			task.GetEmbodiedEntity().MoveTo(CVector3(taskPos.GetX(), taskPos.GetY(), 2), CQuaternion());
			taskData[taskId].isActive = false; //tell the floor not to render the gradient around
		}
		maxTaskId = taskId;

	}
	minTaskId = 0;

	if (taskAppearRate > 0) {
		timeTillNextTaskAppears = taskAppearRate;
	}

}


void MainLoopFunctions::Destroy() {
	outputFileGlobal.close();
	outputFileInfoEvents.close();
}

/* ============================================== */

CColor MainLoopFunctions::GetFloorColor(const CVector2& position_) {
	//-- mark base sections. X and Y coordinates are swapped in the simulator for some reason
	float x = position_.GetX();
	float y = position_.GetY();
	float distanceFromMiddle = sqrt(x*x + y*y);
	if (distanceFromMiddle < 0.8) {
		//-- resting bay
		return CColor::GRAY20;
	} else if (distanceFromMiddle < 2) {
		//-- dance floor
		return CColor::GRAY30;
	} else if (distanceFromMiddle < 3) {
		//-- unloading bay
		return CColor::GRAY10;
	}

	//-- draw gradient around food, goes from 0.5 dark -> black
	for(UInt32 i = 0; i < taskData.size(); ++i) {
		if (taskData[i].isActive) {
			float distanceFromFoodMiddle = (position_ - taskData[i].position).Length() - taskRadius;
			float gradientLength = taskGradientRadius - taskRadius;
			if (distanceFromFoodMiddle <= gradientLength){
				UInt8 grayVal = 102 + 128*distanceFromFoodMiddle/gradientLength; //0.4 - 0.9
				return CColor(grayVal,grayVal,grayVal);
			}
		}
	}
	return CColor::WHITE;
}

/****************************************/
/****************************************/

void MainLoopFunctions::PreStep() {

	//-- go through all robots
	numOfScouts = 0;
	numOfResters = 0;
	numOfBroadcasters = 0;
	numOfPerformers = 0;
	numOfGoingToTask = 0;

	CSpace::TMapPerType& m_cFootbots = GetSpace().GetEntitiesByType("foot-bot");

	for(CSpace::TMapPerType::iterator it = m_cFootbots.begin(); it != m_cFootbots.end(); ++it) {

      CFootBotEntity& footBotEntity = *any_cast<CFootBotEntity*>(it->second);
      BaseRobot& footBot = dynamic_cast<BaseRobot&>(footBotEntity.GetControllableEntity().GetController());
      CVector3 robotPosition = footBotEntity.GetEmbodiedEntity().GetPosition();

      Real amountToProcess = 0;
      Real totalRewardInTask = 0;

      //-- get nearby task info
      int taskId = GetTaskIdByPosition(robotPosition);
      CCylinderEntity* task = NULL;
      footBot.SetNearbyTaskInfo(BaseRobot::TASK_NONE, taskData[taskId].reportedTaskId, -1,-1, 0);
      if (taskId >= 0) {
    	  task = GetTaskById(taskId);
    	  if (task != NULL) {
    		  totalRewardInTask = task->GetHeight()*(task->GetMass()-100000);
    		  footBot.SetNearbyTaskInfo(taskData[taskId].type, taskData[taskId].reportedTaskId, task->GetMass()-100000 , task->GetHeight(), taskData[taskId].numOfRobotsRequired);
    	  }
      }

      footBot.PreControlStep();


      if (footBot.GetIsProcessingTask()) {
    	  //-- check if the task it is gathering from has resource left

		  if (taskId >= 0) {

			  if (task != NULL) {

				  amountToProcess = task->GetHeight();
				  Real loadStep = 1.0/footBot.GetLoadTimePerVolumeUnit();
				  if (amountToProcess > loadStep) {
					  amountToProcess = loadStep;
				  }
				  if (taskData[taskId].isActive == false) {
					  amountToProcess = 0;
				  }

				  /*if (footBot.GetNumId() == 12) {
					  LOG << footBot.GetId() << " loading from " << taskId << " amount " << amountToProcess << std::endl;
				  }*/

				  //-- pass the amount and net resource left (quantity*quality) to the robot. Quality offset by 100000

				  Real amountProcessed = footBot.ProcessTask(taskData[taskId].type, taskData[taskId].reportedTaskId, amountToProcess, task->GetMass()-100000, totalRewardInTask, taskData[taskId].numOfRobotsRequired);
				  /*if (footBot.GetNumId() == 12) {
					  LOG << "processed " << amountProcessed << std::endl;
				  }*/
				  task->SetHeight(task->GetHeight() - amountProcessed);

				  //-- count it as performance if task is done on the spot
				  if (taskData[taskId].type != BaseRobot::TASK_COLLECT) {
					  workPerformed += amountProcessed*(task->GetMass()-100000);
					  footBot.OnTaskRewardReceived(taskData[taskId].type, taskData[taskId].reportedTaskId, amountProcessed*(task->GetMass()-100000), task->GetMass()-100000, task->GetHeight());
					  if (amountProcessed > 0) {
						  //LOG << " work performed " << workPerformed << std::endl;
					  } else {
						  //LOG << " empty task with height " << task->GetHeight() << " am to process was " << amountToProcess << std::endl;
					  }
				  }


				  //-- check if task should be renewed or depleted
				  UpdateTaskAfterProcessingStep(*task, taskId);
			  } else {
				  //-- tell robot there is nothing to load
				  footBot.ProcessTask(taskData[taskId].type,taskData[taskId].reportedTaskId, 0, 0, 0, taskData[taskId].numOfRobotsRequired);
			  }

		  } else {
			  //-- tell robot there is nothing to load
			  footBot.ProcessTask(BaseRobot::TASK_NONE, -1, 0, 0, 0, 0);
		  }


      } else if (footBot.GetIsUnloadingResource()) {
    	  //-- collect resource from the robot
		  Real robotOutput = footBot.PerformUnloadResource();
		  //-- test for NAN value
		  if (robotOutput != robotOutput) {
			  robotOutput = 0;
		  }
		  workPerformed += robotOutput;
		  footBot.OnTaskRewardReceived(taskData[taskId].type, taskData[taskId].reportedTaskId, robotOutput, -1, -1);
		  ////LOG << " work performed " << workPerformed << std::endl;
      }
      //========= ROBOT'S REPORTS ======

      //-- info events
      std::vector<BaseRobot::InfoEventReport> infoEventReports = *footBot.GetInfoEventReports();
      for (uint r=0; r<infoEventReports.size(); r++) {
    	  BaseRobot::InfoEventReport infoEventReport = infoEventReports[r];
		  if (infoEventReport.type != BaseRobot::InfoEventReport::TYPE_VOID) {
			  int reportedTaskId = -1;
			  try {
				  reportedTaskId = infoEventReport.taskId;
			  } catch (int e) {
				  LOGERR << "Exception no " << e << std::endl;
			  }
			  if (reportedTaskId < 0) {
				  //-- robot doesn't know task id, get it from the environment based on where the robot is
				  bool includeInactiveTasksInSearch = false;
				  if (infoEventReport.type == BaseRobot::InfoEventReport::TYPE_REWARD_FINISHED || infoEventReport.type == BaseRobot::InfoEventReport::TYPE_TASK_ABANDONED || infoEventReport.type == BaseRobot::InfoEventReport::TYPE_TASK_MISSING) {
					  includeInactiveTasksInSearch = true;
				  }
				  int taskIdFromEnv = GetTaskIdByPosition(robotPosition, includeInactiveTasksInSearch);
				  if (taskIdFromEnv >= 0) {
					  infoEventReport.taskId = taskData[taskIdFromEnv].reportedTaskId;
					  reportedTaskId = infoEventReport.taskId;
				  } else {
					  LOGERR << "!!!!!!!!!!!!!" << std::endl;
				  }
			  }
			  //-- get the task entitiy associated with the task id and its information
			  Real taskValue = infoEventReport.taskValue;
			  Real taskVolume = infoEventReport.taskVolume;
			  Real taskUtility = infoEventReport.taskUtility;
			  Real taskDistanceFromBase = -1;

			  if (reportedTaskId >= 0) {
				  CCylinderEntity* task = GetTaskById(reportedTaskId);
				  if (task != NULL) {
					  //-- if the robot didn't know the task details (didn't remember them previously), get it from the task itself
					  if (taskData.size() > reportedTaskId) {
						  taskDistanceFromBase = taskData[reportedTaskId].position.Length() - 3;
					  } else {
						  LOGERR << " task data doesn't exist task id " << reportedTaskId << " info report: " << infoEventReport.taskId << " mintaskid " << minTaskId << std::endl;
					  }

					  if (infoEventReport.taskValue < 0) {
						  taskValue = task->GetMass()-100000;
						  taskVolume = task->GetHeight();
						  if (taskData.size() > reportedTaskId) {
							  taskUtility = footBot.EstimateTaskUtility(taskData[reportedTaskId].type,taskValue,taskVolume,taskDistanceFromBase, false);
						  }
					  }

					  if (infoEventReport.type == BaseRobot::InfoEventReport::TYPE_TASK_SCOUTED || infoEventReport.type == BaseRobot::InfoEventReport::TYPE_TASK_REACHED) {
						  taskVolume += amountToProcess; // for scout task, note down the volume before loading
					  }
				  } else {
					  //LOGERR << " task doesn't exist for task id " << reportedTaskId << " info report: " << infoEventReport.taskId << " mintaskid " << minTaskId << std::endl;
				  }

			  }

			  //-- output
			  outputFileInfoEvents << GetSpace().GetSimulationClock()/10 << "\t"
								 << infoEventReport.GetTaskTypeDescription() << "\t"
								 << footBot.GetNumId() << "\t"
								 << infoEventReport.taskId << "\t"
								 << taskValue << "\t"
								 << taskVolume << "\t"
								 << taskUtility << "\t"
								 << infoEventReport.recruiterId << "\t"
								 << taskDistanceFromBase
								 << std::endl;
			  //if (footBot.GetNumId() > 0) {
			  //LOG << "R"  << footBot.GetNumId() << " " << infoEventReport.GetTaskTypeDescription() << " t" << infoEventReport.taskId << " val " << taskValue << std::endl;
			  //}

		  }
      }

      footBot.ClearEventReports();

      //========= ========= ======

   }


   //-- disappear tasks at a certain rate
   if (taskDisappearRate >= 1) {
	   if (timeTillNextTaskDisappears == 0) {
		   timeTillNextTaskDisappears = taskDisappearRate;
		   //LOG << "move tasks " << minTaskId << " to " << (minTaskId + numOfTasks);
		   for (int taskId=minTaskId; taskId<minTaskId + numOfTasks; taskId++) {
			   //uint idToMove = uint(randNumGenerator->Uniform(CRange<Real>(0,numOfTasks)));
			   CCylinderEntity* task = GetTaskById(taskId);
			   task->SetHeight(originalTaskHeight);
			   //LOG << "MOVING " << taskId << std::endl;

			   //-- generate new task id
			   //int oldTaskId = taskData[taskId].reportedTaskId;
			   int oldTaskId = taskId;
			   int newTaskId = maxTaskId+1;
			   maxTaskId++;

			   //-- make the old task data inactive
			   taskData[oldTaskId].isActive = false;

			   //taskData[taskId].reportedTaskId = newTaskId;
			   //taskData[taskId].position = CVector2(GenerateTaskPosition(*task));
			   //-- create new task data, moving the task to the new location

			   taskData.push_back(TaskInfo(CVector2(GenerateTaskPosition(*task)), defaultTaskType, randNumGenerator->Uniform(CRange<uint>(minRobotsForTask,maxRobotsForTask+1)), newTaskId));

			   //-- log it
			   Real taskDistanceFromBase = taskData[taskId].position.Length() - 3;
			   outputFileInfoEvents << GetSpace().GetSimulationClock()/10 << "\t" << GetEnvironmentEventTypeDescription(E_EVENT_TASK_DESTROYED) << "\t" << -1 << "\t"
			   			   			   			<< oldTaskId << "\t" << task->GetMass()-100000 << "\t" << task->GetHeight() << "\t" << 0 << "\t" << -1 << "\t" << taskDistanceFromBase << std::endl;
			   outputFileInfoEvents << GetSpace().GetSimulationClock()/10 << "\t" << GetEnvironmentEventTypeDescription(E_EVENT_TASK_ADDED) << "\t" << -1 << "\t"
			   			   			<< newTaskId << "\t" << task->GetMass()-100000 << "\t" << task->GetHeight() << "\t" << 0 << "\t" << -1 << "\t" << taskDistanceFromBase << std::endl;
		   }
		   minTaskId += numOfTasks;

		   floor->SetChanged();

	   } else {
		   timeTillNextTaskDisappears--;
	   }
   }

   //-- appear tasks at a certain rate
   if (taskAppearRate > 0) {
	   if (timeTillNextTaskAppears == 0) {
		   timeTillNextTaskAppears = taskAppearRate;

		   //- pick the next task and make it available by moving it down and redrawing the floor
		   uint idToMove = numOfActiveTasks;

		   if (idToMove < numOfTasks) {
			   CCylinderEntity* task = GetTaskById(idToMove);
			   task->GetEmbodiedEntity().MoveTo(CVector3(15, 15, 0.5), CQuaternion());
			   task->GetEmbodiedEntity().MoveTo(CVector3(taskData[idToMove].position.GetX(), taskData[idToMove].position.GetY(), 0), CQuaternion());
			   taskData[idToMove].isActive = true;
			   floor->SetChanged();
			   numOfActiveTasks++;

			   //-- log it
			   Real taskDistanceFromBase = taskData[idToMove].position.Length() - 3;
			   outputFileInfoEvents << GetSpace().GetSimulationClock()/10 << "\t" << GetEnvironmentEventTypeDescription(E_EVENT_TASK_ADDED) << "\t" << -1 << "\t"
			   						<< taskData[idToMove].reportedTaskId << "\t" << task->GetMass()-100000 << "\t" << task->GetHeight() << "\t" << 0 << "\t" << -1 << "\t" << taskDistanceFromBase << std::endl;

		   } else {
			   //-- end of appearing tasks
			   taskAppearRate = 0;
		   }

	   } else {
		   timeTillNextTaskAppears--;
	   }
   }

   //-- change quality at a certain rate
   if (taskValueChangeRate > 0) {

	   if (timeTillQualityReassigned == 0) {
		   timeTillQualityReassigned = taskValueChangeRate;
		   //-- create the array that records which group qualities have been assigned
		   bool qualityGroupAssignments[4];
		   uint i;
		   for (i=0; i< 4; i++) { qualityGroupAssignments[i] = false; }
		   //-- go through tasks, renew them and assign them new unique quality

		   for (uint d = 0; d< numOfTasks; d++) {
			   CCylinderEntity* task = GetTaskById(d);
			   task->SetHeight(originalTaskHeight);
			   //-- task could have been depleted, make it active again
			   taskData[d].isActive = true;
			   floor->SetChanged();

			   //-- search for a new unassigned group quality that is different than previous quality of this group
			   int qualityGroupId = randNumGenerator->Uniform(CRange<uint>(0,numOfTasks));
			   bool qualityDifferentForGroup = false;
			   int counter = 0;
			   while (!qualityDifferentForGroup || qualityGroupAssignments[qualityGroupId] == true) {
				   qualityGroupId = randNumGenerator->Uniform(CRange<uint>(0,numOfTasks));
				   counter++;
				   Real oldMass = task->GetMass()-100000;
				   Real newMass = GenerateTaskMass(qualityGroupId)-100000;

				   if (oldMass != newMass || counter > 100) {
					   qualityDifferentForGroup = true;
				   }
			   }
			   //-- mark this group quality as used and assign it
			   qualityGroupAssignments[qualityGroupId] = true;
			   task->SetMass(GenerateTaskMass(qualityGroupId));
			   ////LOG << " Depo " << d << " group quality " << qualityGroupId << "  q " << task->GetMass()-100000 << "  vol " << task->GetHeight() << std::endl;

		   }

	   } else {
		   timeTillQualityReassigned--;
	   }
   }

   //-- output stuff to files:
   if (GetSpace().GetSimulationClock() % 10 == 0) {
	   outputFileGlobal << GetSpace().GetSimulationClock()/10 << "\t"
				 << workPerformed << "\t"
				 << std::endl;
   }
}

//================================================  TASK UPDATES

void MainLoopFunctions::UpdateTaskAfterProcessingStep(CCylinderEntity& task_, int taskId_) {
	Real taskDistanceFromBase = taskData[taskId_].position.Length() - 3;
	if (taskRenewabilityThreshold >= 0) {
		//---- tasks when depleted dissappear and appear somewhere else
		if (task_.GetHeight() - originalTaskHeight*taskRenewabilityThreshold <= ZERO_WORKSITE_VOLUME) {
		   ////LOG << "MOVING " << taskId << std::endl;
			task_.SetHeight(originalTaskHeight);
			taskData[taskId_].position = CVector2(GenerateTaskPosition(task_));
			floor->SetChanged();

			//-- note down the event
			/*outputFileTaskRenewEvents << GetSpace().GetSimulationClock()/10 << "\t"
								  << taskId_ << std::endl;*/
		}
	} else if (task_.GetHeight() <= ZERO_WORKSITE_VOLUME && taskValueChangeRate > 0) {
		//---- tasks that will be renewed at the same location but need to disappear for now - just make them inactive, this will assure that robots will still avoid them
		taskData[taskId_].isActive = false;
		floor->SetChanged();

		//-- output
		outputFileInfoEvents << GetSpace().GetSimulationClock()/10 << "\t" << GetEnvironmentEventTypeDescription(E_EVENT_TASK_COMPLETED) << "\t" << -1 << "\t"
						<< taskData[taskId_].reportedTaskId << "\t" << task_.GetMass()-100000 << "\t" << 0 << "\t" << 0 << "\t" << -1 << "\t" << taskDistanceFromBase << std::endl;

		//LOG << GetEnvironmentEventTypeDescription(E_EVENT_TASK_COMPLETED) << " t" << taskId_ << std::endl;


	} else {
		//---- tasks when depleted dissapear forever
		//LOG << "task " << taskId_ << " height  " << task_.GetHeight() << std::endl;
		if (task_.GetHeight() <= ZERO_WORKSITE_VOLUME) {
			//-- depleted task - move it away and update floor color
			task_.GetEmbodiedEntity().MoveTo(CVector3(30 + taskId_,30,0),CQuaternion());
			taskData[taskId_].position = CVector2(30,30);
			floor->SetChanged();

			//-- output
			outputFileInfoEvents << GetSpace().GetSimulationClock()/10 << "\t" << GetEnvironmentEventTypeDescription(E_EVENT_TASK_COMPLETED) << "\t" << -1 << "\t"
							<< taskData[taskId_].reportedTaskId << "\t" << task_.GetMass()-100000 << "\t" << 0 << "\t" << 0 << "\t" << -1 << "\t" << taskDistanceFromBase << std::endl;

			//LOG << GetEnvironmentEventTypeDescription(E_EVENT_TASK_COMPLETED) << " t" << taskId_ << std::endl;
		}
	}
}

/* ================================================== SCENARIO GENERATION */

Real MainLoopFunctions::GenerateTaskMass(int taskId_) {
	return 100001;
}

/**
 * Generate a position between min and max distance from base for a task and move it there
 */
CVector2 MainLoopFunctions::GenerateTaskPosition(CCylinderEntity& task_) {
	CVector2 taskPos;
	bool can = false;
	int attemptCounter = 0;
	while (!can) {
		if (numOfPredefinedLocations > 0) {

			//-- if there are pre-defined locations, pick randomly from them
			bool prespecifiedLocationFree = false;
			int attemptCounter2 = 0;
			while (!prespecifiedLocationFree) {
				//-- find a free location, this check is needed for groups where group middle could already be depleted so a simple check against it would not work
				taskPos = CVector2(preSpecifiedTaskPositions[uint(randNumGenerator->Uniform(CRange<Real>(0,numOfPredefinedLocations)))]);
				if (GetTaskIdByPosition(CVector3(taskPos.GetX(),taskPos.GetY(), 0), 2.0) >= 0) {
					prespecifiedLocationFree = false;
				} else {
					prespecifiedLocationFree = true;
				}

				attemptCounter2++;
				if (attemptCounter2 > 1000) {
					LOGERR << "    !!!!!!!!!!!!!!!! used all attempts to find a free position, generating a new position " << std::endl;
					taskPos = CVector2(randNumGenerator->Uniform(CRange<Real>(taskMinDistanceFromBase,taskMaxDistanceFromBase)),randNumGenerator->Uniform(CRange<CRadians>(CRadians(0),CRadians(6.5))));
				}

			}

		} else {
			//-- pick a new random position
			taskPos = CVector2(randNumGenerator->Uniform(CRange<Real>(taskMinDistanceFromBase,taskMaxDistanceFromBase)),randNumGenerator->Uniform(CRange<CRadians>(CRadians(0),CRadians(6.5))));
		}
		can = task_.GetEmbodiedEntity().MoveTo(CVector3(taskPos.GetX(), taskPos.GetY(), 0), CQuaternion());

		//-- check additionally if the task's gradient doesn't intersect any other task's gradient
		if (numOfPredefinedLocations == 0) {
			CSpace::TMapPerType& tasks = GetSpace().GetEntitiesByType("cylinder");
			for(CSpace::TMapPerType::iterator it = tasks.begin(); it != tasks.end(); ++it) {
				CCylinderEntity& anotherTask = *any_cast<CCylinderEntity*>(it->second);
				if (anotherTask.GetId() != task_.GetId()) {
					CVector3 anotherTaskPosition = anotherTask.GetEmbodiedEntity().GetPosition();
					if ((anotherTaskPosition - task_.GetEmbodiedEntity().GetPosition()).Length() < taskGradientRadius*2.2) {
						can = false;
					}
				}
			}
		}

		if (!can) {
			attemptCounter ++;
			//LOGERR << "!! Can't use location "<<  taskPos.GetX() << " " << taskPos.GetY() << " attempt counter " << attemptCounter << std::endl;
			if (attemptCounter > 1000) {
				LOGERR << "    !!!!!!!!!!!!!!!! used all attempts, returning the same task position " << std::endl;
				return CVector2(task_.GetEmbodiedEntity().GetPosition().GetX(), task_.GetEmbodiedEntity().GetPosition().GetY());


			}
		}
	}
	return taskPos;
}

/* ================================================== OTHER */

/**
 * Get task array id by a reported id
 */
int MainLoopFunctions::GetRealTaskIdByReportedId(const int& reportedId_) {
	for(UInt32 i = 0; i < taskData.size(); ++i) {
		if (taskData[i].reportedTaskId == reportedId_) {
			return i;
		}
	}
	return -1;
}
/**
 * Returns a task id that is very near a position
 */
int MainLoopFunctions::GetTaskIdByPosition(CVector3 position_, bool includeInactiveTasks_) {
	//return GetTaskIdByPosition(position_, BaseRobot::GetLoadingProximity() + 0.1);
	return GetTaskIdByPosition(position_, taskGradientRadius, includeInactiveTasks_);
}

int MainLoopFunctions::GetTaskIdByPosition(CVector3 position_) {
	return GetTaskIdByPosition(position_, false);
}

/**
 * Returns the first task id that is a certain distance from a position
 */
int MainLoopFunctions::GetTaskIdByPosition(CVector3 position_, Real maxDistance_, bool includeInactiveTasks_) {
	CVector2 position = CVector2(position_.GetX(),position_.GetY());
	for(UInt32 i = 0; i < taskData.size(); ++i) {
		if (taskData[i].isActive || includeInactiveTasks_) {
			float distanceFromFoodMiddle = (position - taskData[i].position).Length() - taskRadius;
			if (distanceFromFoodMiddle <= maxDistance_) {
				return i;
			}
		}
	}
	return -1;
}

int MainLoopFunctions::GetTaskIdByPosition(CVector3 position_, Real maxDistance_) {
	return GetTaskIdByPosition(position_, maxDistance_, false);
}



/**
 * Returns a task that is very near a position
 */
CCylinderEntity* MainLoopFunctions::GetTaskByPosition(CVector3 position_) {
	CVector2 position = CVector2(position_.GetX(),position_.GetY());
	for(UInt32 i = 0; i < taskData.size(); ++i) {
		float distanceFromFoodMiddle = (position - taskData[i].position).Length() - taskRadius;
		if (distanceFromFoodMiddle <= BaseRobot::GetLoadingProximity() + 0.09) {
			//-- found it, retreieve it from the cylinder entities
			std::ostringstream oss;
			oss << "task" << i;
			CSpace::TMapPerType& tasks = GetSpace().GetEntitiesByType("cylinder");
			CCylinderEntity* task = any_cast<CCylinderEntity*>(tasks[oss.str()]);
			return task;
		}
	}
	return NULL;
}

/**
 * Returns a task that has a certain id
 */
CCylinderEntity* MainLoopFunctions::GetTaskById(const int& arrayId_) {
	CSpace::TMapPerType& tasks = GetSpace().GetEntitiesByType("cylinder");
	int id = arrayId_ - minTaskId;
	if (id >= 0 && id < tasks.size()) {
		std::ostringstream oss;
		oss << "task" << id;
		CCylinderEntity* task = any_cast<CCylinderEntity*>(tasks[oss.str()]);
		return task;
	}
	return NULL;
}

/**
 * Returns a robot that has a certain id
 */

BaseRobot& MainLoopFunctions::GetRobotById(const int& id_) {
	CSpace::TMapPerType& robots = GetSpace().GetEntitiesByType("foot-bot");
	std::ostringstream oss;
	oss << "robot" << id_;
	CFootBotEntity& footBotEntity = *any_cast<CFootBotEntity*>(robots[oss.str()]);
	BaseRobot& robot = dynamic_cast<BaseRobot&>(footBotEntity.GetControllableEntity().GetController());
	return robot;

}

/****************************************/
/****************************************/

REGISTER_LOOP_FUNCTIONS(MainLoopFunctions, "mainLoopFunctions")
