
#ifndef BASE_ROBOT_H
#define BASE_ROBOT_H

/*
 * Include some necessary headers.
 */
#include <argos3/core/control_interface/ci_controller.h>
#include <argos3/plugins/robots/generic/control_interface/ci_differential_steering_actuator.h>
#include <argos3/plugins/robots/generic/control_interface/ci_leds_actuator.h>
#include <argos3/plugins/robots/generic/control_interface/ci_range_and_bearing_actuator.h>
#include <argos3/plugins/robots/generic/control_interface/ci_range_and_bearing_sensor.h>
#include <argos3/plugins/robots/foot-bot/control_interface/ci_footbot_proximity_sensor.h>
#include <argos3/plugins/robots/foot-bot/control_interface/ci_footbot_light_sensor.h>
#include <argos3/plugins/robots/foot-bot/control_interface/ci_footbot_motor_ground_sensor.h>
#include <argos3/plugins/robots/generic/control_interface/ci_differential_steering_sensor.h>
#include <argos3/core/utility/math/rng.h>

#include <gsl/gsl_randist.h>


using namespace argos;

class BaseRobot : public CCI_Controller {

#define DELAY_BETWEEN_SCOUTINGS 300; // travel speed: 0.05m/s, gradient is 1m wide -> needs 20s to traverse 1 gradient + add some more for collisions etc

public:

	struct DiffusionParams {
	  /*
	   * Maximum tolerance for the proximity reading between
	   * the robot and the closest obstacle.
	   * The proximity reading is 0 when nothing is detected
	   * and grows exponentially to 1 when the obstacle is
	   * touching the robot.
	   */
	  Real Delta;
	  /* Angle tolerance range to go straight. */
	  CRange<CRadians> GoStraightAngleRange;
	  DiffusionParams();
	  void Init(TConfigurationNode& t_tree);
	};

	struct WheelTurningParams {
	  enum ETurningMechanism {
		 NO_TURN = 0, // go straight
		 SOFT_TURN,   // both wheels are turning forwards, but at different speeds
		 HARD_TURN    // wheels are turning with opposite speeds
	  } turningMechanism;

	  CRadians hardTurnOnAngleThreshold;
	  CRadians softTurnOnAngleThreshold;
	  CRadians noTurnAngleThreshold;
	  Real maxSpeed;

	  void Init(TConfigurationNode& t_tree);
	};

	struct BaseParameters {
		bool isLinuxBased; //true if the simulation is called from linux, e.g. iridis
		uint groundSensorFLId; //id of front left ground sensor
		uint groundSensorBLId;
		uint groundSensorFRId;
		uint groundSensorBRId;
		Real cartCapacity;
		int loadTimePerVolumeUnit;
		int unloadTimePerVolumeUnit;
		uint neighbourhoodSearchRadius;
		uint neighbourhoodSearchTime;
		uint expectedArenaDiameter; //distance from base at which the robot turns back towards the base when searching for tasks
		bool useOpportunism;
		Real opportunismThreshold; // by how much U a task must be better to opportunistically subscribe to it
		bool useRelU;
		void Init(TConfigurationNode& t_tree);
	};

	enum TASK_TYPE {
		TASK_NONE = 0,
		TASK_COLLECT,
		TASK_PERFORM
	};

	struct TaskInfo {
		BaseRobot::TASK_TYPE taskType;
		CVector2 relLocation;
		bool isNull;
		int infoOriginatorId;
		int taskId;
		Real taskValue;
		Real taskVolume;
		Real taskUtility;
		Real relTaskUtility;
		uint numOfRobotsRequired;

		TaskInfo(BaseRobot::TASK_TYPE taskType_, int taskId_, Real taskValue_, CVector2 relLocation_, uint infoOriginatorId_, Real taskUtility_, Real relTaskUtility_, uint numOfRobotsRequired_) {
			taskType = taskType_;
			relLocation = CVector2(relLocation_);
			isNull = false;
			taskVolume = -1;
			infoOriginatorId = infoOriginatorId_;
			taskId = taskId_;
			taskValue = taskValue_;
			taskUtility = taskUtility_;
			relTaskUtility = relTaskUtility_;
			numOfRobotsRequired = numOfRobotsRequired_;
		}

		TaskInfo(BaseRobot::TASK_TYPE taskType_, int taskId_, Real taskValue_, Real taskVolume_, uint numOfRobotsRequired_ ) {
			taskType = taskType_;
			relLocation = CVector2();
			isNull = false;
			taskVolume = taskVolume_;
			infoOriginatorId = -1;
			taskId = taskId_;
			taskValue = taskValue_;
			taskUtility = -1;
			relTaskUtility = -1;
			numOfRobotsRequired = numOfRobotsRequired_;
		}

		void Nullify() {
			isNull = true;
			relTaskUtility = 0;
		}

		TaskInfo() {
			taskType = BaseRobot::TASK_NONE;
			relLocation = CVector2();
			isNull = true;
			infoOriginatorId = -1;
			taskId = -1;
			taskValue = -1;
			taskVolume = -1;
			taskUtility = -1;
			relTaskUtility = -1;
			numOfRobotsRequired = 0;
		}


	};

	struct InfoEventReport {
		enum TYPE {
			TYPE_VOID = 0,
			TYPE_TASK_SCOUTED,
			TYPE_TASK_REACHED,
			TYPE_TASK_MISSING,
			TYPE_TASK_PAUSED,
			TYPE_TASK_ABANDONED,
			TYPE_ROBOT_RECRUITED,
			TYPE_TASK_SWITCHED,
			TYPE_TASK_STARTED,
			TYPE_REWARD_STARTED,
			TYPE_REWARD_FINISHED
		} type;
		int taskId;
		Real taskValue;
		Real taskVolume;
		Real taskUtility;
		Real recruiterId;

		/*InfoEventReport(TYPE type_, int taskId_, Real taskValue_, Real taskVolume_, Real taskUtility_, Real recruiterId_) {
			type = type_; taskId = taskId_; taskValue = taskValue_; taskVolume = taskVolume_; taskUtility = taskUtility_; recruiterId = recruiterId_;
		}*/

		std::string GetTaskTypeDescription() {
			switch (type) {
			case TYPE_VOID: return "VOID";
			case TYPE_TASK_SCOUTED: return "TASK_SCOUTED";
			case TYPE_TASK_REACHED: return "TASK_REACHED";
			case TYPE_TASK_MISSING: return "TASK_MISSING";
			case TYPE_TASK_PAUSED: return "TASK_PAUSED";
			case TYPE_TASK_ABANDONED: return "TASK_ABANDONED";
			case TYPE_ROBOT_RECRUITED: return "ROBOT_RECRUITED";
			case TYPE_TASK_SWITCHED: return "TASK_SWITCHED";
			case TYPE_TASK_STARTED: return "TASK_STARTED";
			case TYPE_REWARD_STARTED: return "REWARD_STARTED";
			case TYPE_REWARD_FINISHED: return "REWARD_FINISHED";

			}
			return "VOID";
		}

	};

	enum STATE {
		STATE_RESTING = 0,
		STATE_SCOUTING,
		STATE_PROCESSING_TASK,
		STATE_GOING_TO_TASK,
		STATE_RETURN_TO_BASE,
		STATE_RETURN_TO_BASE_UNSUCCESSFUL,
		STATE_RETURN_TO_BASE_UNSUCCESSFUL_BLIND,
		STATE_UNLOADING_RESOURCE,
		STATE_BEE_RECRUITER_WAGGLE_DANCING,
		STATE_BEE_OBSERVING,
		STATE_BEE_READY_TO_DO_TASK,

	};

	enum LOCATION {
		LOCATION_RESTING_BAY = 0,
		LOCATION_DANCE_FLOOR,
		LOCATION_UNLOADING_BAY,
		LOCATION_DEPOSIT,
		LOCATION_OUT
	};



	enum SIGNAL_TYPE {
		SIGNAL_NONE = 0,
		SIGNAL_TASK_INFO_AVAILABLE,
		SIGNAL_TASK_INFO_TO_ID,
		SIGNAL_TASK_INFO_ANGLE_TO_TARGET,
		SIGNAL_TASK_INFO_DISTANCE_TO_TARGET,
		SIGNAL_TASK_INFO_UTILITY,
		SIGNAL_TASK_INFO_REL_UTILITY,
		SIGNAL_TASK_INFO_NUM_OF_ROBOTS_REQUIRED,
		SIGNAL_TASK_INFO_TASK_ID,
		SIGNAL_TASK_INFO_RECEIVING,
		SIGNAL_READY_TO_DO_TASK,

	};

public:

	static float GetLoadingProximity() { return 0.25; } //proximity to deposit at which loading can begin

	BaseRobot();
	virtual ~BaseRobot() {}

	virtual void Init(TConfigurationNode& t_node);
	virtual void PreControlStep();
	virtual void ControlStep();
	virtual void Reset();
	virtual void Destroy() {}


	virtual Real ProcessTask(TASK_TYPE taskType_, int taskId_,  Real processedAmount_, Real taskValue_, Real taskTotalReward_, uint numOfRobotsRequired_);

	float PerformUnloadResource();

	TaskInfo GetTaskInfo() { return targetTaskInfo; }
	CVector2 GetRelativeBaseLocation() { return relBaseLocation; }
	uint GetNumId() { return id; }
	void SetId(const uint& id_) { id = id_; }
	STATE GetState() { return state; }
	bool GetIsProcessingTask() { return state == STATE_PROCESSING_TASK; }
	bool GetIsUnloadingResource() { return state == STATE_UNLOADING_RESOURCE; }
	int GetLoadTimePerVolumeUnit() { return baseParameters.loadTimePerVolumeUnit; }
	void OnTaskRewardReceived(TASK_TYPE taskType_, int taskId_, double reward_, double taskValue_, double taskVolume_);

	//-- perceptions
	void SetNearbyTaskInfo(TASK_TYPE taskType_, int taskId_, Real taskValue_, Real taskVolume_, uint numOfRobotsRequired_);
	virtual Real EstimateTaskUtility(TASK_TYPE taskType_, Real taskValue_, Real taskVolume_, Real distanceFromBase_);
	virtual Real EstimateTaskUtility(TASK_TYPE taskType_, Real taskValue_, Real taskVolume_, Real distanceFromBase_, bool remeberEstimation_);
	virtual Real DoTaskUtilityCalculation(TASK_TYPE taskType_, Real taskValue_, Real taskVolume_, Real distanceFromBase_);

	//-- reporting
	std::vector<InfoEventReport>* GetInfoEventReports() { return &infoEventReports; }
	void ClearEventReports() { infoEventReports.clear(); } //called by the loop functions, after current loop's records have been read

protected:

	virtual void SetState(STATE state_, const bool& isNewState_ = true);

	virtual void Rest();

	virtual void Scout();
	virtual void DoNeighbourhoodSearch();
	virtual void OnTaskScouted();

	virtual void GoToTask();
	virtual void OnTaskReached();
	virtual void ProcessTaskReached();
	virtual void OnTaskReachedButDoesntExist();
	virtual void OnUnableToReachTask();
	virtual void AbandonTask(bool reportEvent_ = true);
	virtual void OnTaskProcessingDone(TASK_TYPE taskType_, int taskId_,  Real processedAmount_, Real taskValue_, Real taskRewardLeft_, uint numOfRobotsRequired_);


	virtual void ReturnToBase();

	virtual void OnUnloadResourceDone();

	void SetNewTarget(TASK_TYPE taskType_, int taskId_, Real taskValue_, CVector2 relLocation_, uint infoOriginatorId_, Real taskUtility_, Real relTaskUtility_, uint numOfRobotsRequired_);

	//-- navigation
	bool NavigateTowardsDeposit();
	void NavigateToLight();
	void NavigateToLight(const float baseSpeed_);
	void NavigateFromLight();
	void DoRandomWalk(const float baseSpeed_, const float maxDistanceFromTargetDeposit_ = 0);
	void DoLevyWalk();
	void UpdateOdometryToLocation(CVector2& location_);
	void ReadCurrentLocation();

	//-- other
	int* GenerateRandomPacketOrder(const CCI_RangeAndBearingSensor::TReadings& packets_);
	void NewInfoEventReport(InfoEventReport::TYPE type_, int taskId_, Real taskValue_, Real taskVolume_, Real taskUtility_, Real recruiterId_);

	//-- vectors
	CVector2 CalculateVectorToLight();
	CVector2 DiffusionVector(bool b_collision);
	bool IsSomethingNearby();
	bool NavigateTowardsVector(const CVector2& vector_);
	void SetWheelSpeedsFromVector(const CVector2& c_heading);
	void SetWheelSpeeds(Real left_, Real right_);


protected:
	//-- sensors
	CCI_RangeAndBearingSensor* rangeAndBearingSensor;
	CCI_FootBotProximitySensor* proximitySensor;
	CCI_FootBotLightSensor* lightSensor;
	CCI_FootBotMotorGroundSensor* groundSensor;
	CCI_DifferentialSteeringSensor* steeringSensor;

	//-- actuators
	CCI_DifferentialSteeringActuator* wheels;
	CCI_LEDsActuator* leds;
	CCI_RangeAndBearingActuator*  rangeAndBearingActuator;

	//-- ARGoS system params
	WheelTurningParams wheelParams;
	DiffusionParams diffustionParams;
	BaseParameters baseParameters;
	CRandom::CRNG* randNumGen;
	gsl_rng * gslRandNumGen;
	CRadians collisionAvoidingAngle;

	//-- basic
	uint id;
	STATE state;
	UInt64 timeCounter;
	uint timeScouting;
	uint timeGoingToTask;
	int timeTillNextScoutAllowed;
	int timeTillNextRelUChecked;

	//-- movement
	int randWalkTurnCounter;
	bool randWalkTurnLeft;

	//-- behaviour
	TaskInfo targetTaskInfo;
	TaskInfo nearbyTaskInfo; //set by the environment in pre-step
	CVector2 relBaseLocation;
	LOCATION location;
	//bool willPerformTaskOnNextStep;
	Real resourceInCartVolume;
	Real resourceInCartValue;
	int resourceInCartTaskId;
	bool supressNormalStateBehaviour;

	uint neighbourhoodSearchTimeLeft;



	//-- reporting
	bool didReportScoutEvent;
	bool didReportTaskReachedEvent;
	std::vector<InfoEventReport> infoEventReports;
	bool didReportReportTaskStartedEvent;
	bool didReportRewardStartReceivingEvent;
	bool didReportRewardStopReceivingEvent;

   //Memory memory;
   //Parameters parameters;
   //Behaviour behaviour;



};

#endif
