#include "solitary.h"

#include <argos3/core/utility/configuration/argos_configuration.h>
#include <argos3/core/utility/math/vector2.h>
#include <argos3/core/utility/logging/argos_log.h>
#include <algorithm>


//======================================= INITIALISATIONS

/**
 * Initialisation of set / explored parameters
 */
void Solitary::Parameters::Init(TConfigurationNode& t_node) {
   try {
	   GetNodeAttribute(t_node, "maxTeamSizeUnchangedTime", maxTeamSizeUnchangedTime);

	   //-- convert times in seconds to time in ticks
	   maxTeamSizeUnchangedTime *= 10;

   } catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initializing parameters.", ex);
   }

}


Solitary::Solitary() {

}

void Solitary::Init(TConfigurationNode& t_node) {
	BaseRobot::Init(t_node);
	try {
		parameters.Init(GetNode(t_node, "parameters"));
	}
	catch(CARGoSException& ex) {
		THROW_ARGOSEXCEPTION_NESTED("Error initializing the foot-bot foraging controller for robot \"" << GetId() << "\"", ex);
	}
}

void Solitary::Reset() {
	BaseRobot::Reset();
	SetState(STATE_SCOUTING);
}

//======================================= UPDATE LOOP

/**
 * Main update loop function
 */
void Solitary::ControlStep() {
	BaseRobot::ControlStep();
}



/**
 * Triggered once when a robot scouts and approaches a new task
 */
void Solitary::OnTaskScouted() {
	BaseRobot::OnTaskScouted();
	timeSinceLastRobotArrivedToTask = 0;
	lastNumOfRobotsAtTask = 0;
}

/**
 * Navigate towards a newly acquired / remembered target task
 */
void Solitary::GoToTask() {
	BaseRobot::GoToTask();
	if (location == LOCATION_DEPOSIT) {
		BroadcastTaskCommitment();
	}
	timeSinceLastRobotArrivedToTask = 0;
	lastNumOfRobotsAtTask = 0;
}

//============================================ TASK PROCESSING =======================
/**
 *  Triggered when robot reached current task location and task can be executed
 */
void Solitary::ProcessTaskReached() {
	BaseRobot::ProcessTaskReached();
	SetState(STATE_PROCESSING_TASK);
}


/**
 * Called when robot should be performing a task
 */
Real Solitary::ProcessTask(TASK_TYPE taskType_, int taskId_,  Real processedAmount_, Real taskValue_, Real taskTotalReward_, uint numOfRobotsRequired_) {
	FindOtherTaskBroadcasters();

	//-- if any new robots have been recruited, reset the timeSinceLastRecruitment time
	if (lastNumOfRobotsAtTask < encounteredIdsAtTask.size()) {
		timeSinceLastRobotArrivedToTask = 0;
		lastNumOfRobotsAtTask = encounteredIdsAtTask.size();
		////LOGERR << this->GetId() << " new recruit, reset timer " << std::endl;
	}


	////LOGERR << this->GetId() << "  others at task = " << encounteredIdsAtTask.size() << " time since last rec? " << timeSinceLastRobotArrivedToTask << " taskId" << taskId_ << std::endl;

	if (encounteredIdsAtTask.size() >= numOfRobotsRequired_-1 || taskId_ < 0 ) { //  || taskId_ < 0 : so that it can properly abandon task in the BaseRobot::ProcessTask method

		if (location != LOCATION_DEPOSIT) {
			//-- the robot didn't reach the middle of the gradient fast enough and others have already done the work
			if (processedAmount_ > 0) {
				NewInfoEventReport(InfoEventReport::TYPE_TASK_STARTED,taskId_,taskValue_,taskTotalReward_/taskValue_,EstimateTaskUtility(taskType_, taskValue_, taskTotalReward_/taskValue_, relBaseLocation.Length(), false), targetTaskInfo.infoOriginatorId);
			}
			AbandonTask();
			SetState(STATE_SCOUTING);
		} else {
			//-- remember the task and broadcast it to others if it's valid
			if (taskId_ >= 0) {
				SetNewTarget(taskType_, taskId_, taskValue_, CVector2(0,0), id, targetTaskInfo.taskUtility, targetTaskInfo.relTaskUtility, numOfRobotsRequired_);
				BroadcastTaskCommitment();
			}
			//-- at the same time, re-travel to the task's middle to be able to perform it
			if (NavigateTowardsDeposit()) {
				return BaseRobot::ProcessTask(taskType_, taskId_, processedAmount_, taskValue_, taskTotalReward_, numOfRobotsRequired_);
			}
		}
	} else {

		//-- check if no one has been recruited for a long time
		timeSinceLastRobotArrivedToTask ++;
		if (timeSinceLastRobotArrivedToTask > parameters.maxTeamSizeUnchangedTime) {
			AbandonTask();
			SetState(STATE_SCOUTING);
			timeTillNextScoutAllowed = 1200;
			return 0;
		}


		DoRandomWalk(wheelParams.maxSpeed/8,0.1);

		//-- remember the task if it's new
		if (targetTaskInfo.taskId != taskId_ || targetTaskInfo.isNull) {
			SetNewTarget(taskType_, taskId_, taskValue_, CVector2(0,0), id, targetTaskInfo.taskUtility, 1, numOfRobotsRequired_);
		}
		BroadcastTaskCommitment();


	}
	return 0;
}

//===================================================
/**
 * Return to nest with resource in cart
 */
void Solitary::ReturnToBase() {
	BaseRobot::ReturnToBase();
	if (location == LOCATION_DEPOSIT) {
		BroadcastTaskCommitment();
	}
}

/**
 * Triggered once when all resource has been unloaded
 */
void Solitary::OnUnloadResourceDone() {

	BaseRobot::OnUnloadResourceDone();
	if (targetTaskInfo.isNull) {
		////LOGERR << this->GetId() << "!! DONE UNLOADING, no task in memory " << std::endl;
		SetState(STATE_SCOUTING);
	} else {
		////LOGERR << this->GetId() << "!! DONE UNLOADING, return to task " << std::endl;
		SetState(STATE_GOING_TO_TASK);
	}

}

//====================================== COMMUNICATION


/**
 * Broadcast to others the intention to do task
 */
void Solitary::BroadcastTaskCommitment() {
	//-- send signals
	rangeAndBearingActuator->ClearData();
	rangeAndBearingActuator->SetData(4, SIGNAL_READY_TO_DO_TASK);
	rangeAndBearingActuator->SetData(5, targetTaskInfo.taskId);
	rangeAndBearingActuator->SetData(1,id);
}


/*
 * Find and count how many others are having the intention of doing a task
 */
void Solitary::FindOtherTaskBroadcasters() {
	encounteredIdsAtTask.clear();
	//-- generate random order in which packets are read
	const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
	int* indeces = GenerateRandomPacketOrder(packets);

	for(int p=0; p < packets.size(); p++) {
		int packetId = indeces[p];
		//if ((packets[packetId].Data[0] >= SIGNAL_TASK_INFO_TO_ID && packets[packetId].Data[0] <= SIGNAL_TASK_INFO_TASK_ID) ||
		if (packets[packetId].Data[4] == SIGNAL_READY_TO_DO_TASK && packets[packetId].Data[5] == targetTaskInfo.taskId ) {
			////LOGERR << this->GetId() << " found another at task " << packets[i].Data[1] << std::endl;
			bool alreadyNoted = false;
			for(UInt32 j = 0; j < encounteredIdsAtTask.size(); j++) {
				if (encounteredIdsAtTask[j] == packets[packetId].Data[1]) {
					alreadyNoted = true;
					break;
				}
			}
			if (!alreadyNoted) {
				////LOGERR << this->GetId() << " encountered " << packets[i].Data[1] << std::endl;
				encounteredIdsAtTask.push_back(packets[packetId].Data[1]);
			}
		}
	}
	delete[] indeces;
}







/****************************************/
/****************************************/

/*
 * This statement notifies ARGoS of the existence of the controller.
 * It binds the class passed as first argument to the string passed as
 * second argument.
 * The string is then usable in the XML configuration file to refer to
 * this controller.
 * When ARGoS reads that string in the XML file, it knows which controller
 * class to instantiate.
 * See also the XML configuration files for an example of how this is used.
 */
REGISTER_CONTROLLER(Solitary, "solitaryController")


